package gov.va.med.vos.patientProfile.data;

import gov.va.med.vos.common.AuthSettings;
import gov.va.med.vos.common.CalDAVManager;
import gov.va.med.vos.common.Command;
import gov.va.med.vos.patientProfile.domain.Profile;
import gov.va.med.vos.patientProfile.util.DOMUtil;
import gov.va.med.vos.patientProfile.util.PositionalXMLReader;

import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Properties;

import javax.annotation.PostConstruct;
import javax.xml.namespace.NamespaceContext;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathFactory;

import org.springframework.stereotype.Repository;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

@Repository
public class CalDAVDAOImpl implements CalDAVDAO {

	static final String PROPS_FILE_NAME = "profile.properties";

	XPathFactory xPathFactory = XPathFactory.newInstance();

	AuthSettings authSettings = null;
	String managerURI = null;

	NamespaceContext context = new NamespaceContextMap("d", "DAV:", //
			"V", "http://gov.va.med.vos", //
			"s", "http://sabredav.org/ns", //
			"cal", "urn:ietf:params:xml:ns:caldav",//
			"cs", "http://calendarserver.org/ns/");

	@PostConstruct
	private void getPropertiesFromClasspath() throws IOException {
		Properties props = new Properties();
		InputStream inputStream = this.getClass().getClassLoader()
				.getResourceAsStream(PROPS_FILE_NAME);
		if (inputStream == null)
			throw new FileNotFoundException(
					"property file 'profile.properties' not found in the classpath");
		props.load(inputStream);
		managerURI = props.getProperty("managerURI");
		authSettings = new AuthSettings(props.getProperty("authUsername"),
				props.getProperty("authPassword"));
	}

	public Profile getPatientProfile(String uri) {
		try {
			CalDAVManager manager = new CalDAVManager(managerURI, authSettings,
					null);

			String body = "<?xml version=\"1.0\" encoding=\"utf-8\" ?>"
					+ "<D:propfind xmlns:D=\"DAV:\" xmlns:V=\"http://gov.va.med.vos\">" //
					+ "<D:prop>" //
					+ "<V:SMS />" //
					+ "</D:prop>" //
					+ "</D:propfind>";

			/* A command encapsulates a single request to the server */
			Command command = new Command("PROPFIND", "/server.php" + uri,
					body, new HashMap<String, String>(), false);

			Map<String, Object> resultMap = manager.executeCommand(command);

			int statusCode = (Integer) resultMap.get("statusCode");
			Exception e = (Exception) resultMap.get("responseException");
			String resp = (String) resultMap.get("response");

			if (e != null) {
				throw e;
			} else if (statusCode != 207) {
				throw new Exception("Unexpected status code " + statusCode
						+ ": " + resp);
			} else {
				Document document = PositionalXMLReader
						.readXML(new ByteArrayInputStream(resp.getBytes()));

				XPath xPath = xPathFactory.newXPath();
				xPath.setNamespaceContext(context);

				XPathExpression xPathExpression = xPath
						.compile("/d:multistatus/d:response[1]/d:propstat/d:prop/V:SMS/text()");
				Object result = xPathExpression.evaluate(document);

				Profile p = new Profile();
				p.setUri(uri);
				p.setSms(result.toString());
				return p;
			}
		} catch (Exception e1) {
			throw new RuntimeException(e1);
		}
	}

	public Profile updatePatientProfile(Profile profile) {
		try {
			CalDAVManager manager = new CalDAVManager(managerURI, authSettings,
					null);

			String body = "<?xml version=\"1.0\" encoding=\"utf-8\" ?>"
					+ "<D:propertyupdate xmlns:D=\"DAV:\" xmlns:V=\"http://gov.va.med.vos\">" //
					+ "<D:set>" //
					+ "<D:prop>" //
					+ "<V:SMS>" + profile.getSms() + "</V:SMS>" //
					+ "</D:prop>" //
					+ "</D:set>" //
					+ "</D:propertyupdate>";

			/* A command encapsulates a single request to the server */
			Command command = new Command("PROPPATCH", "/server.php"
					+ profile.getUri(), body, new HashMap<String, String>(),
					false);

			Map<String, Object> resultMap = manager.executeCommand(command);

			int statusCode = (Integer) resultMap.get("statusCode");
			Exception e = (Exception) resultMap.get("responseException");
			String resp = (String) resultMap.get("response");

			if (e != null)
				throw e;
			if (statusCode != 207)
				throw new Exception("Unexpected status code " + statusCode
						+ ": " + resp);

			Profile updatedProfile = getPatientProfile(profile.getUri());
			return updatedProfile;
		} catch (Exception e1) {
			throw new RuntimeException(e1);
		}
	}

	public Map<String, String> getAllPatients(String relativeContext) {
		try {
			CalDAVManager manager = new CalDAVManager(managerURI, authSettings,
					null);

			String body = "<?xml version=\"1.0\" encoding=\"utf-8\" ?>"
					+ "<V:node-property-search xmlns:D=\"DAV:\" xmlns:V=\"http://gov.va.med.vos\">" //
					+ "<D:prop>" //
					+ "<D:displayname />" //
					+ "</D:prop>" //
					+ "</V:node-property-search>";

			Map<String, String> headers = new HashMap<String, String>();
			headers.put("Depth", "1");

			/* A command encapsulates a single request to the server */
			Command command = new Command("REPORT",
					"/server.php/vascheduling/principals/users/patients", body,
					headers, false);

			Map<String, Object> resultMap = manager.executeCommand(command);

			int statusCode = (Integer) resultMap.get("statusCode");
			Exception e = (Exception) resultMap.get("responseException");
			String resp = (String) resultMap.get("response");

			if (e != null)
				throw e;
			if (statusCode != 207)
				throw new Exception("Unexpected status code " + statusCode
						+ ": " + resp);

			Document document = PositionalXMLReader
					.readXML(new ByteArrayInputStream(resp.getBytes()));

			XPath xPath = xPathFactory.newXPath();
			xPath.setNamespaceContext(context);
			XPathExpression statusSelector = xPath
					.compile("d:propstat/d:status/text()");
			XPathExpression hrefSelector = xPath.compile("d:href/text()");
			XPathExpression nameSelector = xPath
					.compile("d:propstat/d:prop/d:displayname/text()");

			NodeList list = (NodeList) xPath.evaluate(
					"/d:multistatus/d:response", document,
					XPathConstants.NODESET);

			Map<String, String> results = new LinkedHashMap<String, String>();

			for (Node n : DOMUtil.iterable(list)) {
				String status = statusSelector.evaluate(n);
				if ("HTTP/1.1 200 OK".equals(status)) {
					String name = nameSelector.evaluate(n);
					String href = hrefSelector.evaluate(n);
					if (href.startsWith(relativeContext))
						href = href.substring(relativeContext.length());
					results.put(href, name);
				}
			}

			return results;
		} catch (Exception e1) {
			throw new RuntimeException(e1);
		}
	}
}
