package gov.va.med.vos.common;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpVersion;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.auth.params.AuthPNames;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpResponseException;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpHead;
import org.apache.http.client.methods.HttpOptions;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpTrace;
import org.apache.http.client.params.AuthPolicy;
import org.apache.http.conn.params.ConnRoutePNames;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.message.BasicHttpEntityEnclosingRequest;
import org.apache.http.message.BasicHttpRequest;
import org.apache.http.message.BasicRequestLine;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * A utility class to orchestrate communication with the CalDAV server.
 * 
 * @author aagarwal
 * @author vhaisdbarryc
 */
public class CalDAVManager {

	private static final String SAMPLE_BROWSER_USER_AGENT = "Mozilla/5.0 (Windows NT 6.1; WOW64; rv:19.0) Gecko/20100101 Firefox/19.0";
	private static final int DEFAULT_HTTP_PORT = 80;

	private DefaultHttpClient httpclient;
	private URI baseURI;

	/**
	 * Constructs a new HTTPExchanger
	 * 
	 * @param baseURIStr
	 *            A string containing the scheme, hostname, port (if different
	 *            than 80) and base path for all subsequent requests, e.g.
	 *            "http://172.31.106.7/server.php/vascheduling"
	 * @param authSettings
	 *            If the CalDAV server requires authentication, create an
	 *            AuthSettings here, otherwise pass in null
	 * @param proxySettings
	 *            If you need to go through a proxy server to connect to the
	 *            CalDAV server, construct a ProxySettings object and pass it in
	 *            here. Otherwise, pass in null.
	 * @throws URISyntaxException
	 *             If the schemeHostnamePort doesn't conform to the requirements
	 *             defined in the java.net.URI(String) constructor
	 */
	public CalDAVManager(String baseURIStr, AuthSettings authSettings,
			ProxySettings proxySettings) throws URISyntaxException {
		baseURI = new URI(baseURIStr);

		httpclient = new DefaultHttpClient();
		httpclient.setRedirectStrategy(new CustomRedirectStrategy());

		if (authSettings != null) {
			httpclient.getParams().setParameter(AuthPNames.TARGET_AUTH_PREF,
					Arrays.asList(AuthPolicy.DIGEST));
			httpclient.getCredentialsProvider().setCredentials(
					new AuthScope(baseURI.getHost(), getOrDefaultPort()),
					new UsernamePasswordCredentials(authSettings.getUsername(),
							authSettings.getPassword()));
		}

		if (proxySettings != null) {
			HttpHost proxy = new HttpHost(proxySettings.getServer(),
					proxySettings.getPort(), proxySettings.getType());
			httpclient.getParams().setParameter(ConnRoutePNames.DEFAULT_PROXY,
					proxy);

			if (proxySettings.getUsername() != null) {
				httpclient.getCredentialsProvider().setCredentials(
						new AuthScope(proxySettings.getServer(),
								proxySettings.getPort()),
						new UsernamePasswordCredentials(proxySettings
								.getUsername(), proxySettings.getPassword()));
			}
		}
	}

	private int getOrDefaultPort() {
		int port = baseURI.getPort();
		if (port < 0)
			port = DEFAULT_HTTP_PORT;
		return port;
	}

	/**
	 * 
	 * @throws IOException
	 * @throws ClientProtocolException
	 * @throws URISyntaxException
	 * @throws Exception
	 */
	public Map<String, Object> executeCommand(Command command)
			throws ClientProtocolException, IOException, URISyntaxException {
		String path = StringUtils.removeStart(command.getPath(), "/");
		URI uri = new URI(baseURI + "/" + path);

		HttpHost target = new HttpHost(uri.getHost(), uri.getPort(),
				uri.getScheme());

		String method = command.getMethod().toUpperCase().trim();
		boolean hasBody = StringUtils.isNotBlank(command.getBody());

		HttpRequest request = null;
		if (HttpGet.METHOD_NAME.equals(method)) {
			request = new HttpGet(uri);
		} else if (HttpPost.METHOD_NAME.equals(method)) {
			request = new HttpPost(uri);
		} else if (HttpPut.METHOD_NAME.equals(method)) {
			request = new HttpPut(uri);
		} else if (HttpTrace.METHOD_NAME.equals(method)) {
			request = new HttpTrace(uri);
		} else if (HttpDelete.METHOD_NAME.equals(method)) {
			request = new HttpDelete(uri);
		} else if (HttpHead.METHOD_NAME.equals(method)) {
			request = new HttpHead(uri);
		} else if (HttpOptions.METHOD_NAME.equals(method)) {
			request = new HttpOptions(uri);
		} else {
			String uriStr = uri.toString();
			if (!hasBody) {
				request = new BasicHttpRequest(new BasicRequestLine(method,
						uriStr, HttpVersion.HTTP_1_1));
			} else {
				request = new BasicHttpEntityEnclosingRequest(method, uriStr);
			}
		}

		if (hasBody && request instanceof HttpEntityEnclosingRequest) {
			((HttpEntityEnclosingRequest) request).setEntity(new StringEntity(
					command.getBody()));
		}

		for (Map.Entry<String, String> entry : command.getHeaders().entrySet())
			request.addHeader(entry.getKey(), entry.getValue());
		if (command.isEmulateBrowserUserAgent()
				&& !request.containsHeader("User-Agent"))
			request.addHeader("User-Agent", SAMPLE_BROWSER_USER_AGENT);
		if (!request.containsHeader("Cache-Control"))
			request.addHeader("Cache-Control", "no-cache");

		DetailCapturingResponseHandler responseHandler = new DetailCapturingResponseHandler();
		int statusCode = -1;
		String result = null;
		HttpResponseException exception = null;

		try {
			result = httpclient.execute(target, request, responseHandler);
			statusCode = responseHandler.responseStatus;
		} catch (HttpResponseException e) {
			statusCode = e.getStatusCode();
			result = responseHandler.body;
			exception = e;
		}

		if (result != null) {
			try {
				/*
				 * Attempt to parse and reserialize XML as pretty-print; if
				 * non-XML response, just print below
				 */
				Document document = PositionalXMLReader
						.readXML(new ByteArrayInputStream(result.getBytes()));
				Element root = document.getDocumentElement();
				result = XMLUtil.printXML(root);
			} catch (Exception e) {
			}
		}

		Map<String, Object> resultMap = new HashMap<String, Object>();
		resultMap.put("statusCode", statusCode);
		resultMap.put("response", result);
		resultMap.put("responseException", exception);

		return resultMap;
	}

	/**
	 * 
	 * @param args
	 * @throws Exception
	 */
	public static void main(String[] args) throws Exception {
		CalDAVManager manager = new CalDAVManager("http://172.31.106.7",
				new AuthSettings("admin", "admin"), null);

		String xmlDeclaration = "<?xml version=\"1.0\" encoding=\"utf-8\" ?>";

		String getPatientXML = xmlDeclaration
				+ "<D:principal-property-search xmlns:D=\"DAV:\">"
				+ "<D:property-search>"
				+ "<D:prop>"
				+ "<D:displayname/>"
				+ "</D:prop>"
				+ "<D:match>@PARAM1@</D:match>"
				+ "</D:property-search>"
				+ "<D:prop>"
				+ "<C:calendar-home-set xmlns:C=\"urn:ietf:params:xml:ns:caldav\"/>"
				+ "<D:displayname/>" + "</D:prop>"
				+ "</D:principal-property-search>";

		Map<String, String> inputParam = new HashMap<String, String>();
		inputParam.put("PARAM1", "Super");
		String body = getFormattedText(getPatientXML, inputParam);

		Map<String, String> headers = new HashMap<String, String>();
		// just an example
		headers.put("Depth", "0");

		/* A command encapsulates a single request to the server */
		Command command = new Command(HttpGet.METHOD_NAME,
				"/calendarserver.php", body, headers, false);

		Map<String, Object> resultMap = manager.executeCommand(command);

		int statusCode = (Integer) resultMap.get("statusCode");
		System.out.println("Response Status " + statusCode);
		Exception e = (Exception) resultMap.get("responseException");
		if (e != null) {
			System.out.println("Exception: " + e.getMessage());
		} else {
			System.out.println(resultMap.get("response"));
		}
	}

	/**
	 * 
	 * @param inputParams
	 * @return
	 */
	private static String getFormattedText(String notificationText,
			Map<String, String> inputParams) {
		Set<String> keySet = inputParams.keySet();
		for (String key : keySet) {
			notificationText = notificationText.replaceAll("@" + key + "@",
					inputParams.get(key));
		}
		return notificationText;
	}

}
