<?php

namespace gov\va\med\vos\DAV;

use Sabre\DAV;
use Sabre\DAV\Property;
use Sabre\DAV\IProperties;

/**
 * Customized Server for VOS
 *
 * @author Connor Barry
 */
class Server extends DAV\Server {

	public function __construct($treeOrNode = null) {
		parent::__construct($treeOrNode);
		putenv("WEBDAV_PROPFIND_MAX_DEPTH=infinity");
	}
	
	
	public function getUnfilteredPropertiesForPath($path) {
		$path = rtrim($path,'/');
		$nodeProperties = array();
		$node = $this->tree->getNodeForPath($path);
		
		if ($node instanceof IProperties) {
			$nodeProperties = $node->getProperties(array());
		}
		
		if ($node->getLastModified()) {
			$nodeProperties['{DAV:}getlastmodified'] = new Property\GetLastModified($node->getLastModified());
		}

		if ($node instanceof IFile) {
			$size = $node->getSize();
			if (!is_null($size)) {
				$nodeProperties['{DAV:}getcontentlength'] = (int)$node->getSize();
			}
		}

		if ($node instanceof IQuota) {
			$quotaInfo = $node->getQuotaInfo();
			$nodeProperties['{DAV:}quota-used-bytes'] = $quotaInfo[0];
		}

		if ($node instanceof IQuota) {
			$quotaInfo = $node->getQuotaInfo();
			$nodeProperties['{DAV:}quota-available-bytes'] = $quotaInfo[1];
		}


		if ($node instanceof IFile && $etag = $node->getETag()) {
			$nodeProperties['{DAV:}getetag'] = $etag;
		}

		if ($node instanceof IFile && $ct = $node->getContentType()) {
			$nodeProperties['{DAV:}getcontenttype'] = $ct;
		}

		$reports = array();
		foreach($this->plugins as $plugin) {
			$reports = array_merge($reports, $plugin->getSupportedReportSet($path));
		}
		$nodeProperties['{DAV:}supported-report-set'] = new Property\SupportedReportSet($reports);

		$nodeProperties['{DAV:}resourcetype'] = new Property\ResourceType();
		foreach($this->resourceTypeMapping as $className => $resourceType) {
			if ($node instanceof $className) {
				$nodeProperties['{DAV:}resourcetype']->add($resourceType);
			}
		}

		return $nodeProperties;
	}
	
	/**
	 * Recursively walks the given path to the given depth, returning the list of found nodes.
	 * This function is mainly used by the function below, to support PROPFIND with a depth greater than 1 or 'infinity'.
	 */
	protected function walkPath($depth, $path, &$nodes) {
		 
		// get the node object for the current root
		$parentNode = $this->tree->getNodeForPath($path);
		 
		// add it to the list of nodes
		$nodes[$path] = $parentNode;
		 
		// if the current node is a directory
		if ($parentNode instanceof \Sabre\DAV\ICollection) {
			foreach($this->tree->getChildren($path) as $childNode) {
				$nodes[$path . '/' . $childNode->getName()] = $childNode;
				 
				// recur into the child directories for the remaining depth or for infinity, if needed
				if ($childNode instanceof \Sabre\DAV\FS\Directory and ($depth - 1 > 0 or $depth === Sabre_DAV_Server::DEPTH_INFINITY)) {
					$this->walkPath($depth === \Sabre\DAV\Server::DEPTH_INFINITY ? \Sabre\DAV\Server::DEPTH_INFINITY : ($depth - 1), $path . '/' . $childNode->getName(), $nodes);
				}
			}
		}
	}

}
