<?php

namespace gov\va\med\vos\DAV;

use Sabre\DAV;
use Sabre\DAV\Property\GetLastModified;
use Sabre\DAV\Property\ResourceType;
use Sabre\DAV\IProperties;
use Sabre\CardDAV\IDirectory;

/**
 * NodePropertySearch Plugin
 *
 * @author Connor Barry
 */
class NodePropertySearchPlugin extends DAV\ServerPlugin {

	/**
	 * reference to server class
	 *
	 * @var Sabre\DAV\Server
	 */
	protected $server;

	/**
	 * Creates the object.
	 */
	public function __construct($someParam=true) {
	}

	/**
	 * Initializes the plugin
	 *
	 * @param DAV\Server $server
	 * @return void
	 */
	public function initialize(DAV\Server $server) {
		$this->server = $server;
		$server->subscribeEvent('report', array($this,'report'));
	}

	public function report($reportName, $dom) {
		switch($reportName) {
			case '{http://gov.va.med.vos}node-property-search' :
				$this->nodePropertySearchReport($dom);
				return false;
		}
	}

	/**
	 * nodePropertySearchReport
	 *
	 * This method is responsible for handing the
	 * {http://gov.va.med.vos}node-property-search report. This report can be used for
	 * clients to search for groups of nodes, based on the value of one or more properties.
	 *
	 * @param \DOMDocument $dom
	 * @return void
	 */
	protected function nodePropertySearchReport(\DOMDocument $dom) {
		list($searchProperties, $requestedProperties) = $this->parseNodePropertySearchReportRequest($dom);

		$uri = $this->server->getRequestUri();
		$result = $this->nodeSearch($searchProperties, $requestedProperties, $uri);

		$prefer = $this->server->getHTTPPRefer();

		$this->server->httpResponse->sendStatus(207);
		$this->server->httpResponse->setHeader('Content-Type','application/xml; charset=utf-8');
		$this->server->httpResponse->setHeader('Vary','Brief,Prefer');
		$this->server->httpResponse->sendBody($this->server->generateMultiStatus($result, $prefer['return-minimal']));

	}

	/**
	 * parseNodePropertySearchReportRequest
	 *
	 * This method parses the request body from a
	 * {http://gov.va.med.vos}node-property-search report.
	 *
	 * This method returns an array with three elements:
	 *  1. an array with properties to search on, and their values
	 *  2. a list of propertyvalues that should be returned for the request.
	 *
	 * @param \DOMDocument $dom
	 * @return array
	 */
	protected function parseNodePropertySearchReportRequest($dom) {

		$searchProperties = array();

		// Parsing the search request
		foreach($dom->firstChild->childNodes as $searchNode) {

			if (DAV\XMLUtil::toClarkNotation($searchNode) !== '{DAV:}property-search')
				continue;

			$propertyName = null;
			$propertyValue = null;

			foreach($searchNode->childNodes as $childNode) {

				switch(DAV\XMLUtil::toClarkNotation($childNode)) {

					case '{DAV:}prop' :
						$property = DAV\XMLUtil::parseProperties($searchNode);
						reset($property);
						$propertyName = key($property);
						break;

					case '{DAV:}match' :
						$propertyValue = $childNode->textContent;
						break;

				}

			}

			if (is_null($propertyName) || is_null($propertyValue))
				throw new DAV\Exception\BadRequest('Invalid search request. propertyname: '
						. $propertyName . '. propertyvalue: ' . $propertyValue);

			$searchProperties[$propertyName] = $propertyValue;

		}

		return array($searchProperties,
				array_keys(DAV\XMLUtil::parseProperties($dom->firstChild)));

	}

	/**
	 * Node property search
	 *
	 * This method can search for nodes matching certain values in
	 * properties.
	 *
	 * This method will return a list of properties for the matched properties.
	 *
	 * @param array $searchProperties    The properties to search on. This is a
	 *                                   key-value list. The keys are property
	 *                                   names, and the values the strings to
	 *                                   match them on.
	 * @param array $requestedProperties This is the list of properties to
	 *                                   return for every match.
	 * @param string $collectionUri      The node collection to search on.
	 * @return array     This method returns an array structure similar to
	 *                  Sabre\DAV\Server::getPropertiesForPath. Returned
	 *                  properties are index by a HTTP status code.
	 *
	 */
	public function nodeSearch(array $searchProperties, array $requestedProperties, $collectionUri) {

		$uris = array($collectionUri);
		$httpDepth = $this->server->getHTTPDepth();
		$lookupResults = array();

		foreach($uris as $uri) {

			$searchNode = $this->server->tree->getNodeForPath($uri);
			$results = $this->search($uri, $searchNode, $searchProperties, $httpDepth);

			foreach($results as $result) {
				$lookupResults = array_merge($lookupResults, $results);
			}

		}

		$matches = array();

		foreach($results as $lookupResult) {

			list($matches[]) = $this->server->getPropertiesForPath($lookupResult, $requestedProperties, 0);

		}

		return $matches;
	}

	public function search($relativePath, $node, $searchProperties, $depth) {
		$results = array();

		if (!$node instanceof IProperties)
			return $results;

		$foundProperties = $node->getProperties(array_keys($searchProperties));

		if (sizeof($searchProperties) == 0 || $searchProperties == $foundProperties)
			$results[] = $relativePath;

		if ($depth == 0) return $results;

		if ($node instanceof DAV\ICollection) {
			$children = $node->getChildren();

			foreach($children as $child) {
				$childMatches = $this->search($relativePath . '/' . $child->getName(), $child,
						$searchProperties, $depth == -1 ? -1 : $depth - 1);
				if (!empty($childMatches))
					$results = array_merge($results, $childMatches);
			}
		}

		return $results;
	}

}
