package gov.va.med.vos.service.util;

import gov.va.med.vos.service.CalendarParser;
import gov.va.med.vos.service.CalendarParserException;
import gov.va.med.vos.service.model.AppointmentRequest;
import gov.va.med.vos.service.model.Attendee;
import gov.va.med.vos.service.model.AttendeeType;
import gov.va.med.vos.service.model.Calendar;
import gov.va.med.vos.service.model.Event;
import gov.va.med.vos.service.model.Location;
import gov.va.med.vos.service.model.Participant;
import gov.va.med.vos.service.model.ResourceOption;

import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.SerializationUtils;
import org.jboss.soa.esb.message.Body;
import org.jboss.soa.esb.message.Message;

/**
 * Utility class for extracting values from inserting values into instances of
 * the Message class.
 */
public class MessageUtils {

	/*
	 * ESB Message Keys
	 */
	private static final String BODY_KEY_APPOINTMENT_REQUEST = "appointmentRequest";
	private static final String BODY_KEY_APPOINTMENTS = "appointments";
	private static final String BODY_KEY_RESOURCE_MAP = "resourceMap";
	private static final String BODY_KEY_LOCATION = "location";
	private static final String BODY_KEY_TIMEZONE = "timezone";
	private static final String BODY_KEY_CLINIC_IEN = "clinicIEN";
	private static final String BODY_KEY_PATIENT_IEN = "patientIEN";
	private static final String BODY_KEY_EVENT = "event";
	private static final String BODY_KEY_PATIENT = "patient";
	
	
	private CalendarParser calendarParser;

	public MessageUtils() {
		this.calendarParser = new CalendarParser();
	}

	/**
	 * Extract an AppointmentRequest from a Message.
	 * 
	 * @param message
	 *            A Message that contains an AppointmentRequest.
	 * @return The AppointmentRequest contained by the Message.
	 */
	public AppointmentRequest extractAppointmentRequest(Message message) {
		Body body = message.getBody();
		AppointmentRequest appointmentRequest = (AppointmentRequest) body
				.get(BODY_KEY_APPOINTMENT_REQUEST);
		return appointmentRequest;
	}

	/**
	 * Insert a AppointmentRequest into a Message.
	 * 
	 * @param appointmentRequest
	 *            The AppointmentRequest to insert into the Message.
	 * @param message
	 *            The Message into which the AppointmentRequest will be
	 *            inserted.
	 */
	public void insertAppointmentRequest(AppointmentRequest appointmentRequest,
			Message message) {
		message.getBody().add(BODY_KEY_APPOINTMENT_REQUEST, appointmentRequest);
	}
	
	/**
	 * Add one or more Attendee objects of the same type to the resource map stored on the
	 * ESB message
	 * 
	 * uses lazy loading to initialize the map
	 * 
	 * @param message
	 * 			the ESB message containing the resourceMap
	 * @param type
	 * 			the type of the Attendees to be added
	 * @param attendees
	 * 			one or more Attendee objects to be added to the map
	 */
	@SuppressWarnings("unchecked")
	public void putToResourceMap(Message message, AttendeeType type, ResourceOption ... resources){
		
		Map<AttendeeType, List<ResourceOption>> resourceMap;
		
		if ((resourceMap = (Map<AttendeeType, List<ResourceOption>>) message.getBody().get(BODY_KEY_RESOURCE_MAP)) == null){
			resourceMap = new HashMap<AttendeeType, List<ResourceOption>>();
			message.getBody().add(BODY_KEY_RESOURCE_MAP, resourceMap);
		}
			
		if (resourceMap.get(type) == null){
			resourceMap.put(type, new ArrayList<ResourceOption>());
		}
		
		resourceMap.get(type).addAll(Arrays.asList(resources));
	}
	
	/**
	 * Retrieves the resource map from the ESB message
	 * @param message
	 * 			an ESB Message containing a resource map
	 * @return
	 * 			a Map of appointment attendees grouped by AttendeeType
	 */
	@SuppressWarnings("unchecked")
	public Map<AttendeeType, List<ResourceOption>> extractResourceMap(Message message){
		return (Map<AttendeeType, List<ResourceOption>>) message.getBody().get(BODY_KEY_RESOURCE_MAP);
	}

	/**
	 * Insert a list of appointments into a Message
	 * 
	 * @param cal
	 *            A Calendar containing a list of appointment Events
	 * @param message
	 *            the Message into which the list of appointments will be
	 *            inserted
	 */
	public void insertAppointments(Calendar cal, Message message) {
		message.getBody().add(BODY_KEY_APPOINTMENTS, cal.getEvents());

		message.getBody().add(cal.getSerializedCalendar());
	}
	
	public void insertClinicIEN(String clinicIEN, Message message){
		message.getBody().add(BODY_KEY_CLINIC_IEN, clinicIEN);
	}
	
	public void insertPatientIEN(String patientIEN, Message message){
		message.getBody().add(BODY_KEY_PATIENT_IEN, patientIEN);
	}
	
	public void insertLocation(Location location, Message message){
		message.getBody().add(BODY_KEY_LOCATION, location);
	}
	
	public void insertTimezone(String timezone, Message message){
		message.getBody().add(BODY_KEY_TIMEZONE, timezone);
	}
	
	public void insertEvent(Event event, Message message){
		message.getBody().add(BODY_KEY_EVENT, event);
	}
	
	public void insertPatient(Participant p, Message message){
		message.getBody().add(BODY_KEY_PATIENT, p);
	}
	
	public Attendee extractPatient(Message message){
		return (Attendee) message.getBody().get(BODY_KEY_PATIENT);
	}
	
	public Event extractEvent(Message message){
		return (Event) message.getBody().get(BODY_KEY_EVENT);
	}
	
	public String extractClinicIEN(Message message){
		return (String) message.getBody().get(BODY_KEY_CLINIC_IEN);
	}
	
	public String extractPatientIEN(Message message){
		return (String)message.getBody().get(BODY_KEY_PATIENT_IEN);
	}
	
	public Location extractLocation(Message message){
		return (Location) message.getBody().get(BODY_KEY_LOCATION);
	}
	
	public String extractTimezone(Message message){
		return (String) message.getBody().get(BODY_KEY_TIMEZONE);
	}
	
	/**
	 * Maps an AppointmentRequest onto the defaultEntry key of the message body
	 * 
	 * Used before sending an AppointmentRequest through an HTTP router
	 * 
	 * @param request
	 *            the AppointmentRequest to be serialized
	 * @param message
	 *            an ESB Message that contains an AppointmentRequest
	 */
	public void mapRequestOntoDefault(AppointmentRequest request, HashMap<AttendeeType, List<Attendee>> resourceMap, Message message){
		//declare as HashMap since not all implementers of Map are serializable
		HashMap<String,Object> requestMap = new HashMap<String,Object>();			
		requestMap.put("appointmentRequest", request);
		requestMap.put("resourceMap", resourceMap);
		
		message.getBody().add(SerializationUtils.serialize(requestMap));
	}

	/**
	 * Extracts a AppointmentRequest from an ESB message that has come through
	 * an HTTP gateway.
	 * 
	 * The AppointmentRequest should be serialized and stored as a byte array on
	 * the default body key of the ESB message. This is the opposite of the
	 * {@link MessageUtils#mapRequestOntoDefault} method.
	 * 
	 * @param message
	 * @return
	 */
	public Map<String,Object> extractRequestMapFromHttpPayload(Message message){
		byte[] body = (byte[]) message.getBody().get();
		return (Map<String,Object>) SerializationUtils.deserialize(body);
	}
	
	public Object getDefaultMessageBody(Message message){
		return message.getBody().get();
	}
	
	public void insertResourceMap(Map<AttendeeType, List<Attendee>> resourceMap, Message message){
		message.getBody().add(BODY_KEY_RESOURCE_MAP, resourceMap);
	}

	/**
	 * Parses the Message body as a Calendar.
	 * 
	 * @param message
	 * @return
	 * @throws CalendarParserException
	 */
	public Calendar extractCalendar(Message message)
			throws CalendarParserException {
		// TODO: break this method up
		String vCalendar = (String) message.getBody().get();
		Calendar calendar = calendarParser.parse(vCalendar);
		
		return calendar;
	}
	
	/**
	 * Parses the Message body as a Calendar, looks at the first Event on the
	 * calendar, gets the list of Attendees of the first Event, and returns a
	 * list URIs for the Calendar of each Attendee.
	 * 
	 * @param message
	 * @return
	 * @throws CalendarParserException
	 */
	public List<URI> extractAttendeeCalendarUris(Message message)
			throws CalendarParserException {
		// TODO: break this method up
		String vCalendar = (String) message.getBody().get();
		Calendar calendar = calendarParser.parse(vCalendar);
		List<Event> events = calendar.getEvents();
		Event event = events.get(0);
		List<Attendee> attendees = event.getAttendees();
		
		// Build a List of the Attendees' Calendar URIs
		ArrayList<URI> calendarUris = new ArrayList<URI>();
		for(Attendee attendee: attendees) {
			URI calendarUri = attendee.getPrincipalUri();
			calendarUris.add(calendarUri);
		}

		return calendarUris;
	}

}
