package gov.va.med.vos.service.model;

import java.io.IOException;
import java.io.Serializable;
import java.net.URISyntaxException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import net.fortuna.ical4j.model.DateTime;
import net.fortuna.ical4j.model.Dur;
import net.fortuna.ical4j.model.ParameterList;
import net.fortuna.ical4j.model.Property;
import net.fortuna.ical4j.model.PropertyList;
import net.fortuna.ical4j.model.TextList;
import net.fortuna.ical4j.model.component.VEvent;
import net.fortuna.ical4j.model.parameter.Cn;
import net.fortuna.ical4j.model.parameter.CuType;
import net.fortuna.ical4j.model.parameter.PartStat;
import net.fortuna.ical4j.model.parameter.Role;
import net.fortuna.ical4j.model.property.Categories;
import net.fortuna.ical4j.model.property.Organizer;
import net.fortuna.ical4j.model.property.Sequence;
import net.fortuna.ical4j.model.property.Transp;
import net.fortuna.ical4j.model.property.Uid;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 
 * Corresponds to an ical4j VEvent
 */
public class Event implements Serializable {

	private static final long serialVersionUID = -1687125739696727369L;
	
	private static final String ORGANIZER = "http://172.31.106.7/server.php/principals/admin";
	
	private static Logger logger = LoggerFactory.getLogger(Event.class);

	private VEvent vevent;

	public Event(Date start, AppointmentRequest request, Set<ResourceOption> resourceOptions) throws URISyntaxException, IOException, ParseException {
		
		//calculate end time
		Calendar end = Calendar.getInstance();
		end.setTime(start);
		end.add(Calendar.MINUTE, request.getLength());
		
		this.vevent = new VEvent(new DateTime(start), new Dur(start,
				end.getTime()), request.getSummary());
		
		//set DTEND
		vevent.getEndDate().setDate(new net.fortuna.ical4j.model.Date(end.getTime()));
		
		
		this.setResourceOptions(resourceOptions);
		this.setOrganizer();
		this.setLocation(request);
		this.setCategories(request);
		
		vevent.getProperties().add(new Transp("OPAQUE"));
		vevent.getProperties().add(new Uid(UUID.randomUUID().toString()));
		vevent.getProperties().add(new Sequence("0"));
		
		vevent.getStartDate().setUtc(true);
	}

	public Event(VEvent vevent) {
		this.vevent = vevent;
	}
	
	VEvent getVEvent(){
		return this.vevent;
	}

	public List<Attendee> getAttendees() {

		// Get the ATTENDEE properties from the VEVENT
		PropertyList properties = vevent.getProperties(Property.ATTENDEE);

		// Wrap the ATTENDEE properties in Attendee objects.
		List<Attendee> attendees = new ArrayList<Attendee>(properties.size());
		for (int i = 0; i < properties.size(); ++i) {
			Object attendeeObject = properties.get(i);
			net.fortuna.ical4j.model.property.Attendee iCal4jAttendee = (net.fortuna.ical4j.model.property.Attendee) attendeeObject;
			Attendee attendee = new Attendee(iCal4jAttendee);
			attendees.add(attendee);
		}

		return attendees;
	}
	
	public List<Location> getLocations() throws IllegalStateException {
		

		// Get the Location properties from the VEVENT
		PropertyList properties = vevent.getProperties(Property.LOCATION);

		// Wrap the Location properties in Location objects.
		List<Location> locations = new ArrayList<Location>(properties.size());
		for (int i = 0; i < properties.size(); ++i) {
			Object locationObject = properties.get(i);
			net.fortuna.ical4j.model.property.Location iCal4jLocation = (net.fortuna.ical4j.model.property.Location) locationObject;
			Location Location;
			try {
				Location = new Location(iCal4jLocation);
			} catch (URISyntaxException e) {
				throw new IllegalStateException();
			}
			locations.add(Location);
		}
		
		return locations;

	}
	
	/**
	 * Returns a list of all Participants scheduled for the event, which includes Attendees and Locations.
	 * @return A List of Participants.
	 */
	public List<Participant> getParticipants() {
		
		List<Participant> participants = new ArrayList<Participant>();
		
		// Add the Attendees.
		List<Attendee> attendees = getAttendees();
		participants.addAll(attendees);
		
		// Add the Locations
		List<Location> locations = getLocations();
		participants.addAll(locations);
		
		return participants;
	}
	
	/**
	 * Retrieve the length of this Event in minutes
	 * 
	 * @return
	 * 		the length of this Event in minutes
	 */
	public int getAppointmentLength(){
		Dur dur;
		if (vevent.getDuration() != null){
			dur = vevent.getDuration().getDuration();
		} else {
			dur = new Dur(vevent.getStartDate().getDate(), vevent.getEndDate().getDate());
		}
		
		return dur.getMinutes() + (dur.getHours() * 60) + (dur.getDays() * 1440) + (dur.getWeeks() * 10080);
	}
	
	
	/**
	 * Retrieve the datestamp of this Event
	 * 
	 * @return
	 * 			Date representing the DTSTART property of the VEVENT this Event wraps
	 */
	public Date getAppointmentTimestamp(){
		return vevent.getStartDate().getDate();
	}

	public String getUid() {
		Property uidProperty = vevent.getProperty(Property.UID);
		String uid = uidProperty.getValue();
		return uid;
	}
	
	public List<AppointmentCategory> getCategories(){
		Categories categories = (Categories) vevent.getProperty(Property.CATEGORIES);
		TextList textList = categories.getCategories();
		
		List<AppointmentCategory> appointmentCategories = new ArrayList<AppointmentCategory>();
		
		@SuppressWarnings("unchecked")
		Iterator<String> iterator = textList.iterator();
		while (iterator.hasNext()){ //does not extend List, implement Iterable, or implement a get() method
			appointmentCategories.add(AppointmentCategory.getAppointmentCategory(iterator.next()));
		}
		
		return appointmentCategories;
	}
	
	public Event clone() {
		try {
			return new Event((VEvent)vevent.copy());
		} catch (ParseException e) {
			return null;
		} catch (IOException e) {
			return null;
		} catch (URISyntaxException e) {
			return null;
		}
	}
	
	/**
	 * Retrieves the clinic that is scheduled in the legacy VistA system
	 * 
	 * This is the hub clinic for telehealth appointments. For office visits, 
	 * it will be the location of the visit.
	 * 
	 * @return 
	 * 			a clinic Attendee with only the principal URI set
	 * 		
	 */
	public Attendee getVistaClinic(){
		if (this.getCategories().contains(AppointmentCategory.TELEHEALTH)){
			for (Attendee a : this.getAttendees()){
				if (a.getType() == AttendeeType.HUB_CLINIC){
					return a;
				}
			}
			//don't throw a hard error to maintain backwards compatibility with the old appointments in CalDAV
			logger.info("No hub clinic for telehealth appointment: " + this.toString());
			return null;
		} else {
			Attendee clinic = new Attendee();
			clinic.setPrincipalUri(this.getLocations().get(0).getPrincipalUri());
			return clinic;
		}
	}

	private void setResourceOptions(Set<ResourceOption> attendees) throws URISyntaxException {
		ParameterList paramList;
		PropertyList properties = this.vevent.getProperties();
		for (ResourceOption a : attendees){
			if (a.getType() == AttendeeType.LOCATION){
				continue;
			}
			paramList = new ParameterList();
			paramList.add(new Cn(a.getDisplayname()));
			paramList.add(new CuType(a.getType().toString()));
			paramList.add(new Role("REQ=PARTICIPANT"));
			paramList.add(new PartStat("ACCEPTED"));
			properties.add(new net.fortuna.ical4j.model.property.Attendee(paramList, a.getPrincipalUri()));
		}
	}
	
	private void setOrganizer() throws URISyntaxException{
		ParameterList paramList = new ParameterList();
		paramList = new ParameterList();
		paramList.add(new Cn("Admin"));
		vevent.getProperties().add(new Organizer(paramList, ORGANIZER));
	}
	
	private void setLocation(AppointmentRequest request){
		String locationURI = "";
		ParameterList paramList = new ParameterList();
		paramList.add(new Cn(request.getClinic().getDisplayname()));
		paramList.add(new CuType("LOCATION"));
		if (!(request.getClinic().isHome() || request.getClinic().getDisplayname().equals("OTHER"))){
			paramList.add(new Role("REQ-PARTICIPANT"));
			paramList.add(new PartStat("ACCEPTED"));
			locationURI = request.getClinic().getPrincipalUri().toString();
		}
		vevent.getProperties().add(new net.fortuna.ical4j.model.property.Location(paramList, locationURI));
	}
	
	public void setLocation(Participant p){
		String locationURI = "";
		ParameterList paramList = new ParameterList();
		paramList.add(new Cn(p.getName()));
		if (p.getPrincipalUri() != null){
			paramList.add(new CuType(p.getType().toString()));
			paramList.add(new Role("REQ-PARTICIPANT"));
			paramList.add(new PartStat("ACCEPTED"));
			locationURI = p.getPrincipalUri().toString();
		}
		vevent.getProperties().remove(vevent.getLocation());
		vevent.getProperties().add(new net.fortuna.ical4j.model.property.Location(paramList, locationURI));
	}
	
	public void addAttendee(Participant p){
		if (p == null){
			return;
		}
		
		PropertyList properties = vevent.getProperties();
		
		ParameterList paramList = new ParameterList();
		paramList.add(new Cn(p.getName()));
		paramList.add(new CuType(p.getType().toString()));
		paramList.add(new Role("REQ=PARTICIPANT"));
		paramList.add(new PartStat("ACCEPTED"));
		properties.add(new net.fortuna.ical4j.model.property.Attendee(paramList, p.getPrincipalUri()));
	}
	
	public void clearAttendees(){
		PropertyList properties = vevent.getProperties(Property.ATTENDEE);
		
		vevent.getProperties().removeAll(properties);
	}
	
	private void setCategories(AppointmentRequest request){
		TextList text = new TextList();
		for (AppointmentCategory cat : request.getCategories()){
			text.add(cat.getValue());
		}
		
		vevent.getProperties().add(new Categories(text));
	}
	
	public String toString(){
		return this.vevent.toString();
	}
	

}
