package gov.va.med.vos.service.model;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import net.fortuna.ical4j.model.Component;
import net.fortuna.ical4j.model.ComponentList;
import net.fortuna.ical4j.model.DateRange;
import net.fortuna.ical4j.model.Period;
import net.fortuna.ical4j.model.ValidationException;
import net.fortuna.ical4j.model.component.VEvent;
import net.fortuna.ical4j.model.component.VFreeBusy;
import net.fortuna.ical4j.model.property.FreeBusy;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A wrapper around the ical4j iCalendar class.
 * 
 * A Calendar can contain any number of events that occur within the range of
 * dates defined by the startDate and endDate properties.
 */
public class Calendar implements Serializable {

	private static final long serialVersionUID = 6929972321447790215L;
	
	private static Logger logger = LoggerFactory.getLogger(Calendar.class);

	private Date startDate;
	private Date endDate;
	
	private net.fortuna.ical4j.model.Calendar iCalendar;

	public Calendar() {
		this(new net.fortuna.ical4j.model.Calendar());
	}

	public Calendar(net.fortuna.ical4j.model.Calendar iCalendar) {
		this.iCalendar = iCalendar;
	}

	/**
	 * Validate the calendar.
	 * 
	 * @throws ValidationException
	 */
	public void validate() throws ValidationException {
		this.iCalendar.validate();
	}

	/**
	 * Test whether the Calandar has an appointment available.
	 * 
	 * @param start
	 *            The start time of the appointment.
	 * @param appointmentLength
	 *            The duration of the appointment.
	 * @return True if the appointment is available.
	 */
	public boolean isAppointmentAvailable(Date start, int appointmentLength) {
		VFreeBusy vfb;
		java.util.Calendar end = java.util.Calendar.getInstance();
		end.setTime(start);
		end.add(java.util.Calendar.MINUTE, appointmentLength);

		for (Object o : iCalendar.getComponents()) {
			vfb = (VFreeBusy) o;
			
			FreeBusy fb;
			for (Object ob : vfb.getProperties("FREEBUSY")){
				fb = (FreeBusy) ob;
				for (Object obj : fb.getPeriods()){
					Period p = (Period) obj;
					
//					logger.info("START: " + start);
//					logger.info("END: " + end.getTime());
//					logger.info("FB START: " + p.getStart());
//					logger.info("FB END: " + p.getEnd());
//					logger.info(String.valueOf(p.intersects(new DateRange(start, end.getTime()))));
					
					if (p.intersects(new DateRange(start, end.getTime()))){
						return false;
					}
				}
				
			}

		}
		return true;
	}

	public int countAppointments() {
		return iCalendar.getComponents().size();
	}

	/**
	 * Retrieve the serialized iCalendar formatted as specified here:
	 * http://build
	 * .mnode.org/projects/ical4j/apidocs/net/fortuna/ical4j/model/Calendar.html
	 * 
	 * @return String the serialized form of the iCalendar
	 */
	public String getSerializedCalendar() {
		return this.iCalendar.toString();
	}

	/**
	 * Set some Events as components of this Calendar
	 * 
	 * @param events
	 *            the Events that should be set as components of this Calendar
	 */
	public void addEvent(List<Event> events) {
		for (Event e : events) {
			iCalendar.getComponents().add(e.getVEvent());	
		}
	}

	/**
	 * @param e
	 *            an Event to be added as a component to this Calendar
	 */
	public void addEvent(Event e) {
		iCalendar.getComponents().add(e.getVEvent());
	}

	public void addComponent(Component c) {
		this.iCalendar.getComponents().add(c);
	}

	public Date getStartDate() {
		return startDate;
	}

	public void setStartDate(Date startDate) {
		this.startDate = startDate;
	}

	public Date getEndDate() {
		return endDate;
	}

	public void setEndDate(Date endDate) {
		this.endDate = endDate;
	}

	public List<Event> getEvents() {

		// Find the VEVENT components in the VCALENDAR
		ComponentList components = this.iCalendar
				.getComponents(Component.VEVENT);
		
		// Wrap them in a list of Events
		List<Event> events = new ArrayList<Event>();
		for (int i = 0; i < components.size(); ++i) {
			Object componentObject = components.get(i);
			VEvent vevent = (VEvent) componentObject;
			Event event = new Event(vevent);
			events.add(event);
		}
		
		return events;
	}

}
