package gov.va.med.vos.service.actions;

import gov.va.med.vos.service.CalDavDAO;
import gov.va.med.vos.service.CalDavException;
import gov.va.med.vos.service.ConfigException;
import gov.va.med.vos.service.model.AppointmentCategory;
import gov.va.med.vos.service.model.AppointmentRequest;
import gov.va.med.vos.service.model.AttendeeType;
import gov.va.med.vos.service.model.Clinic;
import gov.va.med.vos.service.model.ResourceOption;
import gov.va.med.vos.service.util.MessageUtils;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;
import java.util.Map;

import org.jboss.soa.esb.actions.AbstractActionLifecycle;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * An ESB Action that selects Provider attendees based on the AppointmentType.
 * 
 * TODO this needs to be more than one action
 */
public class SelectResourceCandidatesAction extends AbstractActionLifecycle {
	
	private static Logger logger = LoggerFactory.getLogger(SelectResourceCandidatesAction.class);

	protected ConfigTree _config;
	MessageUtils messageUtils;
	private CalDavDAO calDavDAO;

	/**
	 * Constructor.
	 * 
	 * @param config
	 *            The runtime configuration determined by the ESB.
	 * @throws CalDavException 
	 * @throws ConfigException 
	 */
	public SelectResourceCandidatesAction(ConfigTree config) throws CalDavException, ConfigException {
		_config = config;
		this.messageUtils = new MessageUtils();
		this.calDavDAO = new CalDavDAO();
	}

	/**
	 * Select Provider attendees based on the AppointmentType.
	 * @throws CalDavException 
	 * @throws URISyntaxException 
	 */
	public Message process(Message message) throws ActionProcessingException {
		
		// Get the request
		AppointmentRequest request = messageUtils
				.extractAppointmentRequest(message);
		if(null == request){
			logger.debug("message has no appointment request");
			throw new IllegalArgumentException(
					"Message has no AppointmentRequest.");
		}
		
		// Get the type
		List<AppointmentCategory> categories = request.getCategories();
		if (categories.size() == 0) {
			logger.debug("AppointmentRequest has no AppointmentType.");
			throw new IllegalArgumentException(
					"AppointmentRequest has no AppointmentType.");
		}
		
		//load principals for already identified resources
		for (URI principal : request.getPrincipalUris()){
			ResourceOption resource;
			try {
				AttendeeType type = AttendeeType.getTypeByPrincipalURI(principal);
				resource = calDavDAO.getAttendeeByURI(principal, type);
			} catch (CalDavException e) {
				throw new ActionProcessingException("couldn't retrieve attendee principal from CalDAV", e);
			}
			messageUtils.putToResourceMap(message, resource.getType(), resource);
		}
		
		if (request.isTelehealth()){		
			Map<AttendeeType, List<ResourceOption>> resources = messageUtils.extractResourceMap(message);
			ResourceOption provider = resources.get(AttendeeType.PROVIDER).get(0);
			
			try {
				List<Clinic> clinics = calDavDAO.getClinicsByProvider(provider, AttendeeType.HUB_CLINIC);
				request.setHubClinic(clinics.get(0));
			} catch (CalDavException e) {
				throw new ActionProcessingException("Unable to identify a hub clinic for provider: " + provider.toString(), e);
			}
		}
		
		//identify candidate resources
		for (AttendeeType attendeeType : request.getRequiredResources()){
			List<ResourceOption> candidates;
			try {
				candidates = calDavDAO.getResourceCandidates(attendeeType, request, messageUtils.extractResourceMap(message));
			} catch (CalDavException e) {
				throw new ActionProcessingException("couldn't retrieve resource candidates for type: " + attendeeType, e);
			}
			messageUtils.putToResourceMap(message, attendeeType, candidates.toArray(new ResourceOption[0]));
		}
	
		//set appointment length
		try {
			if (request.isTelehealth()){
				request.setLength(calDavDAO.getAppointmentLength(request.getHubClinic().getPrincipalUri(),request.getAppointmentVisitType()));
			} else {
				request.setLength(calDavDAO.getAppointmentLength(request.getClinic().getPrincipalUri(), request.getAppointmentVisitType()));
			}
		} catch (CalDavException e) {
			throw new ActionProcessingException(
					"Unable to retrieve appointment length for type "
							+ request.getAppointmentVisitType() + " at URI "
							+ request.getClinic().getPrincipalUri().toString());
		}
		
		
		return message;
	}
	
}
