package gov.va.med.vos.service.actions;

import gov.va.med.vos.service.CalDavDAO;
import gov.va.med.vos.service.CalDavException;
import gov.va.med.vos.service.CalendarParserException;
import gov.va.med.vos.service.ConfigException;
import gov.va.med.vos.service.model.AppointmentCategory;
import gov.va.med.vos.service.model.AttendeeType;
import gov.va.med.vos.service.model.Calendar;
import gov.va.med.vos.service.model.Event;
import gov.va.med.vos.service.model.Location;
import gov.va.med.vos.service.model.Participant;
import gov.va.med.vos.service.util.MessageUtils;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;

import javax.xml.bind.JAXBException;

import org.jboss.soa.esb.actions.AbstractActionPipelineProcessor;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Accepts a Message containing a Calendar with one Event and communicates with
 * a CalDav server to schedule that Event for each participant.
 */
public class ScheduleAppointmentCaldavAction extends
		AbstractActionPipelineProcessor {

	private static Logger log = LoggerFactory
			.getLogger(ScheduleAppointmentCaldavAction.class);

	private MessageUtils messageUtils;

	private CalDavDAO calDavDAO;

	private String calDavUser;
	private String calDavPassword;

	/**
	 * Constructor.
	 * 
	 * @param configTree
	 * @throws JAXBException
	 * @throws URISyntaxException
	 * @throws CalDavException
	 * @throws ConfigException 
	 */
	public ScheduleAppointmentCaldavAction(ConfigTree configTree)
			throws JAXBException, URISyntaxException, CalDavException, ConfigException {

		this.calDavUser = configTree.getAttribute("calDavUser");
		this.calDavPassword = configTree.getAttribute("calDavPassword");

		calDavDAO = new CalDavDAO();

		messageUtils = new MessageUtils();
	}

	/**
	 * Process the Message by scheduling the included appointment.
	 * 
	 * @param Message
	 *            Must contain a Calendar with one Event that is the appointment
	 *            to be scheduled.
	 * @throws ActionProcessingException
	 *             If the appointment cannot not be scheduled.
	 */
	public Message process(Message message) throws ActionProcessingException {
		log.info(message.toString());

		// Extract the Calendar from the Message.
		Calendar calendar = null;
		try {
			calendar = messageUtils.extractCalendar(message);
		} catch (CalendarParserException e) {
			throw new ActionProcessingException(
					"Could not extract a Calendar from the Message body.", e);
		}

		// Get the appointment to schedule.
		List<Event> events = calendar.getEvents();
		if (1 != events.size()) {
			throw new ActionProcessingException(
					"Expected the Calendar in the Message to contain exactly one Event. It has: "
							+ events.size());
		}
		Event event = events.get(0);
		
		if (event.getCategories().contains(AppointmentCategory.TELEHEALTH)){
			scheduleTelehealthAppointment(event, message);
		} else {
			scheduleAppointment(event, message, event.getParticipants().toArray(new Participant[0]));
			
			List<Location> locations = event.getLocations();
			messageUtils.insertLocation(locations.get(0), message);
		}
		
		messageUtils.insertEvent(event, message);

		return message;
	}
	
	private void scheduleTelehealthAppointment(Event event, Message message) throws ActionProcessingException{
		Participant hubClinic = null, spokeClinic = null, hubRoom = null, spokeRoom = null, provider = null, patient = null;
		for (Participant p : event.getParticipants()){
			switch (p.getType()){
				
				case HUB_CLINIC:
					hubClinic = p;
					break;
				
				case LOCATION:
					spokeClinic = p;
					break;
				
				case HUB_CVT_ROOM:
					hubRoom = p;
					break;
					
				case SPOKE_CVT_ROOM:
					spokeRoom = p;
					break;
					
				case PATIENT:
					patient = p;
					break;
					
				case PROVIDER:
					provider = p;
					break;
					
				default:
					throw new ActionProcessingException("Did not recognize Participant type: " + p.getType());
			}
		}
		
		Event patientEvent = replaceParticipants(event, spokeClinic, spokeRoom, patient, provider);
		Event providerEvent = replaceParticipants(event, hubClinic, hubRoom, provider, patient);
		
		log.info("Patient VEVENT: " + patientEvent.toString());
		log.info("Provider VEVENT: " + providerEvent.toString());
		
		scheduleAppointment(patientEvent, message, spokeRoom, patient, spokeClinic, hubClinic);
		scheduleAppointment(providerEvent, message, hubClinic, hubRoom, provider);
		
		messageUtils.insertLocation(providerEvent.getLocations().get(0), message);
	}
	
	private Event replaceParticipants(Event e, Participant location, Participant ... attendees){
		Event event = e.clone();
		event.setLocation(location);
		event.clearAttendees();
		
		for (Participant p : attendees){
			event.addAttendee(p);
		}
		
		return event;
	}
	
	private void scheduleAppointment(Event event, Message message, Participant ... participants) throws ActionProcessingException{
		// Schedule the appointment for each Participant.
		// Avoid re-serializing the Calendar, to preserve newline
		// style.
		Calendar calendar = new Calendar();
		calendar.addEvent(event);
		String unparsedCalendarWithAppointment = calendar.getSerializedCalendar();
		for (Participant participant : participants) {
			//don't schedule hub clinic's calendar
			if (participant == null || participant.getName().equals("HOME")
					|| participant.getName().equals("OTHER")
					|| participant.getType() == AttendeeType.HUB_CLINIC) {
				continue;
			}
			
			String calendarUrl = buildEventUrl(participant, event);
			scheduleAppointment(calendarUrl, unparsedCalendarWithAppointment);
			
			if (participant.getPrincipalUri().toString().contains("patients")){
				messageUtils.insertPatient(participant, message);
			}
			
		}
	}

	private void scheduleAppointment(String calendarUrl,
			String calendarWithAppointment) throws ActionProcessingException {
		try {
			calDavDAO.scheduleEvent(calendarWithAppointment, calendarUrl,
					this.calDavUser, this.calDavPassword);
		} catch (CalDavException e) {
			throw new ActionProcessingException(
					"Could not schedule appointment on calendar at: "
							+ calendarUrl, e);
		}

	}

	private String getUnparsedCalendar(Message message) {
		String calendar = (String) message.getBody().get();
		return calendar;
	}

	private URI getCalendarUri(URI principalUri)
			throws ActionProcessingException {
		URI calendarUri = null;
		try {
			calendarUri = calDavDAO.getCalendarURI(principalUri);
		} catch (URISyntaxException e) {
			throw new ActionProcessingException(
					"Could not look up calendar URI", e);
		} catch (CalDavException e) {
			throw new ActionProcessingException(
					"Could not look up calendar URI", e);
		}
		return calendarUri;
	}

	/**
	 * Build a URL for the Event, under the URL for the Participant's calendar.
	 * 
	 * @param attendee
	 * @param event
	 * @return
	 * @throws ActionProcessingException
	 */
	private String buildEventUrl(Participant attendee, Event event)
			throws ActionProcessingException {

		// The base URL is the URI for the Attendee's calendar
		URI principalUri = attendee.getPrincipalUri();
		URI calendarUri = getCalendarUri(principalUri);
		String baseUrl = calendarUri.getPath();

		// The file name is the Event UID
		String fileName = event.getUid();

		// The Event URL is the base URL, plus the file name, plus the file
		// extension "ics".
		String calendarUrl = String.format("%s/%s.ics", baseUrl, fileName);

		return calendarUrl;
	}

	/**
	 * For testing use.
	 * 
	 * @param calDavDAO
	 */
	public void setCalDavDAO(CalDavDAO calDavDAO) {
		this.calDavDAO = calDavDAO;
	}

}
