package gov.va.med.vos.service.actions;

import gov.va.med.vos.service.CalDavDAO;
import gov.va.med.vos.service.CalDavException;
import gov.va.med.vos.service.CalDavParser;
import gov.va.med.vos.service.CalDavParserException;
import gov.va.med.vos.service.CalendarParserException;
import gov.va.med.vos.service.ConfigException;
import gov.va.med.vos.service.TextMatchPropertyFilter;
import gov.va.med.vos.service.model.AppointmentCategory;
import gov.va.med.vos.service.model.AppointmentRequest;
import gov.va.med.vos.service.model.AttendeeType;
import gov.va.med.vos.service.model.Clinic;
import gov.va.med.vos.service.util.MessageUtils;
import ietf.params.xml.ns.caldav.CalendarData;
import ietf.params.xml.ns.caldav.CalendarQuery;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;

import javax.xml.bind.JAXBException;

import net.fortuna.ical4j.model.DateRange;

import org.jboss.soa.esb.actions.AbstractActionLifecycle;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * An ESB Action that parses the Message contents into an AppointmentRequest.
 */
public class ParseAppointmentRequestAction extends AbstractActionLifecycle {
	
	private static Logger logger = LoggerFactory.getLogger(ParseAppointmentRequestAction.class);

	protected ConfigTree config;
	MessageUtils messageUtils;
	private CalDavParser calDavParser;
	private CalDavDAO calDavDAO;

	/**
	 * Constructor.
	 * 
	 * @param config
	 *            The runtime configuration determined by the ESB.
	 * @throws JAXBException 
	 * @throws CalDavException 
	 * @throws ConfigException 
	 */
	public ParseAppointmentRequestAction(ConfigTree config) throws JAXBException, CalDavException, ConfigException {
		this.config = config;
		this.messageUtils = new MessageUtils();
		this.calDavParser = new CalDavParser();
		this.calDavDAO = new CalDavDAO();
	}

	/**
	 * Parse the Message to add a VisitRequest Object that can be interpreted by
	 * the business rules.
	 * @throws JAXBException 
	 * @throws CalDavParserException 
	 * @throws CalendarParserException 
	 * @throws CalDavException 
	 */
	public Message parseAppointmentRequest(Message message) throws ActionProcessingException {

		String stringBody = (String) messageUtils.getDefaultMessageBody(message);

		CalendarQuery calendarQuery;
		try {
			calendarQuery = calDavParser.parseCalendarQuery(stringBody);
		} catch (CalDavParserException e1) {
			throw new ActionProcessingException("Error while parsing calendar-query", e1);
		}

		
		List<TextMatchPropertyFilter> filters = calDavParser.parseVEventPropertyFilters(calendarQuery);
		
		DateRange dateRange;
		try {
			dateRange = calDavParser.parseDateRange(calendarQuery);
		} catch (CalDavParserException e1) {
			throw new ActionProcessingException("couldn't parse time-range from calendar-query", e1);
		}
		
		try {
			//TODO turn this into a list of properties; stick it on the ESB message
			CalendarData calendarData = calDavParser.parseCalendarData(calendarQuery);
		} catch (CalDavParserException e1) {
			throw new ActionProcessingException("couldn't parse calendar-data from calendar-query", e1);
		}
		
		AppointmentRequest appointmentRequest = new AppointmentRequest();
		
		//parse filters
		String location = null;
		for (TextMatchPropertyFilter filter : filters){
			if (filter.getPropertyName().equals("ATTENDEE")){
				try {
					appointmentRequest.getPrincipalUris().add(new URI(filter.getMatchingText()));
				} catch (URISyntaxException e) {
					throw new ActionProcessingException("unable to parse attendee URI: " + filter.getMatchingText(), e);
				}
			} else if (filter.getPropertyName().equals("LOCATION")) {
				location = filter.getMatchingText();
			} else if (filter.getPropertyName().equals("CATEGORIES")) {
				try {
					appointmentRequest.getCategories().add(AppointmentCategory.getAppointmentCategory(filter.getMatchingText()));
				} catch (IllegalArgumentException e){
					throw new ActionProcessingException("Did not recognize category with name: " + filter.getMatchingText(), e);
				}
			} else if (filter.getPropertyName().equals("SUMMARY")){
				appointmentRequest.setSummary(filter.getMatchingText());
			} else {
				logger.info("Unknown prop filter " + filter.getPropertyName());
			}
		}
		
		appointmentRequest.setDateRange(dateRange);
		
		evaluateLocation(appointmentRequest, location);
		
		messageUtils.insertAppointmentRequest(appointmentRequest, message);
		
		logger.info(message.getBody().toString());
		
		return message;
	}
	
	private void evaluateLocation(AppointmentRequest request, String location) throws ActionProcessingException {
		if (location == null){
			//TODO error or location is derived
		} else if (location.equalsIgnoreCase("HOME")){
			Clinic clinic = new Clinic();
			clinic.setHome(true);
			clinic.setDisplayname("HOME");
			clinic.setType(AttendeeType.LOCATION);
			
			request.setClinic(clinic);
		} else if (location.equalsIgnoreCase("OTHER")){
			Clinic clinic = new Clinic();
			clinic.setHome(true);
			clinic.setDisplayname("OTHER");
			clinic.setType(AttendeeType.LOCATION);
			
			request.setClinic(clinic);
		} else {
			try {
				request.setClinic(calDavDAO.getClinic(new URI(location), true));
				
				String[] pieces = location.split("/");
				
				request.getClinic().setClinicAbbr(pieces[pieces.length - 1]);
				request.getClinic().setFacilityAbbr(pieces[pieces.length - 2]);
				
				logger.info(request.getClinic().toString());
			} catch (URISyntaxException e) {
				throw new ActionProcessingException("Unable to parse clinic principal URI", e);
			} catch (CalDavException e) {
				throw new ActionProcessingException("Unable to parse clinic principal URI", e);
			}
		}

	}

	/**
	 * Used for testing.
	 * 
	 * @param messageUtils
	 *            A replacement for the MessageUtils instance created when this
	 *            ParseVisitRequestAction was instantiated.
	 */
	protected void setMessageUtils(MessageUtils messageUtils) {
		this.messageUtils = messageUtils;
	}

}
