package gov.va.med.vos.service.actions;

import gov.va.med.scheduling.service.NotificationService;
import gov.va.med.vos.service.CalDavDAO;
import gov.va.med.vos.service.CalDavException;
import gov.va.med.vos.service.ConfigException;
import gov.va.med.vos.service.model.Attendee;
import gov.va.med.vos.service.model.Clinic;
import gov.va.med.vos.service.model.Event;
import gov.va.med.vos.service.model.Location;
import gov.va.med.vos.service.util.MessageUtils;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;

import org.jboss.soa.esb.actions.AbstractActionPipelineProcessor;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.actions.annotation.Process;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import vos.med.va.gov.SMS;

public class NotifyAttendeesAction extends AbstractActionPipelineProcessor {
	
	private static NotificationService service;
	private CalDavDAO calDavDAO;
	private MessageUtils messageUtils;
	
	private String email;
	
	private static Logger logger = LoggerFactory.getLogger(NotifyAttendeesAction.class);
	

	public NotifyAttendeesAction(ConfigTree config) throws CalDavException, ConfigException {
		this.email = config.getAttribute("email");
		this.messageUtils = new MessageUtils();
		
		service = new NotificationService();
		
		this.calDavDAO = new CalDavDAO();

	}
	
	@Process
	public Message process(Message message) throws ActionProcessingException {
		
		Attendee patient = messageUtils.extractPatient(message);
		Event event = messageUtils.extractEvent(message);
		Location location = messageUtils.extractLocation(message);
		
		List<Object> properties = new ArrayList<Object>();
		properties.add(new SMS());
		
		String sms;
		try {
			Map<String,String> response = calDavDAO.getPrincipalProperties(patient.getPrincipalUri(), properties);
			sms = response.get("SMS");
		} catch (CalDavException e) {
			logger.info("Unable to retrieve SMS number for patient: " + patient.getName(), e);
			sms = "";
		}
		
		Clinic clinic;
		try {
			clinic = calDavDAO.getClinic(location.getPrincipalUri(), true);
		} catch (CalDavException e) {
			throw new ActionProcessingException("Error retrieving clinic properties from CalDAV: " + location.getPrincipalUri().toString(), e);
		}
		
		Calendar cal = Calendar.getInstance();
		cal.setTime(event.getAppointmentTimestamp());
		TimeZone tz = TimeZone.getTimeZone(clinic.getTimezone());
		
		service.sendEmail(email, "VA Appointment", getNotificationText(patient.getName(), cal.getTime(), clinic.getDisplayname(), clinic.getPhoneNumber(), tz));
		if (!sms.equals("")){
			service.sendSMS(sms, getNotificationText(patient.getName(), cal.getTime(), clinic.getDisplayname(), clinic.getPhoneNumber(), tz));
		}
		
		return message;
	}
	
	private String getNotificationText(String patientName, Date datetime, String clinicName, String clinicPhoneNumber, TimeZone tz){
		SimpleDateFormat f0 = new SimpleDateFormat("hh:mm a z");
		SimpleDateFormat f1 = new SimpleDateFormat("EEE, d MMM yyyy");
		f0.setTimeZone(tz);
		f1.setTimeZone(tz);
		return String.format("Dear %s: You are scheduled at %s on %s for an appointment at %s.\n", 
				patientName, f0.format(datetime), f1.format(datetime), clinicName, clinicPhoneNumber);
	}

}
