package gov.va.med.vos.service.actions;

import gov.va.med.vos.service.CalDavDAO;
import gov.va.med.vos.service.CalDavException;
import gov.va.med.vos.service.ConfigException;
import gov.va.med.vos.service.model.AppointmentRequest;
import gov.va.med.vos.service.model.Attendee;
import gov.va.med.vos.service.model.AttendeeType;
import gov.va.med.vos.service.model.Calendar;
import gov.va.med.vos.service.model.Event;
import gov.va.med.vos.service.model.ResourceOption;
import gov.va.med.vos.service.util.MessageUtils;

import java.io.IOException;
import java.net.URISyntaxException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import net.fortuna.ical4j.model.TimeZone;

import org.jboss.soa.esb.actions.AbstractActionPipelineProcessor;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * Determine possible appointment times given a set of resources
 * and their calendars.
 * 
 *  Algorithm:
 *  	--sort attendees by type
 *  	--iterate through appointment slots attempting to find 
 *  		one available attendee for each type
 *  	--create an event if successful, otherwise continue to next slot
 *  	--slots are appointment-length time periods from 8-17 with start times 0.25 hours apart
 *  		e.g. 8.25 == 8:15am and 14.75 == 2:45pm
 *
 */
public class FindAppointmentTimes extends AbstractActionPipelineProcessor {
	
	private Logger log = LoggerFactory.getLogger(FindAppointmentTimes.class);
	
	private MessageUtils messageUtils;
	private CalDavDAO calDavDAO;
	
	public FindAppointmentTimes(ConfigTree config) throws CalDavException, ConfigException{
		this.messageUtils = new MessageUtils();
		this.calDavDAO = new CalDavDAO();
	}

	public Message process(Message message) throws ActionProcessingException {
		AppointmentRequest request = messageUtils.extractAppointmentRequest(message);
		
		log.info(request.toString());
		
		Map<AttendeeType,List<ResourceOption>> sorted = sortAttendees(message);
		List<Event> possibleAppointments;
		try {
			possibleAppointments = findAppointments(sorted, request);
		} catch (URISyntaxException e) {
			throw new ActionProcessingException(e);
		} catch (IOException e) {
			throw new ActionProcessingException(e);
		} catch (ParseException e) {
			throw new ActionProcessingException(e);
		}
		
		possibleAppointments = filterAppointments(sorted.get(AttendeeType.PATIENT).get(0), request, possibleAppointments);
		
		Calendar cal = new Calendar();
		cal.addEvent(possibleAppointments);
		
		messageUtils.insertAppointments(cal, message);
		
		return message;
	}
	
	private boolean isAvailable(Date start, int appointmentLength, ResourceOption resource){
		Calendar calendar = resource.getCalendar();
		return calendar.isAppointmentAvailable(start, appointmentLength);
	}
	
	private List<Event> findAppointments(Map<AttendeeType,List<ResourceOption>> sorted, AppointmentRequest request) throws URISyntaxException, IOException, ParseException{
		TimeZone.setDefault(TimeZone.getTimeZone("UTC"));
		
		List<Event> possibleAppointments = new ArrayList<Event>();
		Set<ResourceOption> appAttendees;
		boolean available;
		
		java.util.Calendar calVar = java.util.Calendar.getInstance();
		calVar.setTime(request.getStartDate());
		
		for (Date slot = calVar.getTime(); !calVar.getTime().after(request.getEndDate()); calVar.add(java.util.Calendar.MINUTE, 30), slot = calVar.getTime()){
			
			java.util.Calendar slotEnd = java.util.Calendar.getInstance();
			slotEnd.setTime(slot);
			slotEnd.add(java.util.Calendar.MINUTE, request.getLength());
			
			appAttendees = new HashSet<ResourceOption>();
			available = false;
			
			//find an attendee from each group
			for (Entry<AttendeeType, List<ResourceOption>> e : sorted.entrySet()){
				
				available = false;
				
				//find an attendee that is available from this group
				for (ResourceOption a : e.getValue()){
					if (isAvailable(slot, request.getLength(), a)){
						
						appAttendees.add(a);
						
						available = true;
						break;
					}
				}
				
				//No one from group is available for this slot
				if (!available){
					break;
				} 
			}
			
			if (available){
				Event e = new Event(slot, request, appAttendees);
				possibleAppointments.add(e);
			}
		}
		
		return possibleAppointments;
	}
	
	private List<Event> filterAppointments(ResourceOption patient, AppointmentRequest request, List<Event> appointments) throws ActionProcessingException{
		
		//build hashtable
		Map<String,List<Event>> map = new HashMap<String,List<Event>>();
		java.util.Calendar cal = java.util.Calendar.getInstance();
		String key;
		for (Event e : appointments){
			cal.setTime(e.getAppointmentTimestamp());
			
			key = String.valueOf(cal.get(java.util.Calendar.YEAR))
					+ String.valueOf(cal.get(java.util.Calendar.DAY_OF_YEAR))
					+ e.getVistaClinic().getPrincipalUri().toString();
			
			if (!map.containsKey(key)){
				map.put(key, new ArrayList<Event>());
			}
			map.get(key).add(e);
		}
		
		//retrieve the patient's existing appointments
		Calendar patientCalendar;
		try {
			patientCalendar = calDavDAO.getEvents(patient, request.getStartDate(), request.getEndDate());
		} catch (CalDavException e) {
			throw new ActionProcessingException("Unable to retrieve prior appointments for patient: " + patient, e);
		}
		
		//apply the one appointment per patient per clinic per day rule
		for (Event e : patientCalendar.getEvents()){
			cal.setTime(e.getAppointmentTimestamp());
			
			Attendee clinic = e.getVistaClinic();
			if (clinic == null){
				continue;
			}
			
			key = String.valueOf(cal.get(java.util.Calendar.YEAR))
					+ String.valueOf(cal.get(java.util.Calendar.DAY_OF_YEAR))
					+ clinic.getPrincipalUri().toString();
			
			map.remove(key);
		}
		
		//reform the appointments list
		appointments.clear();
		for (Entry<String,List<Event>> entry : map.entrySet()){
			appointments.addAll(entry.getValue());
		}
		
		return appointments;
	}
	
	private Map<AttendeeType,List<ResourceOption>> sortAttendees(Message message){
		Map<AttendeeType, List<ResourceOption>> sorted = messageUtils.extractResourceMap(message);
		
//		//sort attendees into attendee types
//		for (Attendee a : request.getAttendees()){
//			if (!sorted.containsKey(a.getType())){
//				sorted.put(a.getType(), new ArrayList<Attendee>());
//			}
//			sorted.get(a.getType()).add(a);
//		}
//		
//		//sort attendees by number of appointments within their group
//		for (List<Attendee> group : sorted.values()){
//			Collections.sort(group, new Comparator<Attendee>(){
//
//				public int compare(Attendee a1, Attendee a2) {
//					if (a1.getCalendar().countAppointments() < a2.getCalendar().countAppointments()){
//						return 1;
//					} else if (a1.getCalendar().countAppointments() > a2.getCalendar().countAppointments()){
//						return -1;
//					} else {
//						return 0;
//					}
//				}
//				
//			});
//		}
		
		return sorted;
	}
	
}
