package gov.va.med.vos.service;

import ietf.params.xml.ns.caldav.CalendarData;
import ietf.params.xml.ns.caldav.CalendarQuery;
import ietf.params.xml.ns.caldav.CompFilter;
import ietf.params.xml.ns.caldav.Filter;
import ietf.params.xml.ns.caldav.PropFilter;
import ietf.params.xml.ns.caldav.TextMatch;
import ietf.params.xml.ns.caldav.TimeRange;

import java.io.StringReader;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;

import net.fortuna.ical4j.model.DateRange;
import net.fortuna.ical4j.model.DateTime;

import org.apache.xerces.dom.ElementNSImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Node;

import dav.Multistatus;

public class CalDavParser {
	private static Logger logger = LoggerFactory.getLogger(CalDavParser.class);

	private JAXBContext jaxb;

	public CalDavParser() throws JAXBException {
		jaxb = JAXBContext.newInstance("ietf.params.xml.ns.caldav");
	}

	public CalendarData parseCalendarData(CalendarQuery query) throws CalDavParserException {

		Object unmarshalledObject = null;
		try {
			for (Object o : query.getProp().getContent()){
				if (o instanceof ElementNSImpl){
					ElementNSImpl ele = (ElementNSImpl) o;
					
					if (ele.getNodeName().equals("calendar-data")){
						unmarshalledObject = unmarshall(ele);
						break;
					}
				}
			}
			if (unmarshalledObject == null){
				throw new CalDavParserException("calendar-data element is missing");
			}
			
			// TODO: check cast
			CalendarData result = (CalendarData) unmarshalledObject;

			return result;
		} catch (JAXBException e) {
			throw new CalDavParserException("Couldn't unmarshall XML", e);
		}
	}

	public CalendarQuery parseCalendarQuery(String xml) throws CalDavParserException {

		Object unmarshalledObject;
		try {
			xml = correctNS(xml);
			
			unmarshalledObject = unmarshall(xml);
			
			// TODO: check cast
			CalendarQuery result = (CalendarQuery) unmarshalledObject;
			
			return result;
		} catch (JAXBException e) {
			throw new CalDavParserException("Couldn't unmarshall XML", e);
		}
	}
	
	public Multistatus parseMultistatus(String xml) throws CalDavParserException{
		
		Object unmarshalledObject;
		try {
			xml = correctNS(xml);
			
			unmarshalledObject = unmarshall(xml);
			
			// TODO: check cast
			Multistatus result = (Multistatus) unmarshalledObject;

			return result;
		} catch (JAXBException e) {
			throw new CalDavParserException("Couldn't unmarshall XML", e);
		}
	}

	public List<TextMatchPropertyFilter> parseVEventPropertyFilters(
			CalendarQuery calendarQuery) {

		List<TextMatchPropertyFilter> filters = new ArrayList<TextMatchPropertyFilter>();

		// And the calendar-query should contain a filter element
		Filter filter = calendarQuery.getFilter();
		if (null == filter) {
			logger.debug("caldav:calendar-query element has no caldav:filter child.");
			return filters;
		}

		//VCalendar
		CompFilter compFilter = filter.getCompFilter();
		if (null == compFilter) {
			logger.debug("caldav:filter element has no caldav:comp-filter child");
			return filters;
		}
		
		//VEvent
		CompFilter vEventCompFilter = compFilter.getCompFilter().get(0);
		
		
		//prop-filters
		List<PropFilter> propFilters = vEventCompFilter.getPropFilter();
		if (null == propFilters || propFilters.size() < 1) {
			logger.debug("caldav:comp-filter has no caldav:prop-filter children");
			return filters;
		}

		for (PropFilter propFilter : propFilters) {
			
			String name = propFilter.getName();
			
			// Get the text to match against
			TextMatch textMatch = propFilter.getTextMatch();
			if(null == textMatch) {
				logger.debug("caldav:param-filter with name=" + name + "does not specify a caldav:text-match");
				continue;
			}
			String matchingText = textMatch.getContent();

			TextMatchPropertyFilter textMatchPropertyFilter = new TextMatchPropertyFilter(name,matchingText);
			filters.add(textMatchPropertyFilter);
		}

		return filters;
	}
	
	public DateRange parseDateRange(CalendarQuery calendarQuery) throws CalDavParserException{
		TimeRange timeRange = calendarQuery.getFilter().getCompFilter().getCompFilter().get(0).getTimeRange();
		
		try {
			DateRange range = new DateRange(new DateTime(timeRange.getStart()), new DateTime(timeRange.getEnd()));
			return range;
		} catch (ParseException e) {
			throw new CalDavParserException("Couldn't parse time range", e);
		}
	}
	
	/**
	 * Replaces the invalid namespace "DAV:" with "DAV"
	 * 
	 * @param xml
	 * 			the xml to be corrected
	 * @return
	 * 			the corrected XML
	 */
	private String correctNS(String xml){
		return xml.replaceAll("xmlns=\"DAV:\"", "xmlns=\"DAV\"");
	}

	private Object unmarshall(String xml) throws JAXBException {
		Unmarshaller unmarshaller = jaxb.createUnmarshaller();
		StringReader reader = new StringReader(xml);
		Object unmarshalledObject = unmarshaller.unmarshal(reader);
		return unmarshalledObject;
	}
	
	private Object unmarshall(Node node) throws JAXBException{
		Unmarshaller unmarshaller = jaxb.createUnmarshaller();
		Object unmarshalledObject = unmarshaller.unmarshal(node);
		return unmarshalledObject;
	}

}
