package gov.va.med.vos.service;

import gov.va.med.vos.common.AuthSettings;
import gov.va.med.vos.common.CalDAVManager;
import gov.va.med.vos.common.Command;
import gov.va.med.vos.service.model.AppointmentCategory;
import gov.va.med.vos.service.model.AppointmentRequest;
import gov.va.med.vos.service.model.AttendeeType;
import gov.va.med.vos.service.model.Calendar;
import gov.va.med.vos.service.model.Clinic;
import gov.va.med.vos.service.model.ResourceOption;
import ietf.params.xml.ns.caldav.FreeBusyQuery;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import net.fortuna.ical4j.model.DateTime;

import org.apache.http.client.ClientProtocolException;
import org.apache.xerces.dom.ElementNSImpl;
import org.apache.xerces.dom.TextImpl;
import org.osaf.caldav4j.exceptions.DOMValidationException;
import org.osaf.caldav4j.model.request.CalDAVReportRequest;
import org.osaf.caldav4j.util.XMLUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;

import vos.med.va.gov.AppointmentLength;
import vos.med.va.gov.CalendarReference;
import vos.med.va.gov.CalendarTimezone;
import vos.med.va.gov.ClinicReference;
import vos.med.va.gov.Gender;
import vos.med.va.gov.IsTelehealthSupported;
import vos.med.va.gov.MaxConcurrentAppointments;
import vos.med.va.gov.NodePropertySearch;
import vos.med.va.gov.PatientVistAIEN;
import vos.med.va.gov.PhoneNumber;
import vos.med.va.gov.SMS;
import vos.med.va.gov.VistAClinicIEN;
import dav.Allprop;
import dav.Any;
import dav.Displayname;
import dav.Multistatus;
import dav.PropertySearch;
import dav.Propfind;
import dav.Response;


/**
 * Data Access Object for performing CRUD operations on a CalDav server
 * 
 */
public class CalDavDAO {

	private static Logger log = LoggerFactory.getLogger(CalDavDAO.class);
	
	private static String calDavURL, calDavUser, calDavPassword;
	
	private static final String PROVIDERS_PATH = "/server.php/vascheduling/principals/users/providers/";
	
	private static final String HTTP_NOT_FOUND = "HTTP/1.1 404 Not Found";
	private static final String HTTP_OK = "HTTP/1.1 200 OK";
	
	private CalDAVManager calDavManager;
	private JAXBContext jaxb;
	private CalendarParser vCalendarParser;
	private CalDavParser calDavParser;

	public CalDavDAO() throws CalDavException, ConfigException {
		Properties properties = new Properties();
		
		try {
			properties.load(this.getClass().getClassLoader().getResourceAsStream("/META-INF/vos.properties"));
			
			calDavURL = properties.getProperty("caldav_url");
			calDavUser = properties.getProperty("caldav_user");
			calDavPassword = properties.getProperty("caldav_password");
		} catch (FileNotFoundException e) {
			throw new ConfigException("Unable to locate vos.properties", e);
		} catch (IOException e) {
			throw new ConfigException("Error reading vos.properties", e);
		}
		
		try {
			calDavManager = new CalDAVManager(calDavURL, new AuthSettings(calDavUser, calDavPassword), null);
			jaxb = JAXBContext.newInstance(FreeBusyQuery.class, Propfind.class,
					CalendarReference.class, Displayname.class, Allprop.class,
					VistAClinicIEN.class, PatientVistAIEN.class, AppointmentLength.class,
					NodePropertySearch.class, ClinicReference.class, Gender.class,
					IsTelehealthSupported.class, CalendarTimezone.class, SMS.class, PhoneNumber.class,
					MaxConcurrentAppointments.class);
			calDavParser = new CalDavParser();
		} catch (JAXBException e) {
			throw new CalDavException("JAX-B exception", e);
		} catch (URISyntaxException e) {
			throw new CalDavException("Uri Syntax Exception", e);
		}

		vCalendarParser = new CalendarParser();

	}

	/**
	 * Queries CalDav for a VCalendar of VFreeBusy objects for a provider within
	 * a given time range
	 * 
	 * @param Attendee
	 *            the Attendee whose Calendar should be retrieved
	 * @param start
	 *            DateTime object representing the start of the range
	 * @param end
	 *            DateTime object representing the end of the range
	 * @return Calendar containing a list of VFreeBusy objects

	 */
	public Calendar getFreeBusyTimes(ResourceOption resource, DateTime start, DateTime end)
			throws CalDavException {
		log.info("Querying CalDav for free-busy times for " + resource.getDisplayname());
		
		String query = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><free-busy-query xmlns=\"urn:ietf:params:xml:ns:caldav\">"+
			"<time-range end=\"" + end.toString() + "\" start=\"" + start.toString() + "\"/>"+
		"</free-busy-query>";

		Command command;
		try {
			command = new Command("REPORT", resource.getCalendarUri().getPath(), query, new HashMap<String,String>(), false);
			Map<String, Object> response = calDavManager.executeCommand(command);
			log.info(response.toString());
			
			Calendar calendar;
			
			calendar = vCalendarParser.parse((String) response.get("response"));
			return calendar;
		} catch (URISyntaxException e) {
			throw new CalDavException("URI Syntax Exception", e);
		} catch (ClientProtocolException e) {
			throw new CalDavException("Client Protocol Exception", e);
		} catch (IOException e) {
			throw new CalDavException("IO Exception", e);
		} catch (CalendarParserException e) {
			throw new CalDavException("unable to parse response calendar", e);
		} 
		
		
	}
	
	public Calendar getEvents(ResourceOption resource, DateTime start, DateTime end) throws CalDavException{
		String query =   "<?xml version='1.0' encoding='UTF-8'?>"
					   + "<calendar-query xmlns=\"urn:ietf:params:xml:ns:caldav\">"
					   +   "<prop xmlns=\"DAV:\">"
					   +     "<getetag/>"
					   +     "<calendar-data xmlns=\"urn:ietf:params:xml:ns:caldav\">"
					   +       "<comp name=\"VCALENDAR\">"
					   +	     "<prop name=\"VERSION\"/>"
					   +	       "<comp name=\"VEVENT\" />"
					   +		   "</comp>"
					   +     "</calendar-data>"
					   +   "</prop>"
					   +   "<filter>"
					   +     "<comp-filter name=\"VCALENDAR\">"
					   +	   "<comp-filter name=\"VEVENT\">"
					   +	     "<time-range start=\"" + start.toString() + "\" end=\"" + end.toString() + "\" />" 
					   +	   "</comp-filter>"
					   +	 "</comp-filter>"
					   +   "</filter>"
					   + "</calendar-query>";

		Command command;
		try {
			command = new Command("REPORT", resource.getCalendarUri().getPath(), query, new HashMap<String,String>(), false);
			Map<String, Object> response = calDavManager.executeCommand(command);
			log.debug(response.toString());
			
			Multistatus multistatus = calDavParser.parseMultistatus((String) response.get("response"));
			

			Calendar calendar = new Calendar();
			for (Response r : multistatus.getResponse()){
				
				String unparsedCalendar = "";
				Calendar cal;
				for (Object o : r.getPropstat().get(0).getProp().getContent()){
					if (o instanceof ElementNSImpl){
						ElementNSImpl ele = (ElementNSImpl) o;
						
						if (!ele.getNodeName().equals("calendar-data")){
							continue;
						} else {
							unparsedCalendar = ele.getFirstChild().getTextContent();
							break;
						}
					}
				}
				
				cal = vCalendarParser.parse(unparsedCalendar);
				calendar.addEvent(cal.getEvents());
			}
			
			
			return calendar;
		} catch (URISyntaxException e) {
			throw new CalDavException("URI Syntax Exception", e);
		} catch (ClientProtocolException e) {
			throw new CalDavException("Client Protocol Exception", e);
		} catch (IOException e) {
			throw new CalDavException("IO Exception", e);
		} catch (CalendarParserException e) {
			throw new CalDavException("unable to parse response calendar", e);
		} catch (CalDavParserException e) {
			throw new CalDavException("Unable to parse calendar", e);
		} 
	}
	
	public Clinic getClinic(URI clinicURI, boolean isLocation) throws CalDavException{
		Propfind propfind = new Propfind();
		Any props = new Any();
		
		
		propfind.setProp(props);
		
		List<Object> properties = new ArrayList<Object>();
		properties.add(new CalendarReference());
		properties.add(new Displayname());
		properties.add(new CalendarTimezone());
		properties.add(new PhoneNumber());
		properties.add(new MaxConcurrentAppointments());
		
		Map<String, String> result;
		result = getPrincipalProperties(clinicURI, properties);
		
		Clinic clinic = new Clinic();
		
		try {
			clinic.setCalendarUri(new URI(result.get("CalendarReference")));
		} catch (URISyntaxException e) {
			throw new CalDavException("unable to parse clinic calendar URI", e);
		}
		clinic.setDisplayname(result.get("displayname"));
		clinic.setMaxConcurrentAppointments(Integer.valueOf(result.get("MaxConcurrentAppointments")));
		clinic.setPhoneNumber(result.get("PhoneNumber"));
		clinic.setTimezone(result.get("Calendar-timezone"));
		
		if (isLocation){
			clinic.setType(AttendeeType.LOCATION);
		} else {
			clinic.setType(AttendeeType.HUB_CLINIC);
		}
			
		clinic.setPrincipalUri(clinicURI);
		
		return clinic;
	}
	
	public List<ResourceOption> getResourceCandidates(AttendeeType type, AppointmentRequest request, Map<AttendeeType, List<ResourceOption>> resources) throws CalDavException{
		switch (type) {
		
			case PROVIDER:
				return getProvidersByClinic(request.getClinic());
				
			case HUB_CVT_ROOM:
				if (!request.isTelehealth()){
					throw new CalDavException("Should not be retrieving hub telehealth rooms for non-telehealth appointments");
				}
				Clinic hubClinic = request.getHubClinic();
				
				List<ResourceOption> rooms = getRoomsByClinic(hubClinic);
				
				List<Object> properties = new ArrayList<Object>();
				properties.add(new IsTelehealthSupported());
				for (ResourceOption room : rooms){
					Map<String,String> results = getPrincipalProperties(room.getPrincipalUri(), properties);
					if (results.get("IsTelehealthSupported").equals("false")){
						rooms.remove(room);
					}
				}
				
				return rooms;
				
			case SPOKE_CVT_ROOM:
				return getRoomsByClinic(request.getClinic());
				
			default:
				throw new CalDavException("unrecognized AttendeeType");
		}
	}
	
	public int getAppointmentLength(URI clinicUri, AppointmentCategory category) throws CalDavException{
		List<Object> properties = new ArrayList<Object>();
		properties.add(new AppointmentLength());
		
		String uri = calDavURL + clinicUri.getPath() + "/Visits/" + category.getUriValue();
		
		try {
			Map<String,String> response = getPrincipalProperties(new URI(uri), properties);
			return Integer.valueOf(response.get("AppointmentLength"));
		} catch (CalDavException e) {
			throw e;
		} catch (URISyntaxException e) {
			throw new CalDavException("Unable to build clinic visits URI: " + uri, e);
		}
	}
	
	/**
	 * Retrieves an Attendee principal from CalDAV
	 * 
	 * The calendarURI property will be set on the returned Attendee object
	 * 
	 * @param principalURI
	 * 				the full URL of the principal on the CalDAV server
	 * @return 
	 * 				Attendee with the calendarURI field set
	 */
	public ResourceOption getAttendeeByURI(URI principalURI, AttendeeType type) throws CalDavException{
		ResourceOption resource;
		if (type == AttendeeType.HUB_CLINIC || type == AttendeeType.LOCATION){
			resource = new Clinic();
		} else {
			resource = new ResourceOption();
		}
		resource.setPrincipalUri(principalURI);
		resource.setType(type);
		
		List<Object> properties = new ArrayList<Object>();
		properties.add(new CalendarReference());
		properties.add(new Displayname());
		
		Map<String,String> principalProperties = getPrincipalProperties(principalURI, properties);
		
		try {
			resource.setCalendarUri(new URI(principalProperties.get("CalendarReference")));
		} catch (URISyntaxException e) {
			throw new CalDavException("Unable to parse URI: "
					+ principalProperties.toString() + " from principal at "
					+ principalURI.toString(), e);
		}
		resource.setDisplayname(principalProperties.get("displayname"));
		
		return resource;
	}
	
	/**
	 * Fetches the calendarReference field from a CalDAV principal
	 * 
	 * @param principalURI
	 * 				full URL of the principal on the CalDAV server
	 * @return
	 * 				a URI pointing the principal's calendar
	 */
	public URI getCalendarURI(URI principalURI) throws URISyntaxException, CalDavException{
		
		List<Object> properties = new ArrayList<Object>();
		properties.add(new CalendarReference());
		
		try {
			Map<String,String> response = getPrincipalProperties(principalURI, properties);
			return new URI(response.get("CalendarReference"));
		} catch (URISyntaxException e) {
			throw new CalDavException("Unable to parse returned URI" , e);
		}
	}
	
	/**
	 * Retrieves arbitrary properties from a CalDAV principal
	 * 
	 * @param principalURI
	 * 				the URL of the principal for which properties should be retrieved
	 * @param properties
	 * 				the List of properties; each property should be a JAX-B object
	 * 				<p>
	 * 				e.g. to retrieve a CalendarReference this List should contain a gov.va.med.vos.CalendarReference object
	 * @return
	 * 		a Map of the retrieved properties; keys correspond to the name of the field in the CalDAV server
	 */
	public Map<String, String> getPrincipalProperties(URI principalURI, List<Object> properties) throws CalDavException{
		
		Propfind propFind = new Propfind();
		Any prop = new Any();
		prop.getContent().addAll(properties);
		propFind.setProp(prop);
		
		
		Map<String, String> httpHeaders = new HashMap<String, String>();
		httpHeaders.put("DEPTH", "0");
		
		Command command;
		Map<String,Object> response = null;
		try {
			command = new Command("PROPFIND", principalURI.getPath(), prettyPrint(propFind),
					httpHeaders, false);
			response = calDavManager.executeCommand(command);
			
			Multistatus multistatus = (Multistatus) calDavParser.parseMultistatus((String) response.get("response"));
			
			Response r = multistatus.getResponse().get(0);
			
			Map<String,String> principalProperties = new HashMap<String,String>();
			for (Object o : r.getPropstat().get(0).getProp().getContent()){
				if (o instanceof ElementNSImpl){
					ElementNSImpl element = (ElementNSImpl) o;
					TextImpl text = (TextImpl) element.getFirstChild();
					principalProperties.put(element.getNodeName(), text.getData());
				}
			}
			
			return principalProperties;
			
		} catch (ClientProtocolException e) {
			throw new CalDavException("client protocol exception", e);
		} catch (IOException e) {
			throw new CalDavException("IO exception", e);
		} catch (CalDavParserException e) {
			throw new CalDavException("couldn't parse response as MultiStatus", e);
		} catch (NullPointerException e){
			throw new CalDavException("URI: " + principalURI.toString() + "\nResponse: " + response.toString(), e);
		} catch (URISyntaxException e) {
			throw new CalDavException("HTTP client URI syntax exception", e);
		} catch (ClassCastException e) {
			throw new CalDavException("Http response exception", e);
		}
	}
	
	/**
	 * Performs a node-property-search against the CalDAV server to fetch principals that
	 * match a specified property.
	 * 
	 * @param nodeURI
	 * 				the root URI to begin the search at
	 * @param property
	 * 				the JAX-B object representing the property to search on
	 * @param match
	 * 				the matching text
	 * @return
	 * 			a List of principal URIs
	 */
	public List<URI> getPrincipalsByProperty(URI nodeURI, Object property, String match) throws CalDavException{
		
		NodePropertySearch nodePS = new NodePropertySearch();
		
		Any prop = new Any();
		prop.getContent().add(new Displayname());
		
		nodePS.setProp(prop);
		
		PropertySearch ps = new PropertySearch();
		
		prop = new Any();
		prop.getContent().add(property);
		
		ps.setProp(prop);
		ps.setMatch(match);

		nodePS.setPropertySearch(ps);
		
		
		Command command;
		Map<String, Object> response = null;
		try {
			command = new Command("REPORT", nodeURI.getPath(), prettyPrint(nodePS), new HashMap<String,String>(), false);
			response = calDavManager.executeCommand(command);
			
			Multistatus multistatus = (Multistatus) calDavParser.parseMultistatus((String) response.get("response"));
			
			List<URI> principals = new ArrayList<URI>();
			for (Response r : multistatus.getResponse()){

				String[] pieces = r.getFirstHref().split(nodeURI.getPath())[1].split("/");
				
				if (pieces[0].equals("")){
					principals.add(new URI(nodeURI.toString() + "/" + pieces[1]));
				} else {
					principals.add(new URI(nodeURI.toString() + pieces[0]));
				}
				
			}
			
			return principals;
		} catch (URISyntaxException e) {
			throw new CalDavException("Unable to parse URI", e);
		} catch (ClientProtocolException e) {
			throw new CalDavException("", e);
		} catch (IOException e) {
			throw new CalDavException("", e);
		} catch (CalDavParserException e) {
			throw new CalDavException("Unable to parse response into Multistatus");
		} catch (ClassCastException e){
			throw new CalDavException("URL: " + nodeURI.toString() + "\nResponse: " + response.toString(), e);
		}
		
	}
	
	/**
	 * Fetch the providers for an appointment request
	 * 
	 * 
	 * @param request
	 *            the AppointmentRequest
	 * @return null
	 * @throws CalDavException 
	 */
	public List<ResourceOption> getProvidersByClinic(Clinic clinic) throws CalDavException {
		
		try {
			List<URI> principals = getPrincipalsByProperty(new URI(calDavURL + PROVIDERS_PATH), new ClinicReference(), clinic.getPrincipalUri().toString());
			
			List<ResourceOption> resources = new ArrayList<ResourceOption>();
			for (URI uri : principals){
				resources.add(getAttendeeByURI(uri, AttendeeType.PROVIDER));
			}
			
			return resources;
		} catch (URISyntaxException e) {
			throw new CalDavException("Unable to build node URI", e);
		}
		
	}
	
	public List<ResourceOption> getRoomsByClinic(Clinic clinic) throws CalDavException {
		try {
			AttendeeType type;
			if (clinic.getType() == AttendeeType.HUB_CLINIC){
				type = AttendeeType.HUB_CVT_ROOM;
			} else {
				type = AttendeeType.SPOKE_CVT_ROOM;
			}
			
			String path = clinic.getPrincipalUri().getPath();
			String[] pieces = path.split("/");
			
			if (pieces[pieces.length - 1].equals("")){
				pieces[pieces.length - 2] = "Rooms";
			} else {
				pieces[pieces.length - 1] = "Rooms";
			}
			
			path = "";
			for (String piece : pieces){
				if (piece.equals("")){
					continue;
				}
				path += "/" + piece;
			}
			
			List<URI> principals = getPrincipalsByProperty(
					new URI(calDavURL
							+ path),
					new ClinicReference(), clinic.getPrincipalUri().toString());
			
			List<ResourceOption> resources = new ArrayList<ResourceOption>();
			for (URI uri : principals){
				resources.add(getAttendeeByURI(uri, type));
			}
			
			return resources;
		} catch (URISyntaxException e) {
			throw new CalDavException("Unable to build node URI", e);
		}
	}
	
	public List<Clinic> getClinicsByRoom(URI roomURI, AttendeeType type) throws CalDavException{
		List<ResourceOption> resources = getChildPrincipals(roomURI, "", new ClinicReference(), type);
		List<Clinic> clinics = new ArrayList<Clinic>();
		for (ResourceOption r : resources){
			clinics.add((Clinic) r);
		}
		return clinics;
	}
	
	public List<Clinic> getClinicsByProvider(ResourceOption provider, AttendeeType type) throws CalDavException{
		List<ResourceOption> resources = getChildPrincipals(provider.getPrincipalUri(), "Clns", new ClinicReference(), type);
		List<Clinic> clinics = new ArrayList<Clinic>();
		for (ResourceOption r : resources){
			clinics.add((Clinic) r);
		}
		return clinics;
	}
	
	private List<ResourceOption> getChildPrincipals(URI parentURI, String additionalPath, Object childRefField, AttendeeType childType) throws CalDavException{
		Propfind propFind = new Propfind();
		Any prop = new Any();
		prop.getContent().add(childRefField);
		propFind.setProp(prop);

		Command command;
		try {
			command = new Command("PROPFIND", parentURI.getPath() + "/" + additionalPath, prettyPrint(propFind), new HashMap<String,String>(), false);
			Map<String, Object> response = calDavManager.executeCommand(command);
			
			Multistatus multistatus = (Multistatus) calDavParser.parseMultistatus((String) response.get("response"));
			
			List<URI> childURIs = new ArrayList<URI>();
			for (Response r : multistatus.getResponse()){
				if (r.getPropstat().get(0).getStatus().equals(HTTP_NOT_FOUND)){
					continue;
				}
				
				for (Object o : r.getPropstat().get(0).getProp().getContent()){
					if (o instanceof ElementNSImpl){
						ElementNSImpl element = (ElementNSImpl) o;
						TextImpl text = (TextImpl) element.getFirstChild();
						childURIs.add(new URI(text.getData()));
					}
				}
				
			}
			
			List<ResourceOption> resources = new ArrayList<ResourceOption>(); 
			for (URI uri : childURIs){
				ResourceOption c = getAttendeeByURI(uri, childType);
				resources.add(c);
			}
			
			return resources;
		} catch (URISyntaxException e) {
			throw new CalDavException("Unable to parse URI", e);
		} catch (ClientProtocolException e) {
			throw new CalDavException("", e);
		} catch (IOException e) {
			throw new CalDavException("", e);
		} catch (CalDavParserException e) {
			throw new CalDavException("Unable to parse response into Multistatus");
		}
	}
	
	/**
	 * Schedules an event.
	 * @param eventCalendar A VCalendar String with the event on it.
	 * @param eventUrl The URL to which the event will be PUT.
	 * @throws CalDavException If the event could not be scheduled.
	 */
	public void scheduleEvent(String eventCalendar, String eventPath, String calDavUser, String calDavPassword)
			throws CalDavException {
		log.info("PUT to Calendar URL: " + calDavURL + eventPath);
		log.trace("PUT Body: " + eventCalendar);
		try {
			Command c = new Command("PUT", eventPath, eventCalendar,
					new HashMap<String, String>(), true);
			calDavManager.executeCommand(c);

		} catch (URISyntaxException e) {
			throw new CalDavException("Bad calendar URL: " + calDavURL + eventPath, e);
		} catch (ClientProtocolException e) {
			throw new CalDavException("Error sending request to CalDav", e);
		} catch (IOException e) {
			throw new CalDavException("Error sending request to CalDav", e);
		}

		log.info("PUT complete.");
	}


	/**
	 * Print a CalDav request as formatted XML
	 * 
	 * @param request
	 *            the CalDavReportRequest to be printed
	 * @return the formatted XML as a String
	 */
	public String prettyPrint(CalDAVReportRequest request) {
		Document doc;
		try {
			doc = request.createNewDocument(XMLUtils.getDOMImplementation());
			return uncorrectNamespace(XMLUtils.toPrettyXML(doc));
		} catch (DOMValidationException e) {
			e.printStackTrace();
		}

		return null;
	}

	/**
	 * Print a CalDav request as formatted XML
	 * 
	 * @param request
	 *            the JAX-B object to be printed
	 * @return the formatted XML as a String
	 */
	public String prettyPrint(Object o) {
		try {
			Marshaller marshaller = jaxb.createMarshaller();
			StringWriter newMessageBodyWriter = new StringWriter();
			marshaller.marshal(o, newMessageBodyWriter);
			String xml = newMessageBodyWriter.toString();

			Source xmlInput = new StreamSource(new StringReader(xml));
			StreamResult xmlOutput = new StreamResult(new StringWriter());

			Transformer transformer = TransformerFactory.newInstance()
					.newTransformer(); // An identity transformer
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");
			transformer.setOutputProperty(
					"{http://xml.apache.org/xslt}indent-amount", "2");
			transformer.transform(xmlInput, xmlOutput);

			return uncorrectNamespace(xmlOutput.getWriter().toString());
		} catch (JAXBException e) {
			e.printStackTrace();
		} catch (TransformerConfigurationException e) {
			e.printStackTrace();
		} catch (TransformerFactoryConfigurationError e) {
			e.printStackTrace();
		} catch (TransformerException e) {
			e.printStackTrace();
		}
		return null;
	}
	
	
	/**
	 * transforms the namespace DAV into DAV:
	 */
	private String uncorrectNamespace(String xml){
		return xml.replaceAll("=\"DAV\"", "=\"DAV:\"");
	}
	
	
}
