package gov.va.med.vos.engine.batch;

import gov.va.med.vos.common.CalDAVManager;

import java.io.PrintWriter;
import java.net.URISyntaxException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;

import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.HttpResponseException;
import org.apache.log4j.LogManager;

class WorkerThread implements Runnable {

	private final BatchUtil batchUtil;

	private ExecutorService parentService;
	private String groupName;
	private List<Request> requests;
	private CalDAVManager manager;

	private PrintWriter out;

	public WorkerThread(BatchUtil batchUtil, ExecutorService parentService,
			String groupName, List<Request> requests, RequestContext context)
			throws URISyntaxException {
		this.batchUtil = batchUtil;
		this.out = batchUtil.out;

		this.parentService = parentService;
		this.groupName = groupName;
		this.requests = requests;
		this.manager = context.createManager();
	}

	@Override
	public void run() {
		try {
			out.println("Executing request group "
					+ StringUtils.defaultString(groupName, "(default)"));
			for (Request request : requests) {
				executeCommand(request);
			}
		} catch (HttpResponseException e) {
			out.println("Stopping batch due to failure (add "
					+ "failOnError=\"false\" to this command to ignore error and proceed)");
			batchUtil.error = e;
		} catch (Exception e) {
			e.printStackTrace(out);
			batchUtil.error = e;
			parentService.shutdownNow();
		}
	}

	static Object requestSemaphore = new Object();

	private void executeCommand(Request request) throws Exception {
		Map<String, Object> resultMap = null;
		int currentRequestNum = -1;
		synchronized (requestSemaphore) {
			currentRequestNum = batchUtil.totalRequestsProcessed++;

			out.println("Executing request " + currentRequestNum + ": "
					+ request);

			for (Map.Entry<String, String> entry : request.getCommand()
					.getHeaders().entrySet()) {
				if ("depth".equalsIgnoreCase(entry.getKey()))
					out.println("Depth: " + entry.getValue());
			}

			if (request.isLogHeaders()) {
				/*
				 * If we care about logging headers, execute the command inside
				 * of synchronized block to keep header logging next to the
				 * initial PUT
				 */
				LogManager.getLogger("org.apache.http.headers").setLevel(
						org.apache.log4j.Level.ALL);
				resultMap = manager.executeCommand(request.getCommand());
			} else {
				LogManager.getLogger("org.apache.http.headers").setLevel(
						batchUtil.configuredHeaderLoggingLevel);
			}
		}

		/*
		 * If we don't care about logging headers, execute the command outside
		 * of synchronized
		 */
		if (!request.isLogHeaders()) {
			resultMap = manager.executeCommand(request.getCommand());
		}

		int statusCode = (Integer) resultMap.get("statusCode");
		synchronized (requestSemaphore) {
			out.println("Request " + currentRequestNum + " Response: Status "
					+ statusCode + ":");
			Exception e = (Exception) resultMap.get("responseException");
			if (e != null) {
				out.println("Exception: " + e.getMessage());
			}

			Object resp = resultMap.get("response");
			if (resp != null)
				out.println(resp);
			if (request.isFailOnError() && e != null)
				throw e;
		}
	}
}