package gov.va.med.vos.engine.batch;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.apache.commons.io.output.TeeOutputStream;
import org.apache.commons.lang3.time.DurationFormatUtils;
import org.apache.log4j.LogManager;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

public class BatchUtil {

	org.apache.log4j.Level configuredHeaderLoggingLevel;
	PrintWriter out;

	int totalRequestsProcessed = 0;
	int numThreads;
	Exception error = null;

	public BatchUtil(OutputStream os, int numThreads) {
		out = new PrintWriter(os, true);
		this.numThreads = numThreads;
		configuredHeaderLoggingLevel = LogManager.getLogger(
				"org.apache.http.headers").getEffectiveLevel();
	}

	public void runBatch(InputStream is) throws Exception {

		Document document = PositionalXMLReader.readXML(is);
		Element root = document.getDocumentElement();

		RequestContext context = new RequestContext(root);

		Map<String, Element> templateNodeMap = new HashMap<String, Element>();

		List<Element> requestGroups = new ArrayList<Element>();
		List<Element> grouplessRequests = new ArrayList<Element>();
		
		try {
			for (Node node : DOMUtil.iterable(root.getChildNodes())) {
				if (node.getNodeType() != Node.ELEMENT_NODE)
					continue;
				Element element = (Element) node;

				if ("bodyTemplate".equals(node.getLocalName())) {
					templateNodeMap.put(element.getAttribute("id"), element);
				} else if ("requestGroup".equals(node.getLocalName())) {
					requestGroups.add(element);
				} else {
					grouplessRequests.add(element);
				}
			}
		} catch (Exception e) {
			e.printStackTrace(out);
			out.close();
			throw e;
		}

		LinkedHashMap<String, List<Request>> requestGroupMap = new LinkedHashMap<String, List<Request>>();

		for (Element requestGroupEl : requestGroups) {
			String groupName = requestGroupEl.getAttribute("name");

			List<Request> requestList = createRequestList(context,
					templateNodeMap,
					DOMUtil.iterable(requestGroupEl.getChildNodes()));
			requestGroupMap.put(groupName, requestList);
		}

		if (!grouplessRequests.isEmpty()) {
			List<Request> requestList = createRequestList(context,
					templateNodeMap, grouplessRequests);
			requestGroupMap.put(null, requestList);
		}

		ExecutorService executor = Executors.newFixedThreadPool(numThreads);
		for (Map.Entry<String, List<Request>> entry : requestGroupMap
				.entrySet()) {
			Runnable worker = new WorkerThread(this, executor, entry.getKey(),
					entry.getValue(), context);
			executor.execute(worker);
		}

		executor.shutdown();

		while (!executor.isTerminated()) {
			Thread.sleep(1000);
		}
	}

	public List<Request> createRequestList(RequestContext context,
			Map<String, Element> templateNodeMap, Iterable<? extends Node> nodes)
			throws Exception {
		List<Request> requestList = new ArrayList<Request>();
		for (Node n : nodes) {
			if (n.getNodeType() != Node.ELEMENT_NODE)
				continue;
			Element requestNode = (Element) n;

			try {
				Element overrideBodyNode = null;
				if (requestNode.hasAttribute("bodyTemplateId")) {
					overrideBodyNode = templateNodeMap.get(requestNode
							.getAttribute("bodyTemplateId"));
				}
				Request request = Request.buildRequest(context, requestNode,
						overrideBodyNode);
				requestList.add(request);
			} catch (Exception e) {
				throw new Exception(
						"There was an error processing the "
								+ requestNode.getNodeName()
								+ " command on line "
								+ requestNode
										.getUserData(PositionalXMLReader.LINE_NUMBER_KEY_NAME),
						e);

			}
		}
		return requestList;
	}

	public static void main(String[] args) throws Exception {
		if (args.length == 0) {
			System.out.println("Usage: java " + BatchUtil.class.getName()
					+ " <input XML file> [number of concurrent threads]");
			System.exit(0);
		}

		String logFileName = args[0].contains(".") ? args[0].substring(0,
				args[0].lastIndexOf(".")) + ".log" : args[0] + ".log";
		FileOutputStream fos = new FileOutputStream(logFileName);
		TeeOutputStream tos = new TeeOutputStream(fos, System.out);

		int numThreads = -1;
		try {
			numThreads = args.length > 1 ? Integer.parseInt(args[1]) : 10;
		} catch (NumberFormatException e) {
			System.out.println("The second argument must be an integer!");
			System.exit(1);
		}
		System.out.println("Executing with " + numThreads + " threads");
		
		Throwable t = null;

		long startMillis = System.currentTimeMillis();
		try {
			BatchUtil bu = new BatchUtil(tos, numThreads);
			bu.runBatch(new FileInputStream(args[0]));
			if (bu.error != null)
				t = bu.error;
		} catch (Exception e) {
			System.err.println("There was an unexpected error:");
			e.printStackTrace(System.err);
			t = e;
		}
		long endMillis = System.currentTimeMillis();
		System.out.println("The above log was saved to \"" + logFileName
				+ "\".");
		System.out.println("Time elapsed: "
				+ DurationFormatUtils.formatDurationWords(endMillis
						- startMillis, true, true));

		if (t != null) {
			System.err.println("(exiting with status 1)");
			System.exit(1);
		}
	}
}
