package gov.va.medora.mdws.scheduling.manager;

import gov.va.medora.mdws.scheduling.exception.MDWSSchedulingException;
import gov.va.medora.mdws.schedulingsvc.AppointmentTO;
import gov.va.medora.mdws.schedulingsvc.AppointmentTypeTO;
import gov.va.medora.mdws.schedulingsvc.HospitalLocationTO;
import gov.va.medora.mdws.schedulingsvc.PatientArray;
import gov.va.medora.mdws.schedulingsvc.PatientTO;
import gov.va.medora.mdws.schedulingsvc.SchedulingSvc;
import gov.va.medora.mdws.schedulingsvc.SchedulingSvcSoap;
import gov.va.medora.mdws.schedulingsvc.TaggedText;

import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.TimeZone;

import javax.xml.ws.BindingProvider;

/**
 * The Class SchedulingManager. This manager provides
 * an interface/API into JAX-WS/Apache CXF web service client
 * objects and methods created for interacting with MDWS
 * scheduling web service methods. It handles the setup 
 * and teardown methods for interacting with MDWS scheduling
 * and provides two orchestration methods for making and 
 * cancelling an appointment.  The orchestration methods 
 * leverage the select web service method to select the patient
 * and add to the session, otherwise you would need to call the select 
 * patient method within this manager prior to using the make or create 
 * appointment methods.
 * <p>
 *  <A HREF="http://www.hp.com"> Hewlett-Packard Enterprise Services - OSEHRA/VA Scheduling Contest</A> 
 * </p>
 * @author <a href="mailto:david.aar.lucas@hp.com">Aaron Lucas</a>
 */
public class SchedulingManager {
	
	/** The service. */
	private SchedulingSvcSoap service;
	
	/** The site id. */
	private String siteId = "";
	
	/** The user name. */
	private String userName = "";
	
	/** The password. */
	private String password = "";
	
	/** The context. */
	private String context = "";
	
	
	/** The scheduling svc wsdl. */
	private String schedulingSvcWsdl = null;
	
	/** The Constant DEFAULT_CONTEXT. */
	public static final String DEFAULT_CONTEXT = "OR CPRS GUI CHART";
	
	/** The Constant VISTA_FORMAT. */
	private final static SimpleDateFormat VISTA_FORMAT = new SimpleDateFormat("yyyyMMdd.HHmmss");
	
	
	/**
	 * Instantiates a new scheduling manager.
	 *
	 * @param schedulingSvcWsdl the scheduling svc wsdl
	 * @param siteId the site id
	 * @param userName the user name
	 * @param password the password
	 * @param context the context
	 */
	public SchedulingManager(String schedulingSvcWsdl, String siteId, String userName, String password, String context)
	{
		this.schedulingSvcWsdl = schedulingSvcWsdl;
		this.siteId = siteId;
		this.userName = userName;
		this.password = password;
		this.context = context;
	}
	
	
	/**
	 * Gets the patients by clinic.
	 *
	 * @param clinicIEN the clinic ien
	 * @param fromDate the from date
	 * @param toDate the to date
	 * @param timezone the timezone
	 * @return the patients by clinic
	 * @throws Exception the exception
	 */
	public List<PatientTO> getPatientsByClinic(String clinicIEN, Date fromDate, Date toDate, String timezone) throws Exception
	{
		if(isNullOrBlank(clinicIEN))
		{
			throw new MDWSSchedulingException("You must supply the clinic IEN");
		}
		if(isNullOrBlank(timezone))
		{
			throw new MDWSSchedulingException("You must supply the timezone of the appointment.");
		}
		connect();
		PatientArray patients = service.getPatientsByClinic(clinicIEN, formatForVISTA(fromDate, timezone), formatForVISTA(toDate, timezone));
		disconnect();
		return patients.getPatients().getPatientTO();
		
	
	}
	
	
	/**
	 * Orchestrated make patient appointment.
	 *
	 * @param patientIEN the patient ien
	 * @param clinicIEN the clinic ien
	 * @param appointmentTimestamp the appointment timestamp
	 * @param timezone the timezone
	 * @param appointmentType the appointment type
	 * @param appointmentLength the appointment length
	 * @param schedulingRequestType the scheduling request type
	 * @return the appointment to
	 * @throws Exception the exception
	 */
	public AppointmentTO orchestratedMakePatientAppointment(String patientIEN, String clinicIEN, Date appointmentTimestamp, String timezone, String appointmentType, String appointmentLength, String schedulingRequestType) throws Exception
	{
		if(isNullOrBlank(patientIEN))
		{
			throw new MDWSSchedulingException("You must supply the patient's IEN.");
		}
		if(isNullOrBlank(clinicIEN))
		{
			throw new MDWSSchedulingException("You must supply the clinic's IEN.");
		}
		if(null == appointmentTimestamp)
		{
			throw new MDWSSchedulingException("You must supply DATE of the appointment.");
		}
		if(isNullOrBlank(timezone))
		{
			throw new MDWSSchedulingException("You must supply the timezone of the appointment.");
		}
		if(isNullOrBlank(appointmentType))
		{
			throw new MDWSSchedulingException("You must supply the TYPE of the appointment.");
		}
		if(isNullOrBlank(appointmentLength))
		{
			throw new MDWSSchedulingException("You must supply the LENGTH of the appointment.");
		}
		if(isNullOrBlank(schedulingRequestType))
		{
			throw new MDWSSchedulingException("You must supply REQUEST TYPE of the appoinment");
		}
		connect();
		service.select(patientIEN);
		AppointmentTO appt = service.makeAppointment(clinicIEN, formatForVISTA(appointmentTimestamp, timezone), appointmentType, "0", appointmentLength, schedulingRequestType);
		disconnect();
		return appt;
	}
	
	
	/**
	 * Make patient appointment.
	 *
	 * @param clinicIEN the clinic ien
	 * @param appointmentTimestamp the appointment timestamp
	 * @param timezone the timezone
	 * @param appointmentType the appointment type
	 * @param appointmentLength the appointment length
	 * @param schedulingRequestType the scheduling request type
	 * @return the appointment to
	 * @throws Exception the exception
	 */
	public AppointmentTO makePatientAppointment(String clinicIEN, Date appointmentTimestamp, String timezone, String appointmentType, String appointmentLength, String schedulingRequestType) throws Exception
	{
		if(isNullOrBlank(clinicIEN))
		{
			throw new MDWSSchedulingException("You must supply the clinic's IEN.");
		}
		if(null == appointmentTimestamp)
		{
			throw new MDWSSchedulingException("You must supply DATE of the appointment.");
		}
		if(isNullOrBlank(timezone))
		{
			throw new MDWSSchedulingException("You must supply the timezone of the appointment.");
		}
		if(isNullOrBlank(appointmentType))
		{
			throw new MDWSSchedulingException("You must supply the TYPE of the appointment.");
		}
		if(isNullOrBlank(appointmentLength))
		{
			throw new MDWSSchedulingException("You must supply the LENGTH of the appointment.");
		}
		if(isNullOrBlank(schedulingRequestType))
		{
			throw new MDWSSchedulingException("You must supply REQUEST TYPE of the appoinment");
		}
		AppointmentTO appt = service.makeAppointment(clinicIEN, formatForVISTA(appointmentTimestamp, timezone), appointmentType, "0", appointmentLength, schedulingRequestType);
		disconnect();
		return appt;
	}
	
	
	/**
	 * Orchestrated cancel patient appointment.
	 *
	 * @param patientIEN the patient ien
	 * @param clinicIEN the clinic ien
	 * @param appointmentTimestamp the appointment timestamp
	 * @param timezone the timezone
	 * @param appointmentStatus the appointment status
	 * @param cancellationReason the cancellation reason
	 * @param remarks the remarks
	 * @return the appointment to
	 * @throws Exception the exception
	 */
	public AppointmentTO orchestratedCancelPatientAppointment(String patientIEN, String clinicIEN, Date appointmentTimestamp, String timezone, String appointmentStatus, String cancellationReason, String remarks) throws Exception
	{
		if(isNullOrBlank(patientIEN))
		{
			throw new MDWSSchedulingException("You must supply the patient's IEN for the appointment you wish to cancel.");
		}
		if(isNullOrBlank(clinicIEN))
		{
			throw new MDWSSchedulingException("You must supply the clinic's IEN for the appointment you wish to cancel");
		}
		if(null == appointmentTimestamp)
		{
			throw new MDWSSchedulingException("You must supply DATE for the appointment you wish to cancel.");
		}
		if(isNullOrBlank(timezone))
		{
			throw new MDWSSchedulingException("You must supply the timezone of the appointment.");
		}
		if(null == appointmentStatus)
		{
			throw new MDWSSchedulingException("You must supply STATUS for the appointment you wish to cancel.");
		}
		if(null == cancellationReason)
		{
			throw new MDWSSchedulingException("You must supply the CANCELLATION REASON.");
		}
		if(null == remarks)
		{
			remarks = "";
		}
		connect();
		service.select(patientIEN);
		AppointmentTO appt = service.cancelAppointment(clinicIEN, formatForVISTA(appointmentTimestamp, timezone), appointmentStatus, cancellationReason, remarks);
		disconnect();
		return appt;
	}
	
	
	/**
	 * Cancel patient appointment.
	 *
	 * @param clinicIEN the clinic ien
	 * @param appointmentTimestamp the appointment timestamp
	 * @param timezone the timezone
	 * @param appointmentStatus the appointment status
	 * @param cancellationReason the cancellation reason
	 * @param remarks the remarks
	 * @return the appointment to
	 * @throws Exception the exception
	 */
	public AppointmentTO cancelPatientAppointment(String clinicIEN, Date appointmentTimestamp, String timezone, String appointmentStatus, String cancellationReason, String remarks) throws Exception
	{
		if(isNullOrBlank(clinicIEN))
		{
			throw new MDWSSchedulingException("You must supply the clinic's IEN for the appointment you wish to cancel");
		}
		if(null == appointmentTimestamp)
		{
			throw new MDWSSchedulingException("You must supply DATE for the appointment you wish to cancel.");
		}
		if(isNullOrBlank(timezone))
		{
			throw new MDWSSchedulingException("You must supply the timezone of the appointment.");
		}
		if(null == appointmentStatus)
		{
			throw new MDWSSchedulingException("You must supply STATUS for the appointment you wish to cancel.");
		}
		if(null == cancellationReason)
		{
			throw new MDWSSchedulingException("You must supply the CANCELLATION REASON.");
		}
		if(null == remarks)
		{
			remarks = "";
		}
		connect();
		AppointmentTO appt = service.cancelAppointment(clinicIEN, formatForVISTA(appointmentTimestamp, timezone), appointmentStatus, cancellationReason, remarks);
		disconnect();
		return appt;
	}

	
	/**
	 * Gets the appointment types.
	 *
	 * @return List of the appointment types
	 * @throws Exception the exception
	 */
	public List<AppointmentTypeTO> getAppointmentTypes() throws Exception
	{
		connect();
		List<AppointmentTypeTO> types = service.getAppointmentTypes("").getAppointmentTypes().getAppointmentTypeTO();
		disconnect();
		return types;
	}
	
	/**
	 * Gets the clinic scheduling details.
	 *
	 * @param clinicIEN the clinic ien
	 * @return the clinic scheduling details
	 * @throws Exception the exception
	 */
	public HospitalLocationTO getClinicSchedulingDetails(String clinicIEN) throws Exception
	{
		if(isNullOrBlank(clinicIEN))
		{
			throw new MDWSSchedulingException("You must supply the clinic IEN");
		}
		connect();
		HospitalLocationTO clinic = service.getClinicSchedulingDetails(clinicIEN);
		disconnect();
		return clinic;
	}
	
	/**
	 * Select patient.
	 *
	 * @param patientIEN the patient ien
	 * @return the patient object
	 * @throws Exception the exception
	 */
	public PatientTO selectPatient(String patientIEN) throws Exception
	{
		if(isNullOrBlank(patientIEN))
		{
			throw new MDWSSchedulingException("You must supply the patient's IEN");
		}
		connect();
		PatientTO patient = service.select(patientIEN);
		disconnect();
		return patient;
	}
	
	
	
	/**
	 * Gets the cancellation reasons.
	 *
	 * @return the cancellation reasons in tagged format
	 * @throws Exception the exception
	 */
	public List<TaggedText> getCancellationReasons() throws Exception
	{
		connect();
		List<TaggedText> reasons = service.getCancellationReasons("").getResults().getTaggedText();
		disconnect();
		return reasons;
	}
	
	
	/**
	 * Connect to a site.  You must specify the siteId,
	 * username, and password in the constructor of the
	 * SchedulingManager.
	 *
	 * @throws Exception the exception
	 */
	private void connect() throws Exception
	{
		if(isNullOrBlank(schedulingSvcWsdl))
		{
			throw new MDWSSchedulingException("The scheduling service's wsdl url must be provided");
		}
		service = new SchedulingSvc(new URL(schedulingSvcWsdl)).getSchedulingSvcSoap();
		((BindingProvider) service).getRequestContext().put(BindingProvider.SESSION_MAINTAIN_PROPERTY, true);
		if(isNullOrBlank(siteId) || isNullOrBlank(userName) || isNullOrBlank(password) )
		{
			throw new MDWSSchedulingException("Site ID, User name and password must be supplied within SchedulingManager constructor");
		}
		service.connect(siteId);
		if(isNullOrBlank(context))
		{
			service.login(userName, password, DEFAULT_CONTEXT);
		}
		else
		{
			service.login(userName, password, context);
		}
		
		
	}
	
	/**
	 * Disconnect.
	 *
	 * @throws Exception the exception
	 */
	private void disconnect() throws Exception
	{
		service.disconnect();
	}


	/**
	 * Gets the clinics.
	 *
	 * @return the clinics
	 * @throws Exception the exception
	 */
	public List<HospitalLocationTO> getClinics() throws Exception
	{
		connect();
		List<HospitalLocationTO> clinics = service.getClinics("").getLocations().getHospitalLocationTO();
		disconnect();
		return clinics;
	}
	
	//HELPER METHODS
	/**
	 * Checks if a string is null or blank.
	 *
	 * @param param the param
	 * @return true, if checks if is null or blank
	 */
	public static boolean isNullOrBlank(String param) { 
	    return param == null || param.trim().length() == 0;
	}
	
	
	/**
	 * Format for vista.
	 *
	 * @param theDate the the date
	 * @param timezone the timezone
	 * @return the string
	 */
	private String formatForVISTA(Date theDate, String timezone)
	{
		VISTA_FORMAT.setTimeZone(TimeZone.getTimeZone(timezone));		
		return VISTA_FORMAT.format(theDate.getTime());
	}
}
