package gov.va.med.scheduling.util
{
	public class TimeZoneUtil
	{
		/**
		 * List of timezone abbreviations and matching GMT times.
		 * Modified form original code at:
		 * http://blog.flexexamples.com/2009/07/27/parsing-dates-with-timezones-in-flex/
		 * */
		private static var timeZones:Array = 
		[
			/* Hawaii-Aleutian Standard/Daylight Time */
			{abbr:"HAST", zone:"GMT-1000", offset:-10},
			{abbr:"HADT", zone:"GMT-0900", offset:-9},
			/* Alaska Standard/Daylight Time */
			{abbr:"AKST", zone:"GMT-0900", offset:-9},
			{abbr:"ASDT", zone:"GMT-0800", offset:-8},
			/* Pacific Standard/Daylight Time */
			{abbr:"PST", zone:"GMT-0800", offset:-8},
			{abbr:"PDT", zone:"GMT-0700", offset:-7},
			/* Mountain Standard/Daylight Time */
			{abbr:"MST", zone:"GMT-0700", offset:-7},
			{abbr:"MDT", zone:"GMT-0600", offset:-6},
			/* Central Standard/Daylight Time */
			{abbr:"CST", zone:"GMT-0600", offset:-6},
			{abbr:"CDT", zone:"GMT-0500", offset:-5},
			/* Eastern Standard/Daylight Time */
			{abbr:"EST", zone:"GMT-0500", offset:-5},
			{abbr:"EDT", zone:"GMT-0400", offset:-4},
			/* Atlantic Standard/Daylight Time */
			{abbr:"AST", zone:"GMT-0400", offset:-4},
			{abbr:"ADT", zone:"GMT-0300", offset:-3},
			/* Newfoundland Standard/Daylight Time */
			{abbr:"NST", zone:"GMT-0330", offset:-3.5},
			{abbr:"NDT", zone:"GMT-0230", offset:-2.5},
			/* London Standard/Daylight Time */
			{abbr:"BST", zone:"GMT+0100", offset:1},
			{abbr:"GMT", zone:"GMT+0000", offset:0}
		];
		
		/**
		 * Return local system timzezone abbreviation.
		 * */
		public static function getTimeZone():String
		{
			var nowDate:Date = new Date();
			var DST:Boolean = isObservingDTS();
			var GMT:String = buildTimeZoneDesignation(nowDate, DST);
			
			return parseTimeZoneFromGMT(GMT);
		}
		
		public static function getTimeZoneForDate( date:Date ):String
		{
			var DST:Boolean = isObservingDTS();
			var GMT:String = buildTimeZoneDesignation(date, DST);
			
			return parseTimeZoneFromGMT(GMT);
		}
		
		public static function getTimeZoneOffsetForGMT( gmt:String ):Number
		{
			for each (var obj:Object in timeZones) 
			{
				if( obj.zone == gmt )
				{
					return obj.offset;
				}
			}
			
			return 0;
		}
		
		public static function getAbbreviationFromTimeZoneOffset( tzoffset:Number ):String
		{
			for each (var obj:Object in timeZones) 
			{
				if( obj.offset == tzoffset )
				{
					return obj.abbr;
				}
			}
			
			return "";
		}
		
		
		public static function getGMTForAbbreviation( abbr:String ):String
		{
			return parseGMTFromAbbreviation(abbr);
		}
		
		/**
		 * Determines if local computer is observing daylight savings time for US and London.
		 * */
		public static function isObservingDTS():Boolean
		{
			var winter:Date = new Date(2011, 01, 01); // after daylight savings time ends
			var summer:Date = new Date(2011, 07, 01); // during daylight savings time
			var now:Date = new Date();
			
			var winterOffset:Number = winter.getTimezoneOffset();
			var summerOffset:Number = summer.getTimezoneOffset();
			var nowOffset:Number = now.getTimezoneOffset();
			
			if((nowOffset == summerOffset) && (nowOffset != winterOffset)) 
			{
				return true;
			} 
			else 
			{
				return false;
			}	
		}
		
		/**
		 * Goes through the timze zone abbreviations looking for matching GMT time.
		 * */
		public static function parseTimeZoneFromGMT(gmt:String):String 
		{
			for each (var obj:Object in timeZones) 
			{
				if(obj.zone == gmt)
				{
					return obj.abbr;
				}
			}
			return gmt;
		}
		
		private static function parseGMTFromAbbreviation(abbr:String):String 
		{
			for each (var obj:Object in timeZones) 
			{
				if(obj.abbr == abbr)
				{
					return obj.zone;
				}
			}
			return abbr;
		}
		
		/**
		 * Method to build GMT from date and timezone offset and accounting for daylight savings.
		 * 
		 * Originally code befor modifications:
		 * http://flexoop.com/2008/12/flex-date-utils-date-and-time-format-part-i/
		 * */
		private static function buildTimeZoneDesignation( date:Date, dts:Boolean  ):String 
		{
			if ( !date ) {
				return "";
			}
			
			var timeZoneAsString:String = "GMT";
			var timeZoneOffset:Number;
			
			// timezoneoffset is the number that needs to be added to the local time to get to GMT, so
			// a positive number would actually be GMT -X hours
			if ( date.getTimezoneOffset() / 60 > 0 && date.getTimezoneOffset() / 60 < 10 ) {
				timeZoneOffset = (dts)? ( date.getTimezoneOffset() / 60 ):( date.getTimezoneOffset() / 60 - 1 );
				timeZoneAsString += "-0" + timeZoneOffset.toString();
			} else if ( date.getTimezoneOffset() < 0 && date.timezoneOffset / 60 > -10 ) {
				timeZoneOffset = (dts)? ( date.getTimezoneOffset() / 60 ):( date.getTimezoneOffset() / 60 + 1 );
				timeZoneAsString += "+0" + ( -1 * timeZoneOffset ).toString();
			} else {
				timeZoneAsString += "+00";
			}
			
			// add zeros to match standard format
			timeZoneAsString += "00";
			return timeZoneAsString;
		}
		
		public static function getTimeFromDateString(date:String, showTimeZone:Boolean = false):String {
			return showTimeZone ? date.substr(11,12) : date.substr(11,8)
		}
		
		public static function areTimesTheSame(date:Date, string:String):Boolean {
			if (!string){return true;}
			var dateHours24to12:Number = date.getHours() > 12 ? date.getHours() - 12 : date.getHours();
			return dateHours24to12 == Number(string.substr(11,2));
		}
	}
}