﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Shared.Model;
using System.Collections.ObjectModel;

namespace MedRed.Services.Utils
{
    class SchedulingHelper
    {

        //internal static List<_TimeSlot> GenerateTimeSlots(_ResourceOHSD OHSD, Collection<Appointment> appts, DateTime rangeStart, DateTime rangeEnd)
        //{
        //    List<_TimeSlot> timeSlots = ExpandIntoTimeSlots(OHSD, rangeStart, rangeEnd);

        //    foreach (var appt in appts)
        //    {
        //        // find approp timeslot and decrement availability
        //    }

        //    return timeSlots;

        //}

        /// <summary>
        /// Generate a structure of all of the available slots of a resource
        /// Current assuming that this will generate the capacity for the DAY of rangeStart through the DAY of rangeEnd
        /// </summary>
        /// <param name="resource"></param>
        /// <param name="rangeStart"></param>
        /// <param name="rangeEnd"></param>
        /// <returns></returns>
        internal static Dictionary<DateTime, List<SlotInstance>> GenerateCapacity(Resource resource, DateTime rangeStart, DateTime rangeEnd)
        {
            Dictionary<DateTime, List<SlotInstance>> capacity = new Dictionary<DateTime, List<SlotInstance>>();
            int days = (rangeEnd.Date.AddDays(1) - rangeStart.Date).Days;
            DateTime startingDate = rangeStart.Date;

            for (int i = 0; i < days; i++)
            {
                DateTime currDay = startingDate.AddDays(i);

                // get current scheduling policy in effect for this date and add slots to the array
                List<SlotInstance> todaysSlots = CreateCapacityForDay(resource, currDay);                

                // get any Holidays affecting today, and remove slots accordingly
                //// holidays are handled a level above this

                // get any blocks affecting today, and remove slots accordingly
                List<SlotInstance> toRemove = new List<SlotInstance>();
                var blocks = GetBlocksAffectingDate(currDay, resource.Blocks);
                foreach (var block in blocks)
                {
                    foreach (var timeslot in todaysSlots)
                    {
                        if ((timeslot.Time < block.PeriodEnd &&
                            timeslot.Time.AddMinutes(timeslot.Length) > block.PeriodStart) || 
                            // If this block has Recurrance, check to see if the recurring times cover this slot as well
                            // assumption being, it will only be returned from GetBlocksAffectingDate if it does apply to this date
                            (block.Recur != null && (timeslot.Time.TimeOfDay < block.PeriodEnd.TimeOfDay &&
                             timeslot.Time.AddMinutes(timeslot.Length).TimeOfDay > block.PeriodStart.TimeOfDay)))
                        {
                            if (!toRemove.Contains(timeslot))
                            {
                                toRemove.Add(timeslot);
                            }
                        }
                    }
                }

                // get any holds that affect this day, and remove slots accordingly
                var holds = GetHoldsAffectingDate(currDay, resource.Holds);
                foreach (var hold in holds)
                {
                    foreach (var timeslot in todaysSlots)
                    {
                        if (timeslot.Time < hold.PeriodEnd &&
                            timeslot.Time.AddMinutes(timeslot.Length) > hold.PeriodStart)
                        {
                            if (!toRemove.Contains(timeslot))
                            {
                                toRemove.Add(timeslot);
                            }
                        }
                    }
                }

                // remove timeslots affected
                foreach (var remove in toRemove)
                {
                    todaysSlots.Remove(remove);
                }

                // add the adjusted capacity map for this day
                capacity.Add(currDay, todaysSlots);
            }

            return capacity;
        }

        static List<SlotInstance> CreateCapacityForDay(Resource resource, DateTime currDay)
        {
            int dayOfWeek = GetDayOfWeek(currDay.DayOfWeek);         

            // get highest priority scheduling policy in effect for this day of week 

            var schedulingPolicies = resource.SchedulingPolicies.ToList();
            var effectiveSchedulingPolicy = schedulingPolicies.AsQueryable().Where(s => s.StartEffectDate <= currDay && (s.EndEffectDate == null || s.EndEffectDate > currDay))
                .Where(s => s.DayOfWeek == dayOfWeek)
                .OrderByDescending(s => s.Priority);
            if (effectiveSchedulingPolicy != null && effectiveSchedulingPolicy.Count() > 0)
            {
                var activePolicy = effectiveSchedulingPolicy.First();
                return CreateSlotInstances(currDay, activePolicy.Slots);
            }

            return new List<SlotInstance>();
        }

        private static int GetDayOfWeek(DayOfWeek dayOfWeek)
        {
            switch (dayOfWeek)
            {
                case DayOfWeek.Sunday:
                    return 0;
                case DayOfWeek.Monday:
                    return 1;
                case DayOfWeek.Tuesday:
                    return 2;
                case DayOfWeek.Wednesday:
                    return 3;
                case DayOfWeek.Thursday:
                    return 4;
                case DayOfWeek.Friday:
                    return 5;
                case DayOfWeek.Saturday:
                    return 6;
                default:
                    return 0;
            }
        }

        private static List<SlotInstance> CreateSlotInstances(DateTime currentDay, IList<Slot> slots)
        {
            List<SlotInstance> slotInstances = new List<SlotInstance>();

            foreach (var s in slots)
            {
                slotInstances.Add(new SlotInstance() { Time = currentDay.Date.Add(s.Time), Length = s.Length, Slot = s, Remaining = s.Capacity });
            }

            return slotInstances;
        }

        public static bool AddSlotsToPolicy(ref SchedulingPolicy policy, TimeSpan start, TimeSpan end, int defaultLength)
        {
            TimeSpan time = start;
            TimeSpan length = new TimeSpan(0, defaultLength, 0);

            while (time.Add(length) <= end)
            {
                policy.AddSlot(new Slot() { Time = time, Length = defaultLength });
                time = time.Add(length);
            }

            return true;
        }

        public static bool IsTimeAvailable(Dictionary<DateTime, List<SlotInstance>> capacity, DateTime time)
        {
            var day = capacity[time.Date];
            return (day.Exists(s => s.Time == time));
        }

        private static List<Block> GetBlocksAffectingDate(DateTime date, IList<Block> blocks)
        {
            List<Block> affectedBlocks = new List<Block>();
            foreach (var block in blocks)
            {
                if (block.PeriodEnd.Date >= date.Date &&
                    block.PeriodStart.Date <= date.Date)
                {
                    affectedBlocks.Add(block);
                }
                else if (block.Recur != null)
                {
                    // does the period of this recurring block cover today                   
                    if (block.Recur.RecurringStart.Date <= date.Date &&
                        block.Recur.RecurringEnd.Date >= date.Date)
                    {
                        DateTime dateToCheck = block.Recur.RecurringStart.Date;
                        int interval;
                        if (!Int32.TryParse(block.Recur.RecurringInterval, out interval))
                            continue;
                        if (interval > 0)
                        {
                            do
                            {
                                if (block.Recur.RecurringFrequency.Equals("Daily"))
                                {
                                    dateToCheck = dateToCheck.AddDays(interval);
                                }
                                else if (block.Recur.RecurringFrequency.Equals("Weekly"))
                                {
                                    dateToCheck = dateToCheck.AddDays(interval * 7);
                                }
                                else if (block.Recur.RecurringFrequency.Equals("Monthly"))
                                {
                                    dateToCheck = dateToCheck.AddMonths(interval);
                                }
                                else
                                {
                                    break;
                                }

                                if (dateToCheck == date.Date)
                                {
                                    affectedBlocks.Add(block);
                                    continue;
                                }

                            } while (dateToCheck <= block.Recur.RecurringEnd.Date);
                        }

                    }
                }
            }

            return affectedBlocks;
        }

        private static List<Hold> GetHoldsAffectingDate(DateTime date, IList<Hold> holds)
        {
            List<Hold> affectedHolds = new List<Hold>();
            foreach (var hold in holds)
            {
                if ((hold.PeriodEnd >= date.Date && hold.PeriodStart.Date <= date.Date) && 
                    (hold.AutoReleaseAt == null || hold.AutoReleaseAt.Value > DateTime.UtcNow))
                {
                    affectedHolds.Add(hold);
                }
            }

            return affectedHolds;
        }

    }
}
