﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Shared.Model;

namespace MedRed.Services.Utils
{
    public class CommunicationHelper
    {
        public static CommunicationQueueItem PrepareCommunication(Appointment appointment, CommunicationType type)
        {
            CommunicationQueueItem queueItem = null;

            var template = GetCommunicationTemplate(appointment, type);

            if (template != null)
            {
                queueItem = new CommunicationQueueItem();
                queueItem.Appointment = appointment;
                queueItem.Body = template.Message;
                queueItem.Medium = template.Medium;
                queueItem.Recipient = template.Recipient;
                queueItem.SendBy = appointment.Time.AddDays(-template.DateOffset);
                queueItem.Subject = template.Subject;
                queueItem.Template = template;
            }
            return queueItem;
        }

        public static CommunicationTemplate GetCommunicationTemplate(Appointment appointment, CommunicationType type)
        {
            CommunicationTemplate template = null;

            if (appointment.Section != null)
            {
                template = appointment.Section.CommunicationTemplates.FirstOrDefault<CommunicationTemplate>(ct => ct.Type == type);

                // if not found at section level, search at facility level
                if (template == null && appointment.Section.Facility != null)
                {
                    template = appointment.Section.Facility.CommunicationTemplates.FirstOrDefault<CommunicationTemplate>(ct => ct.Type == type);
                }
            }

            return template;
        }

        public static CommunicationQueueItem ProcessCommunicationTemplate(CommunicationQueueItem queueItem)
        {
            string patientName = string.Empty;
            string patientAddress = string.Empty;
            string patientId = string.Empty;
            string facilityName = string.Empty;
            string facilityAddress = string.Empty;
            string facilityPhone = string.Empty;
            string dateOfAppt = string.Empty;
            string timeOfAppt = string.Empty;
            string clinicianName = string.Empty;
            string clinicName = string.Empty;

            string today = DateTime.Now.ToShortDateString();

            if (queueItem.Appointment != null)
            {
                patientName = queueItem.Appointment.Patient.Person.ToString();
                patientAddress = AddressToString(queueItem.Appointment.Patient.Person.Address);
                patientId = queueItem.Appointment.Patient.Id.ToString();
                facilityName = queueItem.Appointment.Section.Facility.Name;
                facilityAddress = AddressToString(queueItem.Appointment.Section.Facility.Address);
                facilityPhone = queueItem.Appointment.Section.Facility.Phone;
                dateOfAppt = queueItem.Appointment.Time.ToShortDateString();
                timeOfAppt = queueItem.Appointment.Time.ToShortTimeString();

                var providerResource = queueItem.Appointment.Resources.FirstOrDefault(r => r.Type == ResourceType.Provider);
                if (providerResource != null)
                {
                    clinicianName = providerResource.Name;
                }
                clinicName = queueItem.Appointment.Section.Name;
            }
            else if (queueItem.AppointmentRequest != null)
            {
                patientName = queueItem.AppointmentRequest.Patient.Person.ToString();
                patientAddress = AddressToString(queueItem.AppointmentRequest.Patient.Person.Address);
                patientId = queueItem.AppointmentRequest.Patient.Id.ToString();

                if (queueItem.AppointmentRequest.DesiredDate.HasValue)
                {
                    dateOfAppt = queueItem.AppointmentRequest.DesiredDate.Value.ToShortDateString();
                    timeOfAppt = queueItem.AppointmentRequest.DesiredDate.Value.ToShortTimeString();
                }
                if (queueItem.AppointmentRequest.RequestedResource != null)
                {
                    clinicianName = queueItem.AppointmentRequest.RequestedResource.Name;
                }

                Facility facility = null;
                if (queueItem.AppointmentRequest.RequestedSection != null)
                {
                    clinicName = queueItem.AppointmentRequest.RequestedSection.Name;
                    facility = queueItem.AppointmentRequest.RequestedSection.Facility;
                }
                else if (queueItem.AppointmentRequest.RequestedResource != null)
                {
                    clinicName = queueItem.AppointmentRequest.RequestedResource.Section.Name;
                    facility = queueItem.AppointmentRequest.RequestedResource.Section.Facility;
                }

                facilityName = facility.Name;
                facilityAddress = AddressToString(facility.Address);
                facilityPhone = facility.Phone;
            }

            queueItem.Body = queueItem.Body.Replace("[PATIENT NAME]", patientName);
            queueItem.Body = queueItem.Body.Replace("[PATIENT ADDRESS]", patientAddress);
            queueItem.Body = queueItem.Body.Replace("[PATIENT ID NO]", patientId);
            queueItem.Body = queueItem.Body.Replace("[FACILITY NAME]", facilityName);
            queueItem.Body = queueItem.Body.Replace("[FACILITY ADDRESS]", facilityAddress);
            queueItem.Body = queueItem.Body.Replace("[FACILITY PHONE]", facilityPhone);
            queueItem.Body = queueItem.Body.Replace("[TODAY]", today);
            queueItem.Body = queueItem.Body.Replace("[DATE OF APPOINTMENT]", dateOfAppt);
            queueItem.Body = queueItem.Body.Replace("[TIME OF APPOINTMENT]", timeOfAppt);
            queueItem.Body = queueItem.Body.Replace("[CLINIC NAME]", clinicName);
            queueItem.Body = queueItem.Body.Replace("[CLINICIAN NAME]", clinicianName);
            queueItem.Body = queueItem.Body.Replace("[DATE]", DateTime.Now.ToString("d MMM yyyy"));

            return queueItem;
        }

        public static string AddressToString(Address address)
        {
            if (address == null)
                return string.Empty;

            StringBuilder sb = new StringBuilder();
            if (!String.IsNullOrEmpty(address.Street1))
            {
                sb.AppendLine(address.Street1);
            }
            if (!String.IsNullOrEmpty(address.Street2))
            {
                sb.AppendLine(address.Street2);
            }
            if (!String.IsNullOrEmpty(address.Street3))
            {
                sb.AppendLine(address.Street3);
            }
            sb.AppendLine(string.Format("{0}, {1} {2}", address.City, address.State, address.Zip));

            return sb.ToString();
        }

    }
}
