﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using NHibernate.Linq;
using System.Text;
using MedRed.Services.Interfaces;
using Shared.Model.Config.MDWS;
using Shared.Model;
using MedRed.Services.Utils;


namespace MedRed.Services.ServiceImpl
{
    class SectionService : BaseService, ISectionService
    {
        public SectionService(Factory factory) :
            base(factory)
        {
        }

        public Collection<Section> GetAll(int facilityId)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                var sections = from s in session.Query<Section>()
                               where s.Facility.Id == facilityId
                               select s;
                transaction.Commit();
                return new Collection<Section>(sections.ToList());
            }
        }

        public Section Get(int id)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                var section = session.Get<Section>(id);

                transaction.Commit();
                return section;
            }
        }

        public Section Add(Section section)
        {
            if (section.Facility == null)
                throw new ArgumentException("Clinic must be associated with a facility");

            if (String.IsNullOrEmpty(section.DSSPrimaryStopCode))
                throw new ArgumentException("Clinic must have a DSS Stop Code");

            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                session.SaveOrUpdate(section);

                transaction.Commit();

                int siteId = 0;
                if (section.Facility != null && section.Facility.Site != null)
                    siteId = section.Facility.Site.Id;
                
                Auditing.Logger.Log(
                Auditing.LogType.Information,
                siteId,
                Auditing.ObjectType.Section,
                Auditing.SectionFunction.SectionCreated,
                section.Id,
                section.Name,
                Auditing.Status.Success,
                false,
                "");
                return section;
            }
        }

        public Section Update(Section section)
        {
            var orig = Get(section.Id);
            bool activeChanged = (orig.Active != section.Active);

            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                session.SaveOrUpdate(section);

                transaction.Commit();            
            }
            var resources = ParentFactory.GetResourceService().GetAll(section.Id);

            if (section.Active == false)
            {
                // we must cancel all affected appointments for all resources under this section
                string reason = section.Name + " was deactivated";

                // any existing appointments for any of the resources need to be cancelled
                foreach (var resource in resources)
                {
                    // get all scheduled appointments
                    var appointments = ParentFactory.GetAppointmentService().GetForResource(resource.Id, null, null, AppointmentStatus.Scheduled);

                    foreach(var appt in appointments)
                    {
                        if (appt.Status == AppointmentStatus.Scheduled)
                        {
                            AppointmentHelper.PerformAutomaticAppointmentCancelActions(ParentFactory, appt, reason);
                        }
                    }                
                }
            }

            // if any appointment types were deleted, delete all affected appointments
            var savedSection = Get(section.Id);
            var deletedAppointmentTypes = new List<AppointmentType>();
            foreach (var existingApptType in orig.AppointmentTypes)
            {
                if (!section.AppointmentTypes.Contains(existingApptType))
                {
                    deletedAppointmentTypes.Add(existingApptType);
                }
            }
            foreach (var appTypeToDelete in deletedAppointmentTypes)
            {
                string reason = "Appointment Type " + appTypeToDelete.Name + " was deleted";
                var appointments = ParentFactory.GetAppointmentService().GetAllForAppointmentType(appTypeToDelete, null, null, AppointmentStatus.Scheduled);

                foreach (var appt in appointments)
                {
                    if (appt.Status == AppointmentStatus.Scheduled)
                    {
                        AppointmentHelper.PerformAutomaticAppointmentCancelActions(ParentFactory, appt, reason);
                    }
                }
            }

            var mdws = GetMDWSDAO(section.Facility.Site.VistaSiteId);

            // was DSS Primary Code changed from before?
            if (!String.Equals(section.DSSPrimaryStopCode, orig.DSSPrimaryStopCode))
            {
                foreach (var resource in resources)
                {
                    if (resource.Type == ResourceType.Provider)
                    {
                        try
                        {
                            mdws.UpdateClinicDSSCode(resource);
                        }
                        catch (Exception e)
                        {
                            // There is a provider resource that doesn't have a corresponding Vista Clinic id
                            // log the error
                        }
                    }
                }
            }

            // update the vista 'active' state of all provider resources under this section
            foreach (var resource in resources)
            {
                if (resource.Type == ResourceType.Provider)
                {
                    try
                    {
                        mdws.SetClinicActive(resource, section.Active, DateTime.Now);
                    }
                    catch(Exception e)
                    {
                        // There is a provider resource that doesn't have a corresponding Vista Clinic id
                        // log the error
                    }
                }
            }

            return section;
        }

        public bool Delete(int id)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                var section = Get(id);

                if (section == null)
                {
                    return false;
                }
                    
                session.Delete(section);
                
                transaction.Commit();
                return true;
            }
        }


        public CommunicationTemplate AddOrUpdateCommunicationTemplate(int sectionId, CommunicationTemplate communicationTemplate)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                var section = Get(sectionId);

                var existingTemplate = section.CommunicationTemplates.Where(c => c.Id == communicationTemplate.Id);
                if (existingTemplate != null && existingTemplate.Count() > 0)
                {
                    var copyOfList = existingTemplate.ToList();
                    copyOfList.All(x => section.CommunicationTemplates.Remove(x));
                }

                section.CommunicationTemplates.Add(communicationTemplate);

                session.SaveOrUpdate(section);
                transaction.Commit();

                int siteId = 0;
                if (section.Facility != null && section.Facility.Site != null)
                    siteId = section.Facility.Site.Id;

                Auditing.Logger.Log(
                Auditing.LogType.Information,
                siteId,
                Auditing.ObjectType.Section,
                Auditing.SectionFunction.CommuncationTemplateCreated,
                communicationTemplate.Id,
                communicationTemplate.Name,
                Auditing.Status.Success,
                false,
                "");
                return communicationTemplate;
            }
        }

        public bool DeleteCommunicationTemplate(int sectionId, CommunicationTemplate communicationTemplate)
        {
            bool somethingRemoved = false;

            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                var section = Get(sectionId);

                var existingTemplate = section.CommunicationTemplates.Where(c => c.Id == communicationTemplate.Id);
                if (existingTemplate != null && existingTemplate.Count() > 0)
                {
                    var copyOfList = existingTemplate.ToList();
                    copyOfList.All(x => section.CommunicationTemplates.Remove(x));
                    somethingRemoved = true;
                }
                session.SaveOrUpdate(section);
                transaction.Commit();
                return somethingRemoved;
            }
        }

        public AppointmentType AddOrUpdateAppointmentType(int sectionId, AppointmentType appointmentType)
        {
            if (appointmentType.AppointmentTypeCategory == null)
                throw new ArgumentException("Appointment Types must have an appointment type category");

            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                var section = Get(sectionId);

                var existing = section.AppointmentTypes.Where(c => c.Id == appointmentType.Id);
                if (existing != null && existing.Count() > 0)
                {
                    var copyOfList = existing.ToList();
                    copyOfList.All(x => section.AppointmentTypes.Remove(x));
                }
//                appointmentType.Section = section;
                section.AppointmentTypes.Add(appointmentType);
                session.SaveOrUpdate(section);
                transaction.Commit();

                //TODO AUDITING

                return appointmentType;
            }
        }

        public bool DeleteAppointmentType(int sectionId, AppointmentType appointmentType)
        {
            bool somethingRemoved = false;

            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                var section = Get(sectionId);

                var existing = section.AppointmentTypes.Where(c => c.Id == appointmentType.Id);
                if (existing != null && existing.Count() > 0)
                {
                    var copyOfList = existing.ToList();
                    copyOfList.All(x => section.AppointmentTypes.Remove(x));
                    somethingRemoved = true;
                }
                session.SaveOrUpdate(section);
                transaction.Commit();
                return somethingRemoved;
            }
        }
    }
}
