﻿using System;
using System.Collections.ObjectModel;
using MedRed.Services.Interfaces;
using System.Linq;
using NHibernate.Linq;
using Shared.Model;
using Shared.Model.Config.MDWS;
using MedRed.Services.Utils;
using System.Collections.Generic;

namespace MedRed.Services.ServiceImpl
{
    internal class ResourceService : BaseService, IResourceService
    {
        public ResourceService(Factory factory) :
            base(factory)
        {
        }

        public Collection<Resource> GetAll(int sectionId)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {

                var resources = from s in session.Query<Resource>()
                                where s.Section.Id == sectionId
                                select s;

                transaction.Commit();
                return new Collection<Resource>(resources.ToList());
            }
        }

        public Collection<Resource> GetAllOfType(int sectionId, ResourceType type)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {

                var resources = from s in session.Query<Resource>()
                                where s.Section.Id == sectionId &&
                                s.Type == type
                                select s;

                transaction.Commit();
                return new Collection<Resource>(resources.ToList());
            }
        }

        public Resource Get(int id)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {

                var resource = session.Get<Resource>(id);

                transaction.Commit();
                return resource;

            }
        }

        public Resource Add(Resource resource)
        {
            if (resource.Section == null)
                throw new ArgumentException("Resource must belong to a clinic");

            if (String.IsNullOrEmpty(resource.Section.DSSPrimaryStopCode))
                throw new ArgumentException("Resources is being added to a clinic that doesn't have a valid DSS Stop Code");

            if (resource.Type == ResourceType.Provider)
            {
                if (resource.FullfillingResourceId == 0)
                {
                    throw new ArgumentException("Provider resources must be associated with an actual provider");
                }
                Provider provider = ParentFactory.GetProviderService().Get(resource.FullfillingResourceId);
                if (provider == null)
                {
                    throw new ArgumentException("Resource is not associated with a valid provider");
                }
            }

            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                session.Save(resource);

                transaction.Commit();
//                return resource;
            }
            
            //TODO CALDAV
            var caldav = CalendarAccess.CalendarFactory.GetCalendarAccess();

            if (caldav.IsCaldavEnabled())
            {
                caldav.GetCalendar(resource);
            }

            // set up Vista Clinic if this is a provider
            if (resource.Type == ResourceType.Provider)
            {
                Provider provider = ParentFactory.GetProviderService().Get(resource.FullfillingResourceId);

                string vistaSite = resource.Section.Facility.Site.VistaSiteId;
                var mdws = GetMDWSDAO(vistaSite);

                var vistaClinicId = mdws.AddClinic(resource, provider);

                resource.VistaClinicIEN = vistaClinicId;

                Update(resource);
            }

            foreach (var block in resource.Blocks)
            {
                //TODO AUDITING
            }

            foreach (var hold in resource.Holds)
            {
                //TODO AUDITING
            }
            return resource;
        }

        public Resource Update(Resource resource)
        {
            var originalResource = Get(resource.Id);

            List<Block> addedBlocks = new List<Block>();
            foreach (var block in resource.Blocks)
            {
                if (!originalResource.Blocks.Contains(block))
                    addedBlocks.Add(block);
            }

            List<Hold> addedHolds = new List<Hold>();
            foreach (var hold in resource.Holds)
            {
                if (!originalResource.Holds.Contains(hold))
                    addedHolds.Add(hold);
            }

            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                session.SaveOrUpdate(resource);

                transaction.Commit();
            }

            // make sure any appointments affected by blocks are cancelled
            foreach (var block in resource.Blocks)
            {
                string reason = resource.Name + " blocked due to: " + block.Reason;
                var appointments = ParentFactory.GetAppointmentService().GetForResource(resource.Id, block.PeriodStart, block.PeriodEnd, AppointmentStatus.Scheduled);

                foreach (var appt in appointments)
                {
                    AppointmentHelper.PerformAutomaticAppointmentCancelActions(ParentFactory, appt, reason);
                }
            }

            int siteId = 0;
            if (resource.Section != null && resource.Section.Facility != null && resource.Section.Facility.Site != null)
                siteId = resource.Section.Facility.Site.Id;

            foreach (var block in addedBlocks)
            {
                Auditing.Logger.Log(
                Auditing.LogType.Information,
                siteId,
                Auditing.ObjectType.Resource,
                Auditing.ResourceFunction.BlockCreated,
                block.Id,
                "",
                Auditing.Status.Success,
                false,
                block.Reason);

            }

            foreach (var hold in addedHolds)
            {
                //TODO AUDITING
            }
            return resource;
        }

        public bool Delete(int id)
        {
            var resource = Get(id);

            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {

                if (resource != null)
                {
                    session.Delete(resource);
                }

                transaction.Commit();
                return true;
            }
        }

        public Dictionary<DateTime, List<SlotInstance>> GetCapacity(int resourceId, DateTime rangeStart, DateTime rangeEnd, AppointmentType appointmentType)
        {
            DateTime start = rangeStart.Date;
            DateTime end = rangeEnd.Date.AddDays(1).AddMinutes(-1);
            var resource = Get(resourceId);
            if (resource == null)
                throw new ArgumentException("Invalid Resource Id");
            if (end <= start)
                throw new ArgumentException("Start Date must be before End Date");

            var capacity = SchedulingHelper.GenerateCapacity(resource, start, end); 

            // handle holidays here
            var holidays = ParentFactory.GetNationalSystemService().GetHolidaysBetween(start, end);
            foreach (var holiday in holidays)
            {
                // if capacity contains holiday, clear out all slots that day
                if (capacity.ContainsKey(holiday.Date))
                {
                    capacity[holiday.Date] = new List<SlotInstance>();
                }
            }

            foreach (var day in capacity.Keys)
            {
                List<SlotInstance> toRemove = new List<SlotInstance>();
                foreach (var slotInstance in capacity[day])
                {
                    if (!((appointmentType == null || slotInstance.Slot.AppointmentType == null || slotInstance.Slot.AppointmentType.Id == appointmentType.Id)))
                    {
                        toRemove.Add(slotInstance);
                        continue;
                    }
                }

                foreach (var slotInstance in toRemove)
                {
                    capacity[day].Remove(slotInstance);
                }
            }

            return capacity;    
        }

        public Dictionary<DateTime, List<SlotInstance>> GetAvailability(int resourceId, DateTime rangeStart, DateTime rangeEnd, AppointmentType appointmentType)
        {
            DateTime start = rangeStart.Date;
            DateTime end = rangeEnd.Date.AddDays(1).AddMinutes(-1);
            var resource = Get(resourceId);

            if (resource == null)
                throw new ArgumentException("Invalid Resource Id");
            if (end <= start)
                throw new ArgumentException("Start Date must be before End Date");

            var availability = GetCapacity(resourceId, start, end, appointmentType);

            // adjust capacity of the rest of the slots
            var appts = ParentFactory.GetAppointmentService().GetForResource(resourceId, start, end);

            foreach (var appt in appts)
            {
                if (appt.Status == AppointmentStatus.Cancelled)
                    continue;
                DateTime date = appt.Time.Date;
                var day = availability[date];
                var slotInstance = day.FirstOrDefault(x => x.Time == appt.Time);
                if (slotInstance != null)
                    slotInstance.Remaining--;

                // how to handle appointments that don't match a slot?

                // how to handle appointments that overbook if overbook not allowed?
            }

            return availability;
        }

        public Dictionary<DateTime, List<SlotInstance>> GetGroupAppointmentSlots(int resourceId, DateTime rangeStart, DateTime rangeEnd, AppointmentType appointmentType)
        {
            DateTime start = rangeStart.Date;
            DateTime end = rangeEnd.Date.AddDays(1).AddMinutes(-1);
            var resource = Get(resourceId);

            if (resource == null)
                throw new ArgumentException("Invalid Resource Id");
            if (end <= start)
                throw new ArgumentException("Start Date must be before End Date");

            var availability = GetCapacity(resourceId, start, end, appointmentType);

            //// adjust capacity of the rest of the slots
            //var appts = ParentFactory.GetAppointmentService().GetForResource(resourceId, start, end);

            //foreach (var appt in appts)
            //{
            //    DateTime date = appt.Time.Date;
            //    var day = availability[date];
            //    var slotInstance = day.FirstOrDefault(x => x.Time == appt.Time);
            //    if (slotInstance != null)
            //        slotInstance.Remaining--;

            //    // how to handle appointments that don't match a slot?

            //    // how to handle appointments that overbook if overbook not allowed?
            //}

            foreach (var day in availability.Keys)
            {
                List<SlotInstance> toRemove = new List<SlotInstance>();
                foreach (var slotInstance in availability[day])
                {
                    if (!slotInstance.Slot.GroupAppointment)
                        toRemove.Add(slotInstance);
                }
                foreach (var slotInstance in toRemove)
                {
                    availability[day].Remove(slotInstance);
                }
            }

            return availability;
        }

        public Dictionary<DateTime, List<SlotInstance>> GetAvailability(Collection<int>resourceIds, DateTime rangeStart, DateTime rangeEnd, AppointmentType appointmentType)
        {
            DateTime start = rangeStart.Date;
            DateTime end = rangeEnd.Date.AddDays(1).AddMinutes(-1);

            if (resourceIds == null || resourceIds.Count == 0)
            {
                throw new ArgumentException("No resources provided to generate availability from");
            }
            if (end <= start)
                throw new ArgumentException("Start Date must be before End Date");

            int firstResourceId = resourceIds[0];
            var firstResource = Get(firstResourceId);
            if (firstResource == null)
                throw new ArgumentException("Invalid Resource Id");

            // start with availability of the first resource and subtract anything that isn't present in the others
            var availability = GetAvailability(firstResourceId, start, end, appointmentType);

            // remove any slots that don't agree with the appointmentType and are already filled
            foreach (var day in availability.Keys)
            {
                List<SlotInstance> toRemove = new List<SlotInstance>();
                foreach (var slotInstance in availability[day])
                {
                    if (!((appointmentType == null || slotInstance.Slot.AppointmentType == null || slotInstance.Slot.AppointmentType.Id == appointmentType.Id)))
                    {
                        toRemove.Add(slotInstance);
                        continue;
                    }

                    if (slotInstance.Remaining <= 0 && !firstResource.AllowOverbooking)
                    {
                        toRemove.Add(slotInstance);
                        continue;
                    }
                }
                foreach (var remove in toRemove)
                {
                    availability[day].Remove(remove);
                }
            }            

            // filter out results based on the other resource's availability
            for (int i = 1; i < resourceIds.Count; i++)
            {
                var nextResourceAvailability = GetAvailability(resourceIds[i], start, end, appointmentType);
                var nextResource = Get(resourceIds[i]);
                if (nextResource == null)
                    throw new ArgumentException("Invalid Resource Id");

                foreach (var day in availability.Keys)
                {
                    List<SlotInstance> toRemove = new List<SlotInstance>();

                    foreach (var slotInstance in availability[day])
                    {
                        if (!nextResourceAvailability.ContainsKey(day))
                        {
                            toRemove.Add(slotInstance);
                            continue;
                        }
                        var matchingSlot = nextResourceAvailability[day].FirstOrDefault(s => s.Time == slotInstance.Time);
                        if (matchingSlot == null || (matchingSlot.Remaining <= 0 && !nextResource.AllowOverbooking))
                        {
                            toRemove.Add(slotInstance);
                            continue;
                        }
                        if (!((appointmentType == null || slotInstance.Slot.AppointmentType == null || slotInstance.Slot.AppointmentType.Id == appointmentType.Id)))
                        {
                            toRemove.Add(slotInstance);
                            continue;
                        }

                        slotInstance.Remaining = Math.Min(slotInstance.Remaining, matchingSlot.Remaining);
                    }

                    foreach (var remove in toRemove)
                    {
                        availability[day].Remove(remove);
                    }
                }
            }

            return availability;
        }

        public Collection<Resource> FindAlternateAvailableProviders(Appointment appointment)
        {
            var allProviders = GetAllOfType(appointment.Section.Id, ResourceType.Provider);

            var availableProviders = new Collection<Resource>();

            var currProvider = appointment.Resources.FirstOrDefault(r => r.Type == ResourceType.Provider);
            if (currProvider != null)
            {
                var prov = allProviders.FirstOrDefault(r => r.Id == currProvider.Id);
                if (prov != null)
                {
                    allProviders.Remove(prov);
                }
            }

            var apptType = appointment.AppointmentType;

            foreach (var provider in allProviders)
            {
                var availability = GetAvailability(provider.Id, appointment.Time, appointment.Time, apptType);
                if (SchedulingHelper.IsTimeAvailable(availability, appointment.Time))
                {
                    availableProviders.Add(provider);
                }
            }

            return availableProviders;
        }

        public Collection<Resource> FindAlternateAvailableGroupProviders(Appointment appointment)
        {
            var allProviders = GetAllOfType(appointment.Section.Id, ResourceType.Provider);

            var availableProviders = new Collection<Resource>();

            var currProvider = appointment.Resources.FirstOrDefault(r => r.Type == ResourceType.Provider);
            if (currProvider != null)
            {
                var prov = allProviders.FirstOrDefault(r => r.Id == currProvider.Id);
                if (prov != null)
                {
                    allProviders.Remove(prov);
                }
            }

            var apptType = appointment.AppointmentType;

            foreach (var provider in allProviders)
            {
                var availability = GetGroupAppointmentSlots(provider.Id, appointment.Time, appointment.Time, apptType);
                if (SchedulingHelper.IsTimeAvailable(availability, appointment.Time))
                {
                    availableProviders.Add(provider);
                }
            }

            return availableProviders;
        }

    }
}