﻿using System;
using System.Collections.Generic;
using System.Linq;
using NHibernate.Linq;
using System.Net;
using Shared.Model;
using MedRed.Services.Interfaces;
using Shared.Model.Config.MDWS;
using System.Collections.ObjectModel;

namespace MedRed.Services.ServiceImpl
{
    class PatientService : BaseService, IPatientService
    {
        public PatientService(Factory factory) :
            base(factory)
        {
        }

        public Collection<Appointment> GetAppointments(string siteId, string patientDfn, DateTime rangeStart, DateTime rangeEnd)
        {
            var patient = Get(siteId, patientDfn);
            return ParentFactory.GetAppointmentService().GetForPatient(patient.Id, rangeStart, rangeEnd);
        }

        public Collection<Patient> Get()
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {

                var patient = from p in session.Query<Patient>()
                              select p;

                transaction.Commit();
                return new Collection<Patient>(patient.ToList());
            }
        }

        public Patient Get(int id)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                var patient = session.Get<Patient>(id);

                transaction.Commit();
                return patient;
            }
        }

        public Patient GetFullDetails(string vistaSiteId, int id)
        {
            var patient = Get(id);

            if (patient != null)
            {
                var mdws = GetMDWSDAO(vistaSiteId);
                mdws.GetAllPatientDetails(ref patient);
            }
            return patient;
        }

        public Patient Add(Patient patient)
        {
            if (patient.Id != 0)
            {
                throw new ArgumentException("Patient is already added");
            }

            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                session.Save(patient);
               
                transaction.Commit();
                
//                var calAccess = CalendarAccess.CalendarFactory.GetCalendarAccess();
                // access (and create) patient's calendar)
//                calAccess.GetCalendar(patient);
                return patient;
            }

            //TODO CALDAV
            var caldav = CalendarAccess.CalendarFactory.GetCalendarAccess();

            if (caldav.IsCaldavEnabled())
            {
                caldav.GetCalendar(patient);
            }
        }

        public Patient Update(Patient patient)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                session.SaveOrUpdate(patient);

                transaction.Commit();
                return patient;
            }
        }

        public bool Delete(int id)
        {
            var pat = Get(id);

            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                if (pat != null)
                {
                    session.Delete(pat);
                }
                transaction.Commit();
                return true;
            }
        }

        public Patient Get(PatientSearchResult patSearchResult)
        {
            return Get(patSearchResult.VistaSiteId, patSearchResult.VistaDFN);
        }

        public Patient Get(string vistaSiteId, string patientDfn)
        {
            //if (siteId != ParentFactory.ConnectionDetails.SiteID)
            //    throw new Exception("Attempting to load patient from a different site than currently accessing");
            Patient patient = null;

            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                var patients = from p in session.Query<Patient>()
//                               where p.PatientSites.Any(s => s.VistaSiteId == vistaSiteId && s.DFN == patientDfn)
                               where p.PatientSites[vistaSiteId] == patientDfn
                               select p;

                transaction.Commit();
                if (patients.Count() > 0)
                    patient = patients.First();
            }

            // if patient not found, try VistA
            if (patient == null)
            {
                var mdws = GetMDWSDAO(vistaSiteId);
                patient = mdws.loadPatient(patientDfn);
                mdws.GetAllPatientDetails(ref patient);
                //// MVI like stub
                // DoesPatientExistOnAnyOtherSite
                MVI_FindOtherPatientLocations(ref patient);
                //// END MVI stub

// pick between 0 and 2 special needs and patient preferences
                Random r = new Random();
                int preferenceCount = r.Next(0, 2);
                int specialNeedCount = r.Next(0, 2);

                var prefs = ParentFactory.GetNationalSystemService().GetPatientPreferenceTypes();
                var needs = ParentFactory.GetNationalSystemService().GetSpecialNeedTypes();

                if (prefs.Count > 0)
                {
                    for (int i = 0; i < preferenceCount; i++)
                    {
                        int index = r.Next(0, prefs.Count - 1);
                        var pref = prefs.ElementAt(index);
                        if (pref != null)
                            patient.AddPatientPreference(pref);
                    }
                }
                if (needs.Count > 0)
                {
                    for (int i = 0; i < specialNeedCount; i++)
                    {
                        int index = r.Next(0, needs.Count - 1);
                        var need = needs.ElementAt(index);
                        if (need != null)
                            patient.AddSpecialNeed(need);
                    }
                }
// end preferences

                // if we found in Vista, add them to scheduling system
                if (patient != null)
                {
                    Add(patient);
                }
            }
            else
            {
                var mdws = GetMDWSDAO(vistaSiteId);
                mdws.GetAllPatientDetails(ref patient);
            }


            return patient;
        }

        private bool MVI_FindOtherPatientLocations(ref Patient patient)
        {
            var allSites = ParentFactory.GetSiteService().Get();
            string ssn = patient.Person.SSN;

            if (String.IsNullOrEmpty(ssn))
                return false;

            foreach (var site in allSites)
            {
                if (ParentFactory.GetImportService().ValidateServiceAccount(site.VistaSiteId))
                {
                    string dfn = patient.GetDFNForVistASite(site.VistaSiteId);
                    if (string.IsNullOrEmpty(dfn))
                    {
                        // see if we can find the patient by ICN or SSN on this site
                        try
                        {
                            var mdws = GetMDWSDAO(site.VistaSiteId);
                            var results = mdws.searchForPatients(ssn);
                            if (results.Count > 0)
                            {
                                if (results.Count == 1)
                                {
                                    // we found them at this site, so add a PatientSite for them  
                                    patient.AddPatientSite(results[0].VistaSiteId, results[0].VistaDFN);
                                }
                                else
                                {
                                    throw new Exception("SSNs seem to not be unique in Vista - " + ssn);
                                }
                            }
                        }
                        catch (ArgumentException e)
                        {
                            //  log that access was attempted for a site that doesn't have a service account set up
                        }
                    }
                }
            }
            return true;
        }

        public Collection<PatientSearchResult> Search(string vistaSiteId, string searchParam)
        {
//            if (vistaSiteId != ParentFactory.ConnectionDetails.SiteID)
//                throw new Exception("Attempting to load patient from a different site than currently accessing");

            //// load our local patients matching criteria
            //var results = FindPatientsForSite(vistaSiteId, searchParam);

            // load Vista patients matching criteria
            var mdws = GetMDWSDAO(vistaSiteId);

            var vistaPatients = mdws.searchForPatients(searchParam);
            return new Collection<PatientSearchResult>(vistaPatients);
            
        }

        private List<Patient> FindPatientsForSite(string vistaSiteId, string searchParam)
        {
            var results = new List<Patient>();

            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                var patients = from p in session.Query<Patient>()
                               where p.PatientSites.ContainsKey(vistaSiteId) &&
                                (p.Person.LastName.Contains(searchParam) || 
                                p.Person.SSN.Contains(searchParam))
                              select p;

                transaction.Commit();
            }

            return results;
        }


        public Provider GetPrimaryCareProvider(int id)
        {
            var appts = ParentFactory.GetAppointmentService().GetForPatient(id, null, DateTime.UtcNow);

            if (appts.Count > 0)
            {
                var mostRecent = appts.Max(a => a.Time);
                var mostRecentAppt = appts.FirstOrDefault(a=>a.Time == mostRecent);
                if (mostRecent != null)
                {
                    var provResource = mostRecentAppt.Resources.FirstOrDefault(r => r.Type == ResourceType.Provider);
                    if (provResource != null)
                    {
                        var provider = ParentFactory.GetProviderService().Get(provResource.FullfillingResourceId);
                        return provider;
                    }
                }
            }

            return null;
        }
    }
}
