﻿using System;
using System.Collections.Generic;
using System.Linq;
using NHibernate.Linq;
using System.Net;
using Shared.Model;
using MedRed.Services.Interfaces;
using Shared.Model.Config.MDWS;
using System.Collections.ObjectModel;
using MedRed.Services.Utils;
using MedRed.CalendarAccess;

namespace MedRed.Services.ServiceImpl
{
    class AppointmentService : BaseService, IAppointmentService
    {
        public AppointmentService(Factory factory) :
            base(factory)
        {
        }

        public Appointment Get(int id)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                var appointment = session.Get<Appointment>(id);

                transaction.Commit();
                return appointment;
            }
        }

        public Appointment Update(Appointment appointment)
        {
            if (appointment.Id == 0)
            {
                throw new ArgumentException("Only existing appointments can be updated");
            }

            if (appointment.Patient == null)
            {
                throw new ArgumentException("An appointment must be associated with patient");
            }

            if (appointment.Resources.Count == 0 || !appointment.Resources.Any(r=>r.Type == ResourceType.Provider))
            {
                throw new ArgumentException("An appointment must be associated with at least one provider resource");
            }
            if (appointment.Section == null)
            {
                throw new ArgumentException("An appointment must be associated with a clinic");
            }

            if (appointment.Time == null || appointment.Time <= DateTime.MinValue)
            {
                throw new ArgumentException("An appointment must have a valid time");
            }
            var orig = Get(appointment.Id);

            // did provider switch?
            var oldProvider = orig.Resources.FirstOrDefault(r => r.Type == ResourceType.Provider);
            var newProvider = appointment.Resources.FirstOrDefault(r => r.Type == ResourceType.Provider);

            if (oldProvider != null && newProvider != null && oldProvider.Id != newProvider.Id)
            {
                //if (!VerifyResourceAvailability(appointment))
                //{
                //    throw new ArgumentException("One or more resources are unavailable at the appointment time");
                //}

                var caldav = CalendarAccess.CalendarFactory.GetCalendarAccess();
                var mdws = GetMDWSDAO(appointment.Section.Facility.Site.VistaSiteId);

                // cancel existing appt in VistA
                mdws.CancelAppointment(orig, "C", "13", "Switching providers");
                // cancel existing appt in Caldav

                if (caldav.IsCaldavEnabled())
                {
                    caldav.DeleteAppointment(appointment);
                }

                // create new appointment in VistA
                mdws.MakeAppointment(appointment);
                // create new appointment in Caldav
                if (caldav.IsCaldavEnabled())
                {
                    caldav.SaveAppointment(appointment);
                }

                var newActivity = new AppointmentActivity() { Time = DateTime.UtcNow, ActivityType = ActivityType.ProviderChange, 
                    ActivityReason = string.Format("Changed from {0} to {1}", oldProvider.Name, newProvider.Name) };
                appointment.Activity.Add(newActivity);
            }
            
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                session.SaveOrUpdate(appointment);

                transaction.Commit();

                int siteId = 0;
                if (appointment.Section != null && appointment.Section.Facility != null && appointment.Section.Facility.Site != null)
                    siteId = appointment.Section.Facility.Site.Id;
                
                Auditing.Logger.Log(
                Auditing.LogType.Information,
                siteId,
                Auditing.ObjectType.Appointment,
                Auditing.AppointmentFunction.AppointmentUpdated,
                appointment.Id,
                null,
                Auditing.Status.Success,
                false,
                string.Format("Patient ID={0}", appointment.Patient.Id));
                //                return appointment;
            }

            var appt = Get(appointment.Id);

            var modActivity = new AppointmentActivity() { Time = DateTime.UtcNow, ActivityType = ActivityType.Modified };
            appt.Activity.Add(modActivity);


            //TODO CALDAV
            var calendar = CalendarFactory.GetCalendarAccess();

            if (calendar.IsCaldavEnabled())
            {
                calendar.UpdateAppointment(appt);
            }

            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                session.SaveOrUpdate(appt);

                transaction.Commit();
            }

            return appt;
        }

        public bool Delete(int id)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                var appt = Get(id);

                if (appt == null)
                {
                    return false;
                }
                //TODO CALDAV
                var calendar = CalendarFactory.GetCalendarAccess();

                if (calendar.IsCaldavEnabled())
                {
                    calendar.DeleteAppointment(appt);
                }

                // TODO VISTA
                // clean up the appointment in Vista if possible
                var mdws = GetMDWSDAO(appt.Section.Facility.Site.VistaSiteId);
                mdws.CancelAppointment(appt, "C", "9", "Test Case");

                session.Delete(appt);
                
                transaction.Commit();
                return true;
            }
        }

        public Collection<Appointment> GetForPatient(int patientId, DateTime? start, DateTime? end)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                var appointments = from a in session.Query<Appointment>()
                                 where (a.Patient.Id == patientId  && 
                                      (!start.HasValue || a.Time >= start) &&
                                      (!end.HasValue || a.Time <= end))
                                 select a;

                transaction.Commit();
                return new Collection<Appointment>(appointments.ToList());
            }
        }

        public Collection<Appointment> GetNoShowAppointmentsForPatient(int patientId, DateTime? start, DateTime? end)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                var appointments = from a in session.Query<Appointment>()
                                   where (a.Patient.Id == patientId &&
                                        (!start.HasValue || a.Time >= start) &&
                                        (!end.HasValue || a.Time <= end) &&
                                        a.Activity.Any(ac=>ac.ActivityType == ActivityType.NoShow))
                                   select a;

                transaction.Commit();
                return new Collection<Appointment>(appointments.ToList());
            }            
        }

        public Collection<Appointment> GetForResource(int resourceId, DateTime? start, DateTime? end, AppointmentStatus status)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                var appointments = from a in session.Query<Appointment>()
//                                   where (a.Resources.Any(r => r.Resource.Id == resourceId) &&
                                    where (a.Resources.Any(r => r.Id == resourceId) && 
                                      (!start.HasValue || a.Time >= start) &&
                                      (!end.HasValue || a.Time <= end) &&
                                      (status == AppointmentStatus.Unknown || status == a.Status))
                                   select a;

                transaction.Commit();
                return new Collection<Appointment>(appointments.ToList());
            }
        }

        public Collection<Appointment> GetAllForAppointmentType(AppointmentType apptType, DateTime? start, DateTime? end, AppointmentStatus status)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                var appointments = from a in session.Query<Appointment>()
                                   //                                   where (a.Resources.Any(r => r.Resource.Id == resourceId) &&
                                   where (a.AppointmentType == apptType &&
                                     (!start.HasValue || a.Time >= start) &&
                                     (!end.HasValue || a.Time <= end) &&
                                     (status == AppointmentStatus.Unknown || status == a.Status))
                                   select a;

                transaction.Commit();
                return new Collection<Appointment>(appointments.ToList());
            }
        }

        // return all appointments related to this appointment, parent or child
        public Collection<Appointment> GetAllRelatedAppointments(Appointment appointment)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {

                // get all parents of this appointment
                var relations = from a in session.Query<Appointment>()
                                   where a.ChildAppointments.Any(x => x.ChildAppointment.Id == appointment.Id)                                    
                                   select a;
                List<Appointment> list = relations.ToList();

                // get all children of this appointment
                foreach (var childAppt in appointment.ChildAppointments)
                {
                    if (!list.Contains(childAppt.ChildAppointment))
                        list.Add(childAppt.ChildAppointment);
                }

                transaction.Commit();
                return new Collection<Appointment>(list);
            }
        }

        //public Appointment DoAction(Appointment appointment, AppointmentAction action, string reason)
        //{
        //    switch (action)
        //    {
        //        case AppointmentAction.Create:
        //            DoCreateAction(appointment);
        //            break;
        //        case AppointmentAction.CheckIn:
        //            DoCheckInAction(appointment);
        //            break;
        //        case AppointmentAction.CheckOut:
        //            DoCheckOutAction(appointment, reason);
        //            break;
        //        //case AppointmentAction.Cancel:
        //        //    DoCancelAction(appointment, reason);
        //        //    break;
        //        case AppointmentAction.NoShow:
        //            DoNoshowAction(appointment);
        //            break;
        //        //case AppointmentAction.Reschedule:
        //        //    DoResheduleAction(appointment);
        //        //    break;
        //        case AppointmentAction.WalkIn:
        //            DoWalkinAction(appointment);
        //            break;
        //        default:
        //            throw new ArgumentException("Unhandled action - " + action.ToString());
        //    }

        //    return appointment;
        //}

        public Appointment CreateAppointment(Appointment appointment)
        {
            if (appointment.Id != 0)
                throw new ArgumentException("Appointment is already created");

            if (appointment.Patient == null)
            {
                throw new ArgumentException("An appointment must be associated with patient");
            }

            if (appointment.Resources.Count == 0 || !appointment.Resources.Any(r=>r.Type == ResourceType.Provider))
            {
                throw new ArgumentException("An appointment must be associated with at least one provider resource");
            }
            if (appointment.Section == null)
            {
                throw new ArgumentException("An appointment must be associated with a clinic");
            }

            if (appointment.Time == null || appointment.Time <= DateTime.MinValue)
            {
                throw new ArgumentException("An appointment must have a valid time");
            }

            if (!VerifyResourceAvailability(appointment))
            {
                throw new ArgumentException("One or more resources are unavailable at the appointment time");
            }

            //TODO VISTA
            var mdws = GetMDWSDAO(appointment.Section.Facility.Site.VistaSiteId);
            
            try
            {
                mdws.MakeAppointment(appointment);
            }
            catch(Exception e)
            {
                throw new ApplicationException("Problem creating appointment in Vista " + e.Message, e);
            }
                       
            //TODO CALDAV
            var caldav = CalendarAccess.CalendarFactory.GetCalendarAccess();

            if (caldav.IsCaldavEnabled())
            {
                caldav.SaveAppointment(appointment);
            }

            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                session.Save(appointment);

                transaction.Commit();

                int siteId = 0;
                if (appointment.Section != null && appointment.Section.Facility != null && appointment.Section.Facility.Site != null)
                    siteId = appointment.Section.Facility.Site.Id;

                Auditing.Logger.Log(
                Auditing.LogType.Information,
                siteId,
                Auditing.ObjectType.Appointment,
                Auditing.AppointmentFunction.AppointmentCreated,
                appointment.Id,
                null,
                Auditing.Status.Success,
                false,
                string.Format("Patient ID={0}", appointment.Patient.Id));

            }

            var appt = Get(appointment.Id);

            appt.Status = AppointmentStatus.Scheduled;
            var newActivity = new AppointmentActivity() { Time = DateTime.UtcNow, ActivityType = ActivityType.Create };
            appt.AddActivity(newActivity);

            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                session.SaveOrUpdate(appt);

                transaction.Commit();
            }

            return appt;
        }

        private bool VerifyResourceAvailability(Appointment appointment)
        {
            DateTime apptTimeStart = appointment.Time;
            DateTime apptTimeEnd = appointment.Time.AddMinutes(appointment.Length);
            
            DateTime localApptDate = appointment.Time.Date;
            try
            {
                var tz = appointment.Section.Facility.Site.TimeZoneId;
                var tzInfo = TimeZoneInfo.FindSystemTimeZoneById(tz);
                localApptDate = TimeZoneInfo.ConvertTimeFromUtc(appointment.Time, tzInfo);
            }
            catch
            {
                localApptDate = appointment.Time.Date;
            }

            Collection<int> resourceIds = new Collection<int>();

            foreach (Resource r in appointment.Resources)
            {
                resourceIds.Add(r.Id);
            }

            var availability = ParentFactory.GetResourceService().GetAvailability(resourceIds, localApptDate, localApptDate);

            localApptDate = localApptDate.Date;
            foreach (var slotInstance in availability[localApptDate])
            {
                if (slotInstance.Time <= apptTimeStart && slotInstance.Time.AddMinutes(slotInstance.Length) >= apptTimeEnd)
                {
                    // slot found that will work
                    return true;
                }
            }
//            throw new Exception("No timeslot found for " + appointment.Time);
            return false;
        }

        public Appointment CancelAppointment(Appointment appointment, string cancelType, string cancelReason, string remarks)
        {
            if (appointment.Id == 0)
            {
                throw new ArgumentException("Only existing appointments can be cancelled");
            }

            if (appointment.Patient == null)
            {
                throw new ArgumentException("An appointment must be associated with patient");
            }

            if (appointment.Status != AppointmentStatus.Scheduled && appointment.Status != AppointmentStatus.CheckedIn)
            {
                throw new ArgumentException("Only scheduled or checked in appointments can be cancelled");
            }

            if (!String.Equals(cancelType, "C") && !String.Equals(cancelType, "PC"))
            {
                throw new ArgumentException("Invalid CancelType (Must be C or PC): " + cancelType);
            }

            if (String.IsNullOrEmpty(cancelReason))
            {
                throw new ArgumentException("Cancellations require a reason");
            }

            //TODO VISTA
            var mdws = GetMDWSDAO(appointment.Section.Facility.Site.VistaSiteId);

            try
            {
                mdws.CancelAppointment(appointment, cancelType, cancelReason, remarks);
            }
            catch (Exception e)
            {
                throw new ApplicationException("Problem cancelling appointment in Vista " + e.Message, e);
            }

            appointment.Status = AppointmentStatus.Cancelled;
            var newActivity = new AppointmentActivity() { Time = DateTime.UtcNow, ActivityType = ActivityType.Cancel, ActivityReason = remarks };

            appointment.AddActivity(newActivity);

            //TODO CALDAV
            var caldav = CalendarAccess.CalendarFactory.GetCalendarAccess();

            if (caldav.IsCaldavEnabled())
            {
                caldav.UpdateAppointment(appointment);
            }

            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                session.SaveOrUpdate(appointment);

                transaction.Commit();
            }

            return appointment;
        }

        public Appointment CheckInAppointment(Appointment appointment)
        {
            if (appointment.Id == 0)
            {
                throw new ArgumentException("Only existing appointments can be checked in");
            }

            if (appointment.Patient == null)
            {
                throw new ArgumentException("An appointment must be associated with patient");
            }

            if (appointment.Status != AppointmentStatus.Scheduled)
            {
                throw new ArgumentException("Only scheduled appointments can be checked in");
            }

            //TODO VISTA
            var mdws = GetMDWSDAO(appointment.Section.Facility.Site.VistaSiteId);

            try
            {
                mdws.CheckInAppointment(appointment);
            }
            catch (Exception e)
            {
                throw new ApplicationException("Problem checking in appointment in Vista " + e.Message, e);
            }

            appointment.Status = AppointmentStatus.CheckedIn;
            var newActivity = new AppointmentActivity() { Time = DateTime.UtcNow, ActivityType = ActivityType.CheckIn };

            appointment.AddActivity(newActivity);

            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                session.SaveOrUpdate(appointment);

                transaction.Commit();

                int siteId = 0;
                if (appointment.Section != null && appointment.Section.Facility != null && appointment.Section.Facility.Site != null)
                    siteId = appointment.Section.Facility.Site.Id;

                Auditing.Logger.Log(
                Auditing.LogType.Information,
                siteId,
                Auditing.ObjectType.Appointment,
                Auditing.AppointmentFunction.AppointmentCheckIn,
                appointment.Id,
                null,
                Auditing.Status.Success,
                false,
                string.Format("Patient ID={0}", appointment.Patient.Id));
            }

            return appointment;
        }

        public Appointment CreateWalkInAppointment(Appointment appointment)
        {
            if (appointment.Id != 0)
            {
                throw new ArgumentException("Only new appointments can be Walk Ins");
            }

            if (appointment.Patient == null)
            {
                throw new ArgumentException("An appointment must be associated with patient");
            }

            if (appointment.Time == null || appointment.Time <= DateTime.MinValue )
            {
                throw new ArgumentException("An appointment must have a valid time");
            }

            if (!VerifyResourceAvailability(appointment))
            {
                throw new ArgumentException("One or more resources are unavailable at the appointment time");
            }

            //TODO VISTA
            var mdws = GetMDWSDAO(appointment.Section.Facility.Site.VistaSiteId);

            try
            {
                mdws.MakeUnscheduledAppointment(appointment);
            }
            catch (Exception e)
            {
                throw new ApplicationException("Problem creating walk in appointment in Vista " + e.Message, e);
            }

            //TODO CALDAV
            var caldav = CalendarAccess.CalendarFactory.GetCalendarAccess();

            if (caldav.IsCaldavEnabled())
            {
                caldav.SaveAppointment(appointment);
            }

            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                session.Save(appointment);

                transaction.Commit();

                int siteId = 0;
                if (appointment.Section != null && appointment.Section.Facility != null && appointment.Section.Facility.Site != null)
                    siteId = appointment.Section.Facility.Site.Id;
                
                Auditing.Logger.Log(
                Auditing.LogType.Information,
                siteId,
                Auditing.ObjectType.Appointment,
                Auditing.AppointmentFunction.AppointmentCreated,
                appointment.Id,
                null,
                Auditing.Status.Success,
                false,
                string.Format("Patient ID={0}", appointment.Patient.Id));

            }

            var appt = Get(appointment.Id);

            appt.Status = AppointmentStatus.CheckedIn;
            var newActivity = new AppointmentActivity() { Time = DateTime.UtcNow, ActivityType = ActivityType.WalkIn };
            appt.AddActivity(newActivity);
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                session.SaveOrUpdate(appt);

                transaction.Commit();
            }

            return appt;
        }

        public Appointment CheckOutAppointment(Appointment appointment, string disposition) 
        {
            if (appointment.Id == 0)
            {
                throw new ArgumentException("Only existing appointments can be checked out");
            }

            if (appointment.Patient == null)
            {
                throw new ArgumentException("An appointment must be associated with patient");
            }

            if (appointment.Status != AppointmentStatus.CheckedIn)
            {
                throw new ArgumentException("Only checked in appointments can be checked out");
            }

            //TODO VISTA
            var mdws = GetMDWSDAO(appointment.Section.Facility.Site.VistaSiteId);

            try
            {
                mdws.CheckOutAppointment(appointment);
            }
            catch (Exception e)
            {
                throw new ApplicationException("Problem checking out appointment in Vista " + e.Message, e);
            }

            appointment.Status = AppointmentStatus.Closed;
            var newActivity = new AppointmentActivity() { Time = DateTime.UtcNow, ActivityType = ActivityType.CheckOut, ActivityReason = disposition };

            appointment.AddActivity(newActivity);

            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                session.SaveOrUpdate(appointment);

                int siteId = 0;
                if (appointment.Section != null && appointment.Section.Facility != null && appointment.Section.Facility.Site != null)
                    siteId = appointment.Section.Facility.Site.Id;

                transaction.Commit();
                Auditing.Logger.Log(
                Auditing.LogType.Information,
                siteId,
                Auditing.ObjectType.Appointment,
                Auditing.AppointmentFunction.AppointmentCheckOut,
                appointment.Id,
                null,
                Auditing.Status.Success,
                false,
                string.Format("Patient ID={0}", appointment.Patient.Id));

            }

            return appointment;
        }

        public Appointment NoShowAppointment(Appointment appointment)
        {
            if (appointment.Id == 0)
            {
                throw new ArgumentException("Only existing appointments can be marked no show");
            }

            if (appointment.Patient == null)
            {
                throw new ArgumentException("An appointment must be associated with patient");
            }

            if (appointment.Status != AppointmentStatus.Scheduled)
            {
                throw new ArgumentException("Only checked in appointments can be checked out");
            }

            //TODO VISTA
            var mdws = GetMDWSDAO(appointment.Section.Facility.Site.VistaSiteId);

            try
            {
                mdws.NoShowAppointment(appointment);
            }
            catch (Exception e)
            {
                throw new ApplicationException("Problem marking noshow for appointment in Vista " + e.Message, e);
            }

            appointment.Status = AppointmentStatus.Closed;
            var newActivity = new AppointmentActivity() { Time = DateTime.UtcNow, ActivityType = ActivityType.NoShow };

            appointment.AddActivity(newActivity);

            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                session.SaveOrUpdate(appointment);

                transaction.Commit();
            }

            return appointment;
        }

        //public void DoResheduleAction(Appointment appointment)
        //{
        //}

        public CommunicationQueueItem AddToCommunicationQueue(CommunicationQueueItem queueItem)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                session.SaveOrUpdate(queueItem);

                transaction.Commit();
                return queueItem;
            }
        }

        public Collection<CommunicationQueueItem> GetAllRelatedCommunications(Appointment appointment)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                var queueItems = from q in session.Query<CommunicationQueueItem>()
                                 where q.Appointment == appointment
                                 select q;

                transaction.Commit();
                return new Collection<CommunicationQueueItem>(queueItems.ToList());
            }
        }

        public Collection<CommunicationQueueItem> GetAllRelatedCommunications(AppointmentRequest appointmentRequest)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                var queueItems = from q in session.Query<CommunicationQueueItem>()
                                 where q.AppointmentRequest == appointmentRequest
                                 select q;

                transaction.Commit();
                return new Collection<CommunicationQueueItem>(queueItems.ToList());
            }
        }

        public AppointmentRequest AddAppointmentRequest(AppointmentRequest request)
        {
            // section must be populated
            if (request.RequestedSection == null)
            {
                if (request.RequestedResource == null)
                {
                    throw new ArgumentException("Requests must be associated with a clinic or a resource");
                }
                request.RequestedSection = request.RequestedResource.Section;
            }

            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                session.SaveOrUpdate(request);

                transaction.Commit();
                return request;
            }
        }

        public AppointmentRequest UpdateAppointmentRequest(AppointmentRequest request)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                session.SaveOrUpdate(request);

                transaction.Commit();
                return request;
            }
        }

        public Collection<AppointmentRequest> GetAppointmentRequestsForPatient(Patient patient)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                var list = from r in session.Query<AppointmentRequest>()
                                 where r.Patient.Id == patient.Id                         
                                 select r;
                list.OrderBy(r => r.Priority)
                    .ThenBy(r => r.Time);               

                transaction.Commit();
                return new Collection<AppointmentRequest>(list.ToList());
            }
        }

        public Collection<AppointmentRequest> GetAllOpenAppointmentRequests(Section section)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                var list = from r in session.Query<AppointmentRequest>()
                                 where r.RequestedSection == section && r.Status == RequestStatus.Pending
                                 select r;
                list.OrderBy(r => r.Priority)
                    .ThenBy(r => r.Time);               

                transaction.Commit();
                return new Collection<AppointmentRequest>(list.ToList());
            }
        }

        public Collection<AppointmentRequest> GetAppointmentRescheduleList(Section section)
        {
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                var list = from r in session.Query<AppointmentRequest>()
                                 where r.RequestedSection == section && 
                                 r.Status == RequestStatus.Pending &&
                                 r.Reason == RequestReason.Cancellation
                                 select r;
                list.OrderBy(r => r.Priority)
                    .ThenBy(r => r.Time);               

                transaction.Commit();
                return new Collection<AppointmentRequest>(list.ToList());
            }
        }

        //public string GetAppointmentVistaStatus(Appointment appointment)
        //{
        //    string vistaSiteId = "555"; // appointment.Section.Facility.Site.VistaSiteId;
        //    var mdws = GetMDWSDAO(vistaSiteId);

        //    string patId = "795"; //appointment.Patient.GetDFNForVistASite(vistaSiteId);
        //    string apptId = "3130506.13"; // appointment.VistaId;

        //    return mdws.GetAppointmentStatus(patId, apptId);
        //}

        public CommunicationQueueItem AddToCommunicationQueue(CommunicationTemplate template, Appointment appointment)
        {
            var newQueueItem = new CommunicationQueueItem()
            {
                Appointment = appointment,
                Body = template.Message,
                Medium = template.Medium,
                Recipient = template.Recipient,
                SendBy = DateTime.Today.AddDays(10),
                Subject = template.Subject,
                Template = template
            };

            newQueueItem = CommunicationHelper.ProcessCommunicationTemplate(newQueueItem);

            return this.AddToCommunicationQueue(newQueueItem);
        }

        public CommunicationQueueItem AddToCommunicationQueue(CommunicationTemplate template, AppointmentRequest appointmentRequest)
        {
            var newQueueItem = new CommunicationQueueItem()
            {
                AppointmentRequest = appointmentRequest,
                Body = template.Message,
                Medium = template.Medium,
                Recipient = template.Recipient,
                SendBy = DateTime.Today.AddDays(10),
                Subject = template.Subject,
                Template = template
            };

            newQueueItem = CommunicationHelper.ProcessCommunicationTemplate(newQueueItem);

            return this.AddToCommunicationQueue(newQueueItem);
        }

        public Appointment CreateRecurringAppointments(Appointment appointment, Recur recur)
        {
            // assuming that the first request has already been made
            var firstAppt = CreateAppointment(appointment);

            var previous = firstAppt;

            var provider = firstAppt.Resources.FirstOrDefault(r=> r.Type == ResourceType.Provider);

            // now follow the recurring pattern and create a request and an appointment
            if (recur != null)
            {
                DateTime dateToCheck = firstAppt.Time;
                int interval;
                if (Int32.TryParse(recur.RecurringInterval, out interval))
                {
                    if (interval > 0)
                    {
                        while (dateToCheck <= recur.RecurringEnd.Date) 
                        {
                            if (recur.RecurringFrequency.Equals("Daily"))
                            {
                                dateToCheck = dateToCheck.AddDays(interval);
                            }
                            else if (recur.RecurringFrequency.Equals("Weekly"))
                            {
                                dateToCheck = dateToCheck.AddDays(interval * 7);
                            }
                            else if (recur.RecurringFrequency.Equals("Monthly"))
                            {
                                dateToCheck = dateToCheck.AddMonths(interval);
                            }
                            else
                            {
                                break;
                            }

                            if (dateToCheck <= recur.RecurringEnd.Date)
                            {
                                // create request
                                var newRequest = new AppointmentRequest()
                                {
                                    CreatingUser = ParentFactory.CurrentUser,
                                    DesiredDate = dateToCheck,
                                    Patient = firstAppt.Patient,
                                    Priority = 0,
                                    Reason = RequestReason.Cancellation,
                                    RequestedSection = firstAppt.Section,
                                    RequestedResource = provider,
                                    RequestMethod = RequestMethod.Automatic,
                                    Status = RequestStatus.Pending,
                                    Time = DateTime.UtcNow,
                                };

                                // create appointment
                                var newAppt = new Appointment()
                                {
                                    AppointmentType = firstAppt.AppointmentType,
                                    HighPriorityReschedule = firstAppt.HighPriorityReschedule,
                                    Length = firstAppt.Length,
                                    Section = firstAppt.Section,
                                    Patient = firstAppt.Patient,
                                    TelehealthLocation = firstAppt.TelehealthLocation,
                                    Time = dateToCheck
                                };

                                foreach (var resource in firstAppt.Resources)
                                {
                                    newAppt.AddResource(resource);
                                }

                                // try save appointment
                                try
                                {
                                    newAppt = CreateAppointment(newAppt);

                                    // if success, set request to completed
                                    newRequest.Status = RequestStatus.Complete;
                                    newRequest.ResultingAppointment = newAppt;
                                    previous.AddChildAppointment(newAppt, AppointmentRelationType.Recurrence);
                                    Update(previous);
                                    previous = newAppt;
                                }
                                catch
                                {
                                }

                                // save request
                                this.AddAppointmentRequest(newRequest);
                            } 
                        }

 //                       Update(firstAppt);
                    }
                }
            }

            return firstAppt;
        }
    }
}
