﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using MedRed.CalendarAccess;
using MedRed.MDWSAccess;
using MedRed.Services.Interfaces;
using MedRed.Services.ServiceImpl;
using Shared.Model;
using Shared.Model.Config.MDWS;

namespace MedRed.Services
{
    public class Factory : IDisposable
    {
        internal static MDWSConfiguration Configuration { get; set; }
//        internal MDWSConnection ConnectionDetails { get; set; }
        internal static Dictionary<string, MDWSCalls> mdwsCache;

        private User currentUser;
        internal User CurrentUser { get { return currentUser; } }

        private Factory(User user)
        {
            currentUser = user;
            mdwsCache = new Dictionary<string, MDWSCalls>();
        }

        internal MDWSCalls getCachedMDWS(string vistaSiteId)
        {
            if (!mdwsCache.ContainsKey(vistaSiteId))
            {
                var _MDWSdao = new MDWSCalls();
                _MDWSdao.setConfig(Factory.Configuration);
                _MDWSdao.setConnection(GetVistaServiceAccountForSite(vistaSiteId));

                mdwsCache.Add(vistaSiteId, _MDWSdao);
            }

            return mdwsCache[vistaSiteId];
        }

        internal MDWSConnection GetVistaServiceAccountForSite(string vistaSiteId)
        {
            MDWSConnection connection = null;
            if (!Configuration.ServiceAccounts.TryGetValue(vistaSiteId, out connection))
            {
                throw new ArgumentException("No service account configured for Vista Site " + vistaSiteId);
            }

            return connection;
        }

        public static void SetConfiguration(MDWSConfiguration config)
        {
            Configuration = config;
            //            ConnectionDetails = connectionDetails;
            DataAccess.DataAccess.SetConnectionString(config.DB_ConnectionString);
            CalendarFactory.Initialize(config.Caldav_Source);
        }

        public static Factory GetFactory(MDWSConfiguration config)
        {
            SetConfiguration(config);
            return GetFactory();
        }

        public static Factory GetFactory()
        {
            if (Configuration == null)
                throw new ArgumentException("Call SetConfiguration first");

            var username = System.Threading.Thread.CurrentPrincipal.Identity.Name;

            // specifically load the admin user account to use
            string ADMIN_USERNAME = "admin";

            if (String.IsNullOrEmpty(username))
                username = ADMIN_USERNAME;

            User user = null;
            using (var session = MedRed.DataAccess.DataAccess.OpenSession())
            using (var transaction = session.BeginTransaction())
            {
                var users = session.QueryOver<User>()
                    .Where(u => u.Username == username);
                var userList = users.List();
                transaction.Commit();
                user = userList.FirstOrDefault();
            }

            if (user == null)
            {
                using (var session = MedRed.DataAccess.DataAccess.OpenSession())
                using (var transaction = session.BeginTransaction())
                {
                    var users = session.QueryOver<User>()
                        .Where(u => u.Username == ADMIN_USERNAME);
                    var userList = users.List();
                    transaction.Commit();
                    user = userList.FirstOrDefault();
                }

                if (user == null)
                {
                    throw new ApplicationException("No user exists for " + username);
                }
            }

           return GetFactory(user);
        }

        private static Factory GetFactory(User user)
        {
            if (Configuration == null)
                throw new ArgumentException("Call SetConfiguration first");
         
            return new Factory(user);
        }

        public void ValidateSchema()
        {
            DataAccess.DataAccess.ValidateSchema();
        }

        public IAccountService GetAccountService()
        {
            return new AccountService(this);
        }

        public IAppointmentService GetAppointmentService()
        {
            return new AppointmentService(this);
        }

        public IEquipmentService GetEquipmentService()
        {
            return new EquipmentService(this);
        }

        public IFacilityService GetFacilityService()
        {
            return new FacilityService(this);
        }

        public IImportService GetImportService()
        {
            return new ImportService(this);
        }

        public IPatientService GetPatientService()
        {
            return new PatientService(this);
        }

        public IProviderService GetProviderService()
        {
            return new ProviderService(this);
        }

        public IResourceService GetResourceService()
        {
            return new ResourceService(this);
        }

        public IRoomService GetRoomService()
        {
            return new RoomService(this);
        }

        public ISectionService GetSectionService()
        {
            return new SectionService(this);
        }

        public ISiteService GetSiteService()
        {
            return new SiteService(this);
        }

        public INationalSystemService GetNationalSystemService()
        {
            return new NationalSystemService(this);
        }

        public IUIService GetUIService()
        {
            return new UIService(this);
        }

        public void Dispose()
        {
            foreach (var site in mdwsCache.Keys)
            {
                var mdws = mdwsCache[site];
//                mdwsCache.Remove(site);
                try
                {
                    mdws.Dispose();
                }
                catch(Exception)
                {
                }
            }
        }
    }
}
