using MedRed.Services;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using MedRed.DataAccess;
using NHibernate;
using MedRed.MDWSAccess;
using Shared.Model.Config.MDWS;
using System.Collections.Generic;
using Shared.Model;
using KellermanSoftware.CompareNetObjects;

namespace MedRed.Services.Tests
{
    [TestClass()]
    public class SectionServiceTest
    {   
        private TestContext testContextInstance;

        /// <summary>
        ///Gets or sets the test context which provides
        ///information about and functionality for the current test run.
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        #region Additional test attributes
        // 
        //You can use the following additional attributes as you write your tests:
        //
        //Use ClassInitialize to run code before running the first test in the class
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //Use ClassCleanup to run code after all tests in a class have run
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //Use TestInitialize to run code before running each test
        [TestInitialize()]
        public void MyTestInitialize()
        {
            MDWSConfiguration config = ConnectionHelper.GetTestingConfig();
            var factory = Factory.GetFactory(config);

            TestContext.Properties["Factory"] = factory;

            var hierarchy = (log4net.Repository.Hierarchy.Hierarchy)log4net.LogManager.GetRepository();
            var logger = (log4net.Repository.Hierarchy.Logger)hierarchy.GetLogger("NHibernate.SQL");
            logger.AddAppender(new log4net.Appender.TraceAppender { Layout = new log4net.Layout.SimpleLayout() });
            hierarchy.Configured = true;

        }

        //Use TestCleanup to run code after each test has run
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion


        [TestMethod()]
        public void getAllSectionsTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetSectionService();

            Assert.IsNotNull(service);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            {
                Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
                dbSession.Save(site);
                Facility facility = new Facility() { Name = "Facility 1", Site = site };
                dbSession.Save(facility);

                int facilityId = facility.Id;

                Section[] array = {new Section(){ Name = "Section A", Facility = facility},
                               new Section() { Name = "Section B", Facility = facility}};

                dbSession.Save(array[0]);
                dbSession.Save(array[1]);

                var sections = service.GetAll(facilityId);
                var sectionList = new List<Section>(sections);

                Assert.IsTrue(sectionList.Count >= 2);
                Assert.IsTrue(sectionList.Exists(s => s.Name == "Section A"));
                Assert.IsTrue(sectionList.Exists(s => s.Name == "Section B"));

                dbSession.Delete(array[0]);
                dbSession.Delete(array[1]);
                dbSession.Delete(facility);
                dbSession.Delete(site);
            }
        }

        [TestMethod()]
        public void getSectionTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetSectionService();

            Assert.IsNotNull(service);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            {
                Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
                dbSession.Save(site);
                Facility facility = new Facility() { Name = "Facility 1", Site = site };
                dbSession.Save(facility);

                int facilityId = facility.Id;

                Section[] array = {new Section(){ Name = "Section A", Facility = facility},
                               new Section() { Name = "Section B", Facility = facility}};

                dbSession.Save(array[0]);
                dbSession.Save(array[1]);

                int sectionId = array[0].Id;
                Assert.IsTrue(sectionId != 0);

                var retrievedSection = service.Get(sectionId);

                CompareObjects compareObjects = new CompareObjects();
                Assert.IsTrue(compareObjects.Compare(retrievedSection, array[0]), compareObjects.DifferencesString);

                dbSession.Delete(array[0]);
                dbSession.Delete(array[1]);
                dbSession.Delete(facility);
                dbSession.Delete(site);
            }
        }

        [TestMethod()]
        public void addSectionTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetSectionService();

            Assert.IsNotNull(service);

            Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
            Facility facility = new Facility() { Name = "Facility 1", Site = site };
            
            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(site);
                dbSession.Save(facility);
                dbSession.Clear();

                transaction.Commit();
            }

            Section newSection = Utils.CreateSection(facility);

            Assert.IsTrue(newSection.Id == 0);
            var section = service.Add(newSection);
            int sectionId = section.Id;

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {

                var retrievedSection = service.Get(sectionId);

                CompareObjects compareObjects = new CompareObjects();
                Assert.IsTrue(compareObjects.Compare(retrievedSection, newSection), compareObjects.DifferencesString);

                dbSession.Delete(section);
                dbSession.Delete(facility);
                dbSession.Delete(site);

                transaction.Commit();
            }
        }


        [TestMethod()]
        public void deactivateSectionWithAppoitmentTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetSectionService();

            Assert.IsNotNull(service);

            Site site = Utils.GetExistingSiteforVistaCode(factory, "555");
            Facility facility = Utils.CreateAndSaveFacility(site);
            Section section = Utils.CreateAndSaveSectionWithAppTypes(factory, "555", facility);

            //Provider p = new Provider() { Person = new Person() { FirstName = "Henry", LastName = "Jones" } };
            //p.AddProviderSite("555", "11298");
            //Resource resource1 = new Resource() { Name = "Dr. Jones", Section = section, Type = ResourceType.Provider};

            Resource resource1 = Utils.GetRealProviderResource(factory, "555", section);
            Resource resource2 = Utils.GetRoomResource(factory, "555", section);

            facility.CommunicationTemplates.Add(new CommunicationTemplate()
            {
                Subject = "subject",
                Message = "Body",
                Medium = "Email",
                DateOffset = 0,
                Recipient = "Patient",
                Name = "Standard cancellation letter",
                Type = CommunicationType.Cancellation
            });

            factory.GetFacilityService().Update(facility);

            Patient patienta = Utils.GetFirstPatientFromVista(factory, "555");
            Patient patientb = Utils.GetLastPatientFromVista(factory, "555");

//            using (var dbSession = DataAccess.DataAccess.OpenSession())
//            using (var transaction = dbSession.BeginTransaction())
//            {
//                //dbSession.Save(site);
//                //dbSession.Save(facility);
//                //dbSession.Save(section);
////                dbSession.Save(resource);
//                dbSession.Save(p);

////                dbSession.Save(patienta);
////                dbSession.Save(patientb);

//                transaction.Commit();
//            }

//            resource1.FullfillingResourceId = p.Id;

            //factory.GetResourceService().Add(resource1);
//            factory.GetResourceService().Add(resource2);

            Appointment appointment1 = new Appointment()
            {
                Time = new DateTime(2013, 5, 17, 8, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
                Section = section,
                AppointmentType = section.AppointmentTypes[0]
            };
            appointment1.AddResource(resource1);
            appointment1.AddResource(resource2);

            Appointment appointment2 = new Appointment()
            {
                Time = new DateTime(2013, 5, 17, 9, 0, 0),
                Length = 60,
                Patient = patientb,
                Status = AppointmentStatus.Scheduled,
                Section = section,
                AppointmentType = section.AppointmentTypes[0]
            };
            appointment2.AddResource(resource1);
            appointment2.AddResource(resource2);

            Appointment appointment3 = new Appointment()
            {
                Time = new DateTime(2013, 5, 20, 9, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
                Section = section,
                AppointmentType = section.AppointmentTypes[0]
            };
            appointment3.AddResource(resource1);

            //Appointment appointment4 = new Appointment()
            //{
            //    Time = new DateTime(2013, 5, 1, 9, 0, 0),
            //    Length = 60,
            //    Patient = patienta,
            //    Status = AppointmentStatus.Scheduled,
            //    Section = section,
            //    AppointmentType = section.AppointmentTypes[0]
            //};            
            //appointment4.AddResource(resource2);

            factory.GetAppointmentService().CreateAppointment(appointment1);
            factory.GetAppointmentService().CreateAppointment(appointment2);
            factory.GetAppointmentService().CreateAppointment(appointment3);
            //factory.GetAppointmentService().CreateAppointment(appointment4);
            //using (var dbSession = DataAccess.DataAccess.OpenSession())
            //using (var transaction = dbSession.BeginTransaction())
            //{
            //    dbSession.Save(appointment1);
            //    dbSession.Save(appointment2);
            //    dbSession.Save(appointment3);
            //    dbSession.Save(appointment4);
            //    transaction.Commit();
            //}

            var apptService = factory.GetAppointmentService();
            var appt1 = apptService.Get(appointment1.Id);
            var appt2 = apptService.Get(appointment2.Id);
            var appt3 = apptService.Get(appointment3.Id);
            //var appt4 = apptService.Get(appointment4.Id);

            appt1.AddChildAppointment(appt3, AppointmentRelationType.Followup);
            //appt1.AddChildAppointment(appt4, AppointmentRelationType.Followup);

            appt1 = factory.GetAppointmentService().Update(appt1);

            var retrievedSection = service.Get(section.Id);

            var communications = factory.GetAppointmentService().GetAllRelatedCommunications(appt1);
            Assert.AreEqual(0, communications.Count);
            var requests = factory.GetAppointmentService().GetAllOpenAppointmentRequests(section);

            retrievedSection.Active = false;

            var updatedSection = service.Update(retrievedSection);

            var communications2 = factory.GetAppointmentService().GetAllRelatedCommunications(appt1);
            Assert.AreEqual(communications.Count + 1, communications2.Count);
            var request2 = factory.GetAppointmentService().GetAllOpenAppointmentRequests(section);
            Assert.IsTrue(requests.Count < request2.Count);

            appt1 = apptService.Get(appointment1.Id);
            appt2 = apptService.Get(appointment2.Id);
            appt3 = apptService.Get(appointment3.Id);
            //appt4 = apptService.Get(appointment4.Id);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                foreach (var v in factory.GetAppointmentService().GetAllRelatedCommunications(appt1))
                    dbSession.Delete(v);
                foreach (var v in factory.GetAppointmentService().GetAllRelatedCommunications(appt2))
                    dbSession.Delete(v);
                foreach (var v in factory.GetAppointmentService().GetAllRelatedCommunications(appt3))
                    dbSession.Delete(v);
                //foreach (var v in factory.GetAppointmentService().GetAllRelatedCommunications(appt4))
                //    dbSession.Delete(v);
                foreach (var r in request2)
                    dbSession.Delete(r);

                dbSession.Delete(appt1);
                dbSession.Delete(appt2);
                dbSession.Delete(appt3);
                //dbSession.Delete(appt4);
//                dbSession.Delete(patienta);
//                dbSession.Delete(patientb);
                dbSession.Delete(resource1);
                dbSession.Delete(resource2);
                dbSession.Delete(updatedSection);
                dbSession.Delete(facility);
//                dbSession.Delete(site);

                transaction.Commit();
            }
        }


        [TestMethod()]
        [ExpectedException(typeof(ArgumentException))]
        public void addOrphanSectionTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetSectionService();

            Assert.IsNotNull(service);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            {                
                Section[] array = { new Section() { Name = "Section A" } };

                Assert.IsTrue(array[0].Id == 0);
                var section = service.Add(array[0]);
                int sectionId = section.Id;

                var retrievedSection = service.Get(sectionId);

                CompareObjects compareObjects = new CompareObjects();
                Assert.IsTrue(compareObjects.Compare(retrievedSection, array[0]), compareObjects.DifferencesString);

                dbSession.Delete(array[0]);
            }
        }

        [TestMethod()]
        public void addSectionDontOverrideFacilityTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetSectionService();

            Assert.IsNotNull(service);

            int facilityId =0;
            Site site = Utils.CreateAndSaveSite();
            Facility facility = Utils.CreateAndSaveFacility(site);
            string facilityName = facility.Name;

            facilityId = facility.Id;
 
            Section[] array = { new Section() { Name = "Section A", DSSPrimaryStopCode = "368", Facility = new Facility() { Id = facilityId } } };

            Assert.IsTrue(array[0].Id == 0);
            var section = service.Add(array[0]);
            int sectionId = section.Id;

            var retrievedSection = service.Get(sectionId);

            Assert.AreEqual(facilityName, retrievedSection.Facility.Name);

            array[0].Facility = facility;
            CompareObjects compareObjects = new CompareObjects();
            Assert.IsTrue(compareObjects.Compare(retrievedSection, array[0]), compareObjects.DifferencesString);
            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {

                dbSession.Delete(array[0]);
                dbSession.Delete(facility);
                dbSession.Delete(site);
            }           
        }

        [TestMethod()]
        public void addCommTemplateSectionTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetSectionService();

            Assert.IsNotNull(service);

            int sectionId = 0;
            Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
            Facility facility = new Facility() { Name = "Facility 1", Site = site };
            Section section = new Section() { Name = "Section A", Facility = facility };

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(site);
                dbSession.Save(facility);
                dbSession.Save(section);

                sectionId = section.Id;

                Assert.IsTrue(sectionId != 0);
                transaction.Commit();
            }

            CommunicationTemplate[] commTemplates = 
            {
                new CommunicationTemplate() { Name="Template 1", Type = CommunicationType.PreAppointment, Message = "My message", Recipient = "Patient"},
                new CommunicationTemplate() { Name="Template 2", Type = CommunicationType.Cancellation, Message = "Sorry your appt is cancelled", Recipient = "Patient"},
            };

            service.AddOrUpdateCommunicationTemplate(sectionId, commTemplates[0]);
            service.AddOrUpdateCommunicationTemplate(sectionId, commTemplates[1]);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                var retrievedSection = service.Get(sectionId);

                Assert.AreEqual(retrievedSection.CommunicationTemplates.Count, 2);

                CompareObjects compareObjects = new CompareObjects();
                //                compareObjects.ElementsToIgnore.Add("Id");
                Assert.IsTrue(compareObjects.Compare(retrievedSection.CommunicationTemplates[0], commTemplates[0]), compareObjects.DifferencesString);
                Assert.IsTrue(compareObjects.Compare(retrievedSection.CommunicationTemplates[1], commTemplates[1]), compareObjects.DifferencesString);

                dbSession.Delete(retrievedSection);
                dbSession.Delete(facility);
                dbSession.Delete(site);
                transaction.Commit();
            }
        }

        [TestMethod()]
        public void EditCommTemplateSectionTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetSectionService();

            Assert.IsNotNull(service);

            int sectionId = 0;
            Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
            Facility facility = new Facility() { Name = "Facility 1", Site = site };
            Section section = new Section() { Name = "Section A", Facility = facility };

            CommunicationTemplate[] commTemplates = 
            {
                new CommunicationTemplate() { Name="Template 1", Type = CommunicationType.PreAppointment, Message = "My message", Recipient = "Patient"},
                new CommunicationTemplate() { Name="Template 2", Type = CommunicationType.Cancellation, Message = "Sorry your appt is cancelled", Recipient = "Patient"},
            };

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(site);
                dbSession.Save(facility);

                section.CommunicationTemplates.Add(commTemplates[0]);
                section.CommunicationTemplates.Add(commTemplates[1]);

                dbSession.Save(section);
                sectionId = section.Id;

                Assert.IsTrue(sectionId != 0);
                transaction.Commit();
            }

            var original = service.Get(sectionId);

            var communicationToEdit = original.CommunicationTemplates[1];

            communicationToEdit.Message = "This is my updated message";

            service.AddOrUpdateCommunicationTemplate(sectionId, communicationToEdit);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                var retrieved = service.Get(sectionId);

                Assert.AreEqual(retrieved.CommunicationTemplates.Count, 2);

                Assert.AreEqual(retrieved.CommunicationTemplates[1].Message, "This is my updated message");

                dbSession.Delete(retrieved);
                dbSession.Delete(facility);
                dbSession.Delete(site);
                transaction.Commit();
            }

        }

        [TestMethod()]
        public void DeleteCommTemplateSectionTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetSectionService();

            Assert.IsNotNull(service);

            int sectionId = 0;
            Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
            Facility facility = new Facility() { Name = "Facility 1", Site = site };
            Section section = new Section() { Name = "Section A", Facility = facility };

            CommunicationTemplate[] commTemplates = 
            {
                new CommunicationTemplate() { Name="Template 1", Type = CommunicationType.PreAppointment, Message = "My message", Recipient = "Patient"},
                new CommunicationTemplate() { Name="Template 2", Type = CommunicationType.Cancellation, Message = "Sorry your appt is cancelled", Recipient = "Patient"},
            };

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(site);
                dbSession.Save(facility);

                section.CommunicationTemplates.Add(commTemplates[0]);
                section.CommunicationTemplates.Add(commTemplates[1]);

                dbSession.Save(section);

                sectionId = section.Id;

                Assert.IsTrue(sectionId != 0);
                transaction.Commit();
            }

            var original = service.Get(sectionId);

            var communicationToDelete = original.CommunicationTemplates[1];

            service.DeleteCommunicationTemplate(sectionId, communicationToDelete);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                var retrieved = service.Get(sectionId);

                Assert.AreEqual(retrieved.CommunicationTemplates.Count, 1);

                dbSession.Delete(retrieved);
                dbSession.Delete(facility);
                dbSession.Delete(site);
                transaction.Commit();
            }
        }

        [TestMethod()]
        public void addAppointmentTypeSectionTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetSectionService();

            Assert.IsNotNull(service);

            int sectionId = 0;
            Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
            Facility facility = new Facility() { Name = "Facility 1", Site = site };
            Section section = new Section() { Name = "Section A", Facility = facility };
            AppointmentTypeCategory appTypeCat = new AppointmentTypeCategory(){Name = "Regular"};

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(site);
                dbSession.Save(facility);
                dbSession.Save(section);
                dbSession.Save(appTypeCat);
                sectionId = section.Id;

                Assert.IsTrue(sectionId != 0);
                transaction.Commit();
            }

            AppointmentType[] appTypes = 
            {
                new AppointmentType() { Name="Initial", Duration = 60, DSSPrimaryStopCode="323", AppointmentTypeCategory = appTypeCat},
                new AppointmentType() { Name="Followup", Duration = 30, DSSPrimaryStopCode = "324", AppointmentTypeCategory = appTypeCat} 
            };

            appTypes[0].AddPatientInstructions(new PatientInstructions() { Type = InstructionType.PreAppointment, Instructions = "Don't eat for 24 hours" });

            service.AddOrUpdateAppointmentType(sectionId, appTypes[0]);
            service.AddOrUpdateAppointmentType(sectionId, appTypes[1]);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                var retrievedSection = service.Get(sectionId);

                Assert.AreEqual(retrievedSection.AppointmentTypes.Count, 2);

                CompareObjects compareObjects = new CompareObjects();
                compareObjects.ElementsToIgnore.Add("Section");
                Assert.IsTrue(compareObjects.Compare(retrievedSection.AppointmentTypes[0], appTypes[0]), compareObjects.DifferencesString);
                Assert.IsTrue(compareObjects.Compare(retrievedSection.AppointmentTypes[1], appTypes[1]), compareObjects.DifferencesString);

                dbSession.Delete(retrievedSection);
                dbSession.Delete(facility);
                dbSession.Delete(site);
                transaction.Commit();
            }
        }

        [TestMethod()]
        public void EditAppointmentTypeSectionTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetSectionService();

            Assert.IsNotNull(service);

            int sectionId = 0;
            Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
            Facility facility = new Facility() { Name = "Facility 1", Site = site };
            Section section = new Section() { Name = "Section A", Facility = facility };
            AppointmentTypeCategory appTypeCat = new AppointmentTypeCategory() { Name = "Regular" };

            AppointmentType[] appTypes = 
            {
                new AppointmentType() { Name="Initial", Duration = 60, DSSPrimaryStopCode="323", AppointmentTypeCategory = appTypeCat},
                new AppointmentType() { Name="Followup", Duration = 30, DSSPrimaryStopCode = "324", AppointmentTypeCategory = appTypeCat} 
            };

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(appTypeCat);
                dbSession.Save(site);
                dbSession.Save(facility);

                section.AppointmentTypes.Add(appTypes[0]);
                section.AppointmentTypes.Add(appTypes[1]);

                dbSession.Save(section);
                sectionId = section.Id;

                Assert.IsTrue(sectionId != 0);
                transaction.Commit();
            }

            var original = service.Get(sectionId);

            var appTypeToEdit = original.AppointmentTypes[1];

            appTypeToEdit.Name = "My Updated Followup";

            service.AddOrUpdateAppointmentType(sectionId, appTypeToEdit);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                var retrieved = service.Get(sectionId);

                Assert.AreEqual(2, retrieved.AppointmentTypes.Count);

                Assert.AreEqual(retrieved.AppointmentTypes[1].Name, "My Updated Followup");

                dbSession.Delete(retrieved);
                dbSession.Delete(facility);
                dbSession.Delete(site);
                transaction.Commit();
            }

        }

        [TestMethod()]
        public void DeleteAppointmentTypeSectionTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetSectionService();

            Assert.IsNotNull(service);

            int sectionId = 0;
            Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
            Facility facility = new Facility() { Name = "Facility 1", Site = site };
            Section section = new Section() { Name = "Section A", Facility = facility };
            AppointmentTypeCategory appTypeCat = new AppointmentTypeCategory() { Name = "Regular" };

            AppointmentType[] appTypes = 
            {
                new AppointmentType() { Name="Initial", Duration = 60, DSSPrimaryStopCode="323", AppointmentTypeCategory = appTypeCat},
                new AppointmentType() { Name="Followup", Duration = 30, DSSPrimaryStopCode = "324", AppointmentTypeCategory = appTypeCat} 
            };

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(appTypeCat);
                dbSession.Save(site);
                dbSession.Save(facility);

                section.AppointmentTypes.Add(appTypes[0]);
                section.AppointmentTypes.Add(appTypes[1]);

                dbSession.Save(section);

                sectionId = section.Id;

                Assert.IsTrue(sectionId != 0);
                transaction.Commit();
            }

            var original = service.Get(sectionId);

            var appTypeToDelete = original.AppointmentTypes[1];

            service.DeleteAppointmentType(sectionId, appTypeToDelete);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                var retrieved = service.Get(sectionId);

                Assert.AreEqual(retrieved.AppointmentTypes.Count, 1);

                dbSession.Delete(retrieved);
                dbSession.Delete(facility);
                dbSession.Delete(site);
                transaction.Commit();
            }
        }

    }
}
