﻿using MedRed.Services;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using MedRed.DataAccess;
using NHibernate;
using MedRed.MDWSAccess;
using Shared.Model.Config.MDWS;
using System.Collections.Generic;
using Shared.Model;
using KellermanSoftware.CompareNetObjects;
using MedRed.Services.Utils;

namespace MedRed.Services.Tests
{
    [TestClass()]
    public class SchedulingPolicyTest
    {   
        private TestContext testContextInstance;

        /// <summary>
        ///Gets or sets the test context which provides
        ///information about and functionality for the current test run.
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        #region Additional test attributes
        // 
        //You can use the following additional attributes as you write your tests:
        //
        //Use ClassInitialize to run code before running the first test in the class
        [ClassInitialize()]
        public static void MyClassInitialize(TestContext testContext)
        {            
        }
        
        //Use ClassCleanup to run code after all tests in a class have run
        [ClassCleanup()]
        public static void MyClassCleanup()
        {
        }
        
        //Use TestInitialize to run code before running each test
        [TestInitialize()]
        public void MyTestInitialize()
        {
            MDWSConfiguration config = ConnectionHelper.GetTestingConfig();
            var factory = Factory.GetFactory(config);

            TestContext.Properties["Factory"] = factory;

            var service = factory.GetResourceService();

            Assert.IsNotNull(service);
            Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
            Facility facility = new Facility() { Name = "Facility 1", Site = site };
            Section section = new Section() { Name = "Section A", Facility = facility };
            Resource resource = new Resource() { Name = "Dr. Jones", Section = section, Type = ResourceType.Provider };

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(site);
                dbSession.Save(facility);
                dbSession.Save(section);
                dbSession.Save(resource);

                transaction.Commit();
            }

            int resourceId = resource.Id;
            Assert.IsTrue(resourceId != 0);

            Resource retrievedResource = service.Get(resourceId);

            TestContext.Properties["site"] = site;
            TestContext.Properties["facility"] = facility;
            TestContext.Properties["section"] = section;
            TestContext.Properties["resource"] = retrievedResource;

        }
        
        //Use TestCleanup to run code after each test has run
        [TestCleanup()]
        public void MyTestCleanup()
        {
            var site = TestContext.Properties["site"] as Site;
            var facility = TestContext.Properties["facility"] as Facility;
            var section = TestContext.Properties["section"] as Section;
            var resource = TestContext.Properties["resource"] as Resource;

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(resource);
                dbSession.Delete(section);
                dbSession.Delete(facility);
                dbSession.Delete(site);

                transaction.Commit();
            }
        }
        
        #endregion

        [TestMethod()]
        public void SchedulingPolicyTest1()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetResourceService();
            var retrievedResource = TestContext.Properties["resource"] as Resource;

            SchedulingPolicy monday = new SchedulingPolicy(){ DayOfWeek = 1, StartEffectDate=new DateTime(2013, 1, 1), Priority = 1};
            SchedulingHelper.AddSlotsToPolicy(ref monday, new TimeSpan(8, 0, 0), new TimeSpan(10, 0, 0), 60);

            SchedulingPolicy friday = new SchedulingPolicy(){ DayOfWeek = 5, StartEffectDate=new DateTime(2013, 1, 1), Priority = 1};
            SchedulingHelper.AddSlotsToPolicy(ref friday, new TimeSpan(8, 0, 0), new TimeSpan(10, 0, 0), 60);

            SchedulingPolicy tuesday = new SchedulingPolicy(){ DayOfWeek = 2, StartEffectDate=new DateTime(2013, 1, 1), Priority = 1};
            SchedulingHelper.AddSlotsToPolicy(ref tuesday, new TimeSpan(8, 0, 0), new TimeSpan(9, 0, 0), 60);

            SchedulingPolicy thursday = new SchedulingPolicy(){ DayOfWeek = 4, StartEffectDate=new DateTime(2013, 1, 1), Priority = 1};
            SchedulingHelper.AddSlotsToPolicy(ref thursday, new TimeSpan(8, 0, 0), new TimeSpan(9, 0, 0), 60);

            retrievedResource.AddShedulingPolicy(monday);
            retrievedResource.AddShedulingPolicy(friday);
            retrievedResource.AddShedulingPolicy(tuesday);
            retrievedResource.AddShedulingPolicy(thursday);

            service.Update(retrievedResource);

            DateTime apptRequest_Avail = new DateTime(2013, 4, 15, 9, 0, 0);
            DateTime apptRequest_NotAvail = new DateTime(2013, 4, 15, 10, 0, 0);

            var capacity = service.GetCapacity(retrievedResource.Id, apptRequest_Avail.Date, apptRequest_Avail.Date);

            Assert.IsTrue(SchedulingHelper.IsTimeAvailable(capacity, apptRequest_Avail));
            Assert.IsFalse(SchedulingHelper.IsTimeAvailable(capacity, apptRequest_NotAvail));
        }
    }
}
