﻿using MedRed.Services;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using MedRed.DataAccess;
using NHibernate;
using System.Linq;
using MedRed.MDWSAccess;
using Shared.Model.Config.MDWS;
using System.Collections.Generic;
using Shared.Model;
using KellermanSoftware.CompareNetObjects;
using MedRed.Services.Utils;
using MedRed.Services.Interfaces;

namespace MedRed.Services.Tests
{
    [TestClass()]
    public class SchedulingPolicyHoldBlockTest
    {   
        private TestContext testContextInstance;

        /// <summary>
        ///Gets or sets the test context which provides
        ///information about and functionality for the current test run.
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        #region Additional test attributes
        // 
        //You can use the following additional attributes as you write your tests:
        //
        //Use ClassInitialize to run code before running the first test in the class
        [ClassInitialize()]
        public static void MyClassInitialize(TestContext testContext)
        {            
        }
        
        //Use ClassCleanup to run code after all tests in a class have run
        [ClassCleanup()]
        public static void MyClassCleanup()
        {
        }
        
        //Use TestInitialize to run code before running each test
        [TestInitialize()]
        public void MyTestInitialize()
        {
            MDWSConfiguration config = ConnectionHelper.GetTestingConfig();
            var factory = Factory.GetFactory(config);

            TestContext.Properties["Factory"] = factory;

            var service = factory.GetResourceService();

            var site = factory.GetSiteService().GetFromVistaSiteId("555");

            Assert.IsNotNull(service);
//            Site site = new Site() { VistaSiteId = "555", Name = "Test Site 2" };
            Facility facility = new Facility() { Name = "Facility 1", Site = site };
            Section section = new Section() { Name = "Section A", Facility = facility, DSSPrimaryStopCode = "368"  };

            var providers = factory.GetProviderService().GetAllForSite(site.Id);

            Assert.IsTrue(providers.Count > 0);
            var provider = providers.First();

            Resource resource = new Resource() { Name = "Dr. Jones", Section = section, Type = ResourceType.Provider, FullfillingResourceId = provider.Id };

            var patSearchResults = factory.GetPatientService().Search("555", "");
            Assert.IsTrue(patSearchResults.Count > 0);

            var patSearchResult = patSearchResults.First();
            var patient = factory.GetPatientService().Get(patSearchResult);

            var apptTypeCategores = factory.GetNationalSystemService().GetAppointmentTypeCategories("555");
            Assert.IsTrue(apptTypeCategores.Count > 0);
            var appType = new AppointmentType() { AppointmentTypeCategory = apptTypeCategores[0], Name = "Initial", Duration = 30 };

            section.AddAppointmentType(appType);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
//                dbSession.Save(site);
                dbSession.Save(facility);
                dbSession.Save(section);
//                dbSession.Save(patient);

                transaction.Commit();
            }

            service.Add(resource);

            int resourceId = resource.Id;
            Assert.IsTrue(resourceId != 0);

            Resource retrievedResource = service.Get(resourceId);

            SchedulingPolicy monday = new SchedulingPolicy() { DayOfWeek = 1, StartEffectDate = new DateTime(2013, 1, 1), Priority = 1 };
            SchedulingHelper.AddSlotsToPolicy(ref monday, new TimeSpan(8, 0, 0), new TimeSpan(10, 0, 0), 60);

            SchedulingPolicy friday = new SchedulingPolicy() { DayOfWeek = 5, StartEffectDate = new DateTime(2013, 1, 1), Priority = 1 };
            SchedulingHelper.AddSlotsToPolicy(ref friday, new TimeSpan(8, 0, 0), new TimeSpan(10, 0, 0), 60);
            
            SchedulingPolicy wednesday = new SchedulingPolicy() { DayOfWeek = 3, StartEffectDate = new DateTime(2013, 1, 1), Priority = 1 };
            SchedulingHelper.AddSlotsToPolicy(ref wednesday, new TimeSpan(8, 0, 0), new TimeSpan(9, 0, 0), 60);

            SchedulingPolicy tuesday = new SchedulingPolicy() { DayOfWeek = 2, StartEffectDate = new DateTime(2013, 1, 1), Priority = 1 };
            SchedulingHelper.AddSlotsToPolicy(ref tuesday, new TimeSpan(8, 0, 0), new TimeSpan(9, 0, 0), 60);

            SchedulingPolicy thursday = new SchedulingPolicy() { DayOfWeek = 4, StartEffectDate = new DateTime(2013, 1, 1), Priority = 1 };
            SchedulingHelper.AddSlotsToPolicy(ref thursday, new TimeSpan(8, 0, 0), new TimeSpan(9, 0, 0), 60);

            retrievedResource.AddShedulingPolicy(monday);
            retrievedResource.AddShedulingPolicy(friday);
            retrievedResource.AddShedulingPolicy(wednesday);
            retrievedResource.AddShedulingPolicy(tuesday);
            retrievedResource.AddShedulingPolicy(thursday);

            retrievedResource = service.Update(retrievedResource);

            TestContext.Properties["site"] = site;
            TestContext.Properties["facility"] = facility;
            TestContext.Properties["section"] = section;
            TestContext.Properties["resource"] = retrievedResource;
            TestContext.Properties["Patient"] = patient;
        }
        
        //Use TestCleanup to run code after each test has run
        [TestCleanup()]
        public void MyTestCleanup()
        {
//            var site = TestContext.Properties["site"] as Site;
            var facility = TestContext.Properties["facility"] as Facility;
            var section = TestContext.Properties["section"] as Section;
            var resource = TestContext.Properties["resource"] as Resource;
//            var patient = TestContext.Properties["Patient"] as Patient;

            var factory = TestContext.Properties["Factory"] as Factory;

            resource = factory.GetResourceService().Get(resource.Id);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
//                dbSession.Delete(patient);
                dbSession.Delete(resource);
                dbSession.Delete(section);
                dbSession.Delete(facility);
//                dbSession.Delete(site);

                transaction.Commit();
            }
        }
        
        #endregion

        [TestMethod()]
        public void SchedulingPolicyAddBlock()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetResourceService();
            var retrievedResource = TestContext.Properties["resource"] as Resource;

            DateTime apptRequest_Avail = new DateTime(2013, 4, 15, 9, 0, 0);
            DateTime apptRequest_NotAvail = new DateTime(2013, 4, 15, 10, 0, 0);

            var capacity = service.GetCapacity(retrievedResource.Id, apptRequest_Avail.Date, apptRequest_Avail.Date);

            Assert.IsTrue(SchedulingHelper.IsTimeAvailable(capacity, apptRequest_Avail));
            Assert.IsFalse(SchedulingHelper.IsTimeAvailable(capacity, apptRequest_NotAvail));

            retrievedResource.AddBlock(new Block() { 
                PeriodStart = new DateTime(2013, 4, 15, 8, 30, 0), 
                PeriodEnd = new DateTime(2013, 4, 15, 9, 30, 0), 
                Reason = "Office meeting" });

            service.Update(retrievedResource);

            capacity = service.GetCapacity(retrievedResource.Id, apptRequest_Avail.Date, apptRequest_Avail.Date);

            Assert.IsFalse(SchedulingHelper.IsTimeAvailable(capacity, apptRequest_Avail));
            Assert.IsFalse(SchedulingHelper.IsTimeAvailable(capacity, apptRequest_NotAvail));
        }

        [TestMethod()]
        public void SchedulingPolicyAddRecurringDailyBlock()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetResourceService();
            var retrievedResource = TestContext.Properties["resource"] as Resource;

            DateTime apptRequest_NotAvail = new DateTime(2013, 4, 17, 8, 0, 0);
            DateTime apptRequest_Avail = new DateTime(2013, 4, 18, 8, 0, 0);

            var capacity = service.GetCapacity(retrievedResource.Id, new DateTime(2013, 04, 17), new DateTime(2013, 04, 18));

            Assert.IsTrue(SchedulingHelper.IsTimeAvailable(capacity, apptRequest_Avail));
            Assert.IsTrue(SchedulingHelper.IsTimeAvailable(capacity, apptRequest_NotAvail));

            retrievedResource.AddBlock(new Block()
            {
                PeriodStart = new DateTime(2013, 4, 15, 7, 30, 0),
                PeriodEnd = new DateTime(2013, 4, 15, 8, 30, 0),
                Reason = "Office meeting",
                Recur = new Recur()
                {
                    RecurringStart = new DateTime(2013, 4, 15),
                    RecurringEnd = new DateTime(2013, 5, 15), 
                    RecurringFrequency = "Daily",
                    RecurringInterval = "2"
                }
            });

            service.Update(retrievedResource);

            capacity = service.GetCapacity(retrievedResource.Id, new DateTime(2013, 04, 17), new DateTime(2013, 04, 18));

            Assert.IsTrue(SchedulingHelper.IsTimeAvailable(capacity, apptRequest_Avail));
            Assert.IsFalse(SchedulingHelper.IsTimeAvailable(capacity, apptRequest_NotAvail));
        }

        [TestMethod()]
        public void SchedulingPolicyAddRecurringWeeklyBlock()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetResourceService();
            var retrievedResource = TestContext.Properties["resource"] as Resource;

            DateTime apptRequest_NotAvail = new DateTime(2013, 4, 29, 9, 0, 0);
            DateTime apptRequest_Avail = new DateTime(2013, 4, 22, 9, 0, 0);

            var capacity = service.GetCapacity(retrievedResource.Id, new DateTime(2013, 4, 22), new DateTime(2013, 4, 29));

            Assert.IsTrue(SchedulingHelper.IsTimeAvailable(capacity, apptRequest_Avail));
            Assert.IsTrue(SchedulingHelper.IsTimeAvailable(capacity, apptRequest_NotAvail));

            retrievedResource.AddBlock(new Block()
            {
                PeriodStart = new DateTime(2013, 4, 15, 8, 30, 0),
                PeriodEnd = new DateTime(2013, 4, 15, 9, 30, 0),
                Reason = "Office meeting",
                Recur = new Recur()
                {
                    RecurringStart = new DateTime(2013, 4, 15),
                    RecurringEnd = new DateTime(2013, 5, 15),
                    RecurringFrequency = "Weekly",
                    RecurringInterval = "2"
                }
            });

            service.Update(retrievedResource);

            capacity = service.GetCapacity(retrievedResource.Id, new DateTime(2013, 4, 22), new DateTime(2013, 4, 29));

            Assert.IsTrue(SchedulingHelper.IsTimeAvailable(capacity, apptRequest_Avail));
            Assert.IsFalse(SchedulingHelper.IsTimeAvailable(capacity, apptRequest_NotAvail));
        }


        [TestMethod()]
        public void SchedulingPolicyAddHold()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetResourceService();
            var retrievedResource = TestContext.Properties["resource"] as Resource;

            DateTime apptRequest_Avail = new DateTime(2013, 4, 15, 9, 0, 0);
            DateTime apptRequest_NotAvail = new DateTime(2013, 4, 15, 10, 0, 0);

            var capacity = service.GetCapacity(retrievedResource.Id, apptRequest_Avail.Date, apptRequest_Avail.Date);

            Assert.IsTrue(SchedulingHelper.IsTimeAvailable(capacity, apptRequest_Avail));
            Assert.IsFalse(SchedulingHelper.IsTimeAvailable(capacity, apptRequest_NotAvail));

            retrievedResource.AddHold(new Hold()
            {
                PeriodStart = new DateTime(2013, 4, 15, 8, 30, 0),
                PeriodEnd = new DateTime(2013, 4, 15, 9, 30, 0)                
            });

            service.Update(retrievedResource);

            capacity = service.GetCapacity(retrievedResource.Id, apptRequest_Avail.Date, apptRequest_Avail.Date);

            Assert.IsFalse(SchedulingHelper.IsTimeAvailable(capacity, apptRequest_Avail));
            Assert.IsFalse(SchedulingHelper.IsTimeAvailable(capacity, apptRequest_NotAvail));
        }

        [TestMethod()]
        public void SchedulingPolicyCapacityTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetResourceService();
            var retrievedResource = TestContext.Properties["resource"] as Resource;
            var patient = TestContext.Properties["Patient"] as Patient;
            var section = TestContext.Properties["section"] as Section;

            DateTime apptRequest_Avail = new DateTime(2013, 4, 22, 9, 0, 0);
            DateTime apptRequest_NotAvail = new DateTime(2013, 4, 22, 10, 0, 0);

            var capacity = service.GetAvailability(retrievedResource.Id, apptRequest_Avail.Date, apptRequest_Avail.Date);

            DateTime apptTime = capacity[apptRequest_Avail.Date][0].Time;
            int firstSlotCapacity = capacity[apptRequest_Avail.Date][0].Remaining;

            var appt = new Appointment()
            {
                Time = apptTime,
                Length = 60,
                Patient = patient,
                Section = section,
//                Status = AppointmentStatus.Scheduled,
                AppointmentType = section.AppointmentTypes[0]
            };
            appt.AddResource(retrievedResource);

            factory.GetAppointmentService().CreateAppointment(appt);

            capacity = service.GetAvailability(retrievedResource.Id, apptRequest_Avail.Date, apptRequest_Avail.Date);

            int secondSlotCapacity = capacity[apptRequest_Avail.Date][0].Remaining;

            Assert.AreEqual(firstSlotCapacity - 1, secondSlotCapacity);

            factory.GetAppointmentService().Delete(appt.Id);

            //using (var dbSession = DataAccess.DataAccess.OpenSession())
            //using (var transaction = dbSession.BeginTransaction())
            //{
            //    dbSession.Delete(appt);

            //    transaction.Commit();
            //}

        }
    }
}
