﻿using MedRed.Services;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using MedRed.DataAccess;
using NHibernate;
using MedRed.MDWSAccess;
using Shared.Model.Config.MDWS;
using System.Collections.Generic;
using Shared.Model;
using KellermanSoftware.CompareNetObjects;

namespace MedRed.Services.Tests
{
    [TestClass()]
    public class RoomServiceTest
    {   
        private TestContext testContextInstance;

        /// <summary>
        ///Gets or sets the test context which provides
        ///information about and functionality for the current test run.
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        #region Additional test attributes
        // 
        //You can use the following additional attributes as you write your tests:
        //
        //Use ClassInitialize to run code before running the first test in the class
        [ClassInitialize()]
        public static void MyClassInitialize(TestContext testContext)
        {
            log4net.Config.XmlConfigurator.Configure();
        }
        
        //Use ClassCleanup to run code after all tests in a class have run
        [ClassCleanup()]
        public static void MyClassCleanup()
        {
        }
        
        //Use TestInitialize to run code before running each test
        [TestInitialize()]
        public void MyTestInitialize()
        {
            MDWSConfiguration config = ConnectionHelper.GetTestingConfig();
            var factory = Factory.GetFactory(config);

            TestContext.Properties["Factory"] = factory;
        }
        
        //Use TestCleanup to run code after each test has run
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion


        [TestMethod()]
        public void getAllRoomsTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetRoomService();

            Assert.IsNotNull(service);

            Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };

            Room[] array = {new Room() { Name="100a", Floor="1", Site = site}, 
                            new Room() { Name="100b", Floor="1", Site = site}, 
                            new Room() { Name="200", Floor="2", Site = site}, 
                            };

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction()) 
            {
                dbSession.Save(site);
                dbSession.Save(array[0]);
                dbSession.Save(array[1]);
                dbSession.Save(array[2]);
                transaction.Commit();
            }

            var rooms = service.GetAllForSite(site.Id);
            var roomList = new List<Room>(rooms);

            Assert.IsTrue(roomList.Count >= 3);
            Assert.IsTrue(roomList.Exists(s => s.Name == "100a"));
            Assert.IsTrue(roomList.Exists(s => s.Name == "200"));

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {

                dbSession.Delete(array[0]);
                dbSession.Delete(array[1]);
                dbSession.Delete(array[2]);
                dbSession.Delete(site);

                transaction.Commit();
            }
        }

        [TestMethod()]
        public void getRoomTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetRoomService();

            Assert.IsNotNull(service);

            Room[] array = {new Room() { Name="100a", Floor="1"}, 
                            new Room() { Name="100b", Floor="1"}, 
                            new Room() { Name="200", Floor="2"}, 
                            };


            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(array[0]);
                dbSession.Save(array[1]);
                dbSession.Save(array[2]);
                transaction.Commit();
            }
            int roomId = array[0].Id;
            Assert.IsTrue(roomId != 0);

            var retrievedRoom = service.Get(roomId);

            CompareObjects compareObjects = new CompareObjects();
            Assert.IsTrue(compareObjects.Compare(retrievedRoom, array[0]), compareObjects.DifferencesString);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {

                dbSession.Delete(array[0]);
                dbSession.Delete(array[1]);
                dbSession.Delete(array[2]);
                transaction.Commit();
            }
        }

        [TestMethod()]
        public void addRoomTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetRoomService();

            Assert.IsNotNull(service);

            Room[] array = {new Room() { Name="100a", Floor="1"}, 
                        new Room() { Name="100b", Floor="1"}, 
                        new Room() { Name="200", Floor="2"}, 
                        };

            Assert.IsTrue(array[0].Id == 0);

            var room = service.Add(array[0]);
            int roomId = room.Id;

            var retrievedRoom = service.Get(roomId);

            CompareObjects compareObjects = new CompareObjects();
            Assert.IsTrue(compareObjects.Compare(retrievedRoom, array[0]), compareObjects.DifferencesString);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {

                dbSession.Delete(array[0]);
                transaction.Commit();
            }
        }        
    }
}
