using MedRed.Services;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using MedRed.DataAccess;
using NHibernate;
using MedRed.MDWSAccess;
using Shared.Model.Config.MDWS;
using System.Collections.Generic;
using Shared.Model;
using KellermanSoftware.CompareNetObjects;

namespace MedRed.Services.Tests
{
    [TestClass()]
    public class ResourceServiceTest
    {   
        private TestContext testContextInstance;

        /// <summary>
        ///Gets or sets the test context which provides
        ///information about and functionality for the current test run.
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        #region Additional test attributes
        // 
        //You can use the following additional attributes as you write your tests:
        //
        //Use ClassInitialize to run code before running the first test in the class
        [ClassInitialize()]
        public static void MyClassInitialize(TestContext testContext)
        {            
        }
        
        //Use ClassCleanup to run code after all tests in a class have run
        [ClassCleanup()]
        public static void MyClassCleanup()
        {
        }
        
        //Use TestInitialize to run code before running each test
        [TestInitialize()]
        public void MyTestInitialize()
        {
            MDWSConfiguration config = ConnectionHelper.GetTestingConfig();
            var factory = Factory.GetFactory(config);

            TestContext.Properties["Factory"] = factory;

            var hierarchy = (log4net.Repository.Hierarchy.Hierarchy)log4net.LogManager.GetRepository();
            var logger = (log4net.Repository.Hierarchy.Logger)hierarchy.GetLogger("NHibernate.SQL");
            logger.AddAppender(new log4net.Appender.TraceAppender { Layout = new log4net.Layout.SimpleLayout() });
            hierarchy.Configured = true;

        }
        
        //Use TestCleanup to run code after each test has run
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion


        [TestMethod()]
        public void getAllResourcesTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetResourceService();

            Assert.IsNotNull(service);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            {
                Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
                dbSession.Save(site);
                Facility facility = new Facility() { Name = "Facility 1", Site = site };
                dbSession.Save(facility);
                Section section = new Section() { Name = "Section A", Facility = facility };
                dbSession.Save(section);
                int sectionId = section.Id;

                Resource[] array = {new Resource() { Name = "Dr. Jones", Section = section, Type = ResourceType.Provider},
                                new Resource() { Name = "Room 12", Section = section, Type = ResourceType.Room}};

                dbSession.Save(array[0]);
                dbSession.Save(array[1]);

                var resources = service.GetAll(sectionId);
                var resourceList = new List<Resource>(resources);

                Assert.IsTrue(resourceList.Count >= 2);
                Assert.IsTrue(resourceList.Exists(s => s.Name == "Dr. Jones"));
                Assert.IsTrue(resourceList.Exists(s => s.Name == "Room 12"));

                dbSession.Delete(array[0]);
                dbSession.Delete(array[1]);
                dbSession.Delete(section);
                dbSession.Delete(facility);
                dbSession.Delete(site);
            }
        }

        [TestMethod()]
        public void getResourceTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetResourceService();

            Assert.IsNotNull(service);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            {
                Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
                dbSession.Save(site);
                Facility facility = new Facility() { Name = "Facility 1", Site = site };
                dbSession.Save(facility);
                Section section = new Section() { Name = "Section A", Facility = facility };
                dbSession.Save(section);
                int sectionId = section.Id;

                Resource[] array = {new Resource() { Name = "Dr. Jones", Section = section, Type = ResourceType.Provider},
                                new Resource() { Name = "Room 12", Section = section, Type = ResourceType.Room}};

                dbSession.Save(array[0]);
                dbSession.Save(array[1]);

                int resourceId = array[0].Id;
                Assert.IsTrue(resourceId != 0);

                var retrievedResource = service.Get(resourceId);

                CompareObjects compareObjects = new CompareObjects();
                Assert.IsTrue(compareObjects.Compare(retrievedResource, array[0]), compareObjects.DifferencesString);

                dbSession.Delete(array[0]);
                dbSession.Delete(array[1]);
                dbSession.Delete(section);
                dbSession.Delete(facility);
                dbSession.Delete(site);
            }
        }

        [TestMethod()]
        public void addResourceTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetResourceService();

            Assert.IsNotNull(service);
            Site site = Utils.CreateAndSaveSite();
            Facility facility = Utils.CreateAndSaveFacility(site);
            Section section = Utils.CreateAndSaveSection(facility);

            Provider p = new Provider() {Person = new Person() { FirstName = "Henry", LastName = "Jones" } };
            p.AddProviderSite("555", "11298");

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
//                dbSession.Save(site);
//                dbSession.Save(facility);
//                dbSession.Save(section);

                dbSession.Save(p);
                transaction.Commit();
            }

            int sectionId = section.Id;

            Resource[] array = { new Resource() { Name = "Dr. Jones", Section = section, Type = ResourceType.Provider, FullfillingResourceId = p.Id} };

            Assert.IsTrue(array[0].Id == 0);

            var resource = service.Add(array[0]);
            int resourceId = resource.Id;

            var retrievedResource = service.Get(resourceId);

            CompareObjects compareObjects = new CompareObjects();
            Assert.IsTrue(compareObjects.Compare(retrievedResource, array[0]), compareObjects.DifferencesString);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(array[0]);
                dbSession.Delete(section);
                dbSession.Delete(facility);
                dbSession.Delete(site);
                transaction.Commit();
            }
        }

        [TestMethod()]
        public void addHoldTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetResourceService();

            Assert.IsNotNull(service);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            {
                Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
                dbSession.Save(site);
                Facility facility = new Facility() { Name = "Facility 1", Site = site };
                dbSession.Save(facility);
                Section section = new Section() { Name = "Section A", Facility = facility };
                dbSession.Save(section);
                int sectionId = section.Id;

                Resource[] array = {new Resource() { Name = "Dr. Jones", Section = section, Type = ResourceType.Provider},
                                new Resource() { Name = "Room 12", Section = section, Type = ResourceType.Room}};

                dbSession.Save(array[0]);
                dbSession.Save(array[1]);
                dbSession.Clear();

                int resourceId = array[0].Id;
                Assert.IsTrue(resourceId != 0);

                var retrievedResource = service.Get(resourceId);

                Hold newHold = new Hold()
                {
//                    Resource = retrievedResource,
                    PeriodStart = new DateTime(2013, 04, 20),
                    PeriodEnd = new DateTime(2013, 04, 30),
                    AutoReleaseAt = new DateTime(2013, 04, 18)
                };

//                retrievedResource.Holds.Add(newHold);
                retrievedResource.AddHold(newHold);

                service.Update(retrievedResource);

                var newRetrievedResource = service.Get(resourceId);


                CompareObjects compareObjects = new CompareObjects();
                Assert.IsTrue(compareObjects.Compare(retrievedResource, newRetrievedResource), compareObjects.DifferencesString);

                dbSession.Delete(array[0]);
                dbSession.Delete(array[1]);
                dbSession.Delete(section);
                dbSession.Delete(facility);
                dbSession.Delete(site);
            }
        }

        [TestMethod()]
        public void addBlockTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetResourceService();

            Assert.IsNotNull(service);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            {
                Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
                dbSession.Save(site);
                Facility facility = new Facility() { Name = "Facility 1", Site = site };
                dbSession.Save(facility);
                Section section = new Section() { Name = "Section A", Facility = facility };
                dbSession.Save(section);
                int sectionId = section.Id;

                Resource[] array = {new Resource() { Name = "Dr. Jones", Section = section, Type = ResourceType.Provider},
                                new Resource() { Name = "Room 12", Section = section, Type = ResourceType.Room}};

                dbSession.Save(array[0]);
                dbSession.Save(array[1]);
                dbSession.Clear();

                int resourceId = array[0].Id;
                Assert.IsTrue(resourceId != 0);

                var retrievedResource = service.Get(resourceId);

                Block newBlock = new Block()
                {
                    PeriodStart = new DateTime(2013, 04, 20),
                    PeriodEnd = new DateTime(2013, 04, 30),
                    Reason = "Staff Vacation"                      
                };
                Block newBlock2 = new Block()
                {
                    PeriodStart = new DateTime(2013, 05, 2, 9, 0, 0),
                    PeriodEnd = new DateTime(2013, 05, 2, 12, 30, 0),
                    Reason = "Staff Meeting"
                };


                //                retrievedResource.Holds.Add(newHold);
                retrievedResource.AddBlock(newBlock);
                retrievedResource.AddBlock(newBlock2);

                service.Update(retrievedResource);

                var newRetrievedResource = service.Get(resourceId);


                CompareObjects compareObjects = new CompareObjects();
                Assert.IsTrue(compareObjects.Compare(retrievedResource, newRetrievedResource), compareObjects.DifferencesString);

                dbSession.Delete(array[0]);
                dbSession.Delete(array[1]);
                dbSession.Delete(section);
                dbSession.Delete(facility);
                dbSession.Delete(site);
            }
        }

        [TestMethod()]
        public void addRecurringBlockTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetResourceService();

            Assert.IsNotNull(service);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            {
                Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
                dbSession.Save(site);
                Facility facility = new Facility() { Name = "Facility 1", Site = site };
                dbSession.Save(facility);
                Section section = new Section() { Name = "Section A", Facility = facility };
                dbSession.Save(section);
                int sectionId = section.Id;

                Resource[] array = {new Resource() { Name = "Dr. Jones", Section = section, Type = ResourceType.Provider},
                                new Resource() { Name = "Room 12", Section = section, Type = ResourceType.Room}};

                dbSession.Save(array[0]);
                dbSession.Save(array[1]);
                dbSession.Clear();

                int resourceId = array[0].Id;
                Assert.IsTrue(resourceId != 0);

                var retrievedResource = service.Get(resourceId);

                Block newBlock = new Block()
                {
                    PeriodStart = new DateTime(2013, 4, 20, 9, 0, 0),
                    PeriodEnd = new DateTime(2013, 4, 20, 13, 0, 0),
                    Reason = "Staff Meeting",
                    Recur = new Recur()
                    {
                        RecurringStart = new DateTime(2013, 4, 20),
                        RecurringEnd = new DateTime(2014, 4, 20),
                        RecurringFrequency = "Frequency",
                        RecurringInterval = "Interval"
                    }
                };

                //                retrievedResource.Holds.Add(newHold);
                retrievedResource.AddBlock(newBlock);

                service.Update(retrievedResource);

                var newRetrievedResource = service.Get(resourceId);


                CompareObjects compareObjects = new CompareObjects();
                Assert.IsTrue(compareObjects.Compare(retrievedResource, newRetrievedResource), compareObjects.DifferencesString);

                dbSession.Delete(array[0]);
                dbSession.Delete(array[1]);
                dbSession.Delete(section);
                dbSession.Delete(facility);
                dbSession.Delete(site);
            }
        }

        [TestMethod()]
        public void addScheduleTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetResourceService();

            Assert.IsNotNull(service);
            Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
            Facility facility = new Facility() { Name = "Facility 1", Site = site };
            Section section = new Section() { Name = "Section A", Facility = facility };
            Resource[] array = {new Resource() { Name = "Dr. Jones", Section = section, Type = ResourceType.Provider}};
//                            new Resource() { Name = "Room 12", Section = section, Type = ResourceType.Room}};

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(site);
                dbSession.Save(facility);
                dbSession.Save(section);
                int sectionId = section.Id;

                dbSession.Save(array[0]);
//                dbSession.Save(array[1]);                
               
                transaction.Commit();
            }

            int resourceId = array[0].Id;
            Assert.IsTrue(resourceId != 0);
            
            Resource retrievedResource = service.Get(resourceId);
            AppointmentTypeCategory apptTypeCat = new AppointmentTypeCategory() {Name = "Regular"};
            AppointmentType[] appTypes = {new AppointmentType() { Name="Initial", AppointmentTypeCategory = apptTypeCat},
                                            new AppointmentType() { Name="Followup", AppointmentTypeCategory = apptTypeCat},
                                            new AppointmentType() { Name="Walk In", AppointmentTypeCategory = apptTypeCat}};

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(apptTypeCat);

                dbSession.Save(appTypes[0]);
                dbSession.Save(appTypes[1]);
                dbSession.Save(appTypes[2]);
                transaction.Commit();
            }

            SchedulingPolicy[] policy = {new SchedulingPolicy(){ DayOfWeek = 2, StartEffectDate=new DateTime(2013, 1, 1), EndEffectDate = new DateTime(2015, 12, 31), Priority = 1},
                    new SchedulingPolicy(){ DayOfWeek = 3, StartEffectDate=new DateTime(2013, 1, 1), EndEffectDate = new DateTime(2015, 12, 31), Priority = 1},
                    new SchedulingPolicy(){ DayOfWeek = 4, StartEffectDate=new DateTime(2013, 1, 1), EndEffectDate = new DateTime(2015, 12, 31), Priority = 1}};

            policy[0].AddSlot(new Slot() { Time = new TimeSpan(9, 0, 0), Length = 60, AppointmentType = appTypes[0] });
            policy[0].AddSlot(new Slot() { Time = new TimeSpan(10, 0, 0), Length = 30, AppointmentType = appTypes[1] });
            policy[0].AddSlot(new Slot() { Time = new TimeSpan(10, 30, 0), Length = 30, AppointmentType = appTypes[1] });
            policy[0].AddSlot(new Slot() { Time = new TimeSpan(11, 0, 0), Length = 60, AppointmentType = appTypes[2] });
            policy[0].AddSlot(new Slot() { Time = new TimeSpan(12, 0, 0), Length = 60, AppointmentType = appTypes[2] });

            policy[1].AddSlot(new Slot() { Time = new TimeSpan(12, 0, 0), Length = 60, AppointmentType = appTypes[0] });
            policy[1].AddSlot(new Slot() { Time = new TimeSpan(13, 0, 0), Length = 60, AppointmentType = appTypes[1] });
            policy[1].AddSlot(new Slot() { Time = new TimeSpan(14, 0, 0), Length = 30, AppointmentType = appTypes[1] });
            policy[1].AddSlot(new Slot() { Time = new TimeSpan(15, 0, 0), Length = 60, AppointmentType = appTypes[2] });
            policy[1].AddSlot(new Slot() { Time = new TimeSpan(16, 0, 0), Length = 60, AppointmentType = appTypes[2] });

            policy[2].AddSlot(new Slot() { Time = new TimeSpan(9, 0, 0), Length = 60, AppointmentType = appTypes[0] });
            policy[2].AddSlot(new Slot() { Time = new TimeSpan(11, 0, 0), Length = 30, AppointmentType = appTypes[1] });
            policy[2].AddSlot(new Slot() { Time = new TimeSpan(12, 30, 0), Length = 30, AppointmentType = appTypes[1] });
            policy[2].AddSlot(new Slot() { Time = new TimeSpan(14, 0, 0), Length = 60, AppointmentType = appTypes[2] });
            policy[2].AddSlot(new Slot() { Time = new TimeSpan(15, 0, 0), Length = 60, AppointmentType = appTypes[2] });
                
            retrievedResource.AddShedulingPolicy(policy[0]);
            retrievedResource.AddShedulingPolicy(policy[1]);
            retrievedResource.AddShedulingPolicy(policy[2]);
            
            service.Update(retrievedResource);

            var newRetrievedResource = service.Get(resourceId);

            CompareObjects compareObjects = new CompareObjects();
            Assert.IsTrue(compareObjects.Compare(retrievedResource, newRetrievedResource), compareObjects.DifferencesString);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(newRetrievedResource);
                dbSession.Delete(appTypes[0]);
                dbSession.Delete(appTypes[1]);
                dbSession.Delete(appTypes[2]);
                dbSession.Delete(apptTypeCat);
                dbSession.Delete(section);
                dbSession.Delete(facility);
                dbSession.Delete(site);

                transaction.Commit();
            }
        }

        [TestMethod()]
        public void getScheduleTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetResourceService();

            Assert.IsNotNull(service);
            Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
            Facility facility = new Facility() { Name = "Facility 1", Site = site };
            Section section = new Section() { Name = "Section A", Facility = facility };
            Resource[] array = {new Resource() { Name = "Dr. Jones", Section = section, Type = ResourceType.Provider},
                            new Resource() { Name = "Room 12", Section = section, Type = ResourceType.Room}};

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(site);
                dbSession.Save(facility);
                dbSession.Save(section);
                int sectionId = section.Id;


                dbSession.Save(array[0]);
                dbSession.Save(array[1]);

                transaction.Commit();
            }

            int resourceId = array[0].Id;
            Assert.IsTrue(resourceId != 0);

            Resource retrievedResource = service.Get(resourceId);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                AppointmentTypeCategory apptTypeCat = new AppointmentTypeCategory() { Name = "Regular" };
                dbSession.Save(apptTypeCat);

                AppointmentType[] appTypes = {new AppointmentType() { Name="Initial", AppointmentTypeCategory = apptTypeCat},
                                              new AppointmentType() { Name="Followup", AppointmentTypeCategory = apptTypeCat},
                                              new AppointmentType() { Name="Walk In", AppointmentTypeCategory = apptTypeCat}};

                dbSession.Save(appTypes[0]);
                dbSession.Save(appTypes[1]);
                dbSession.Save(appTypes[2]);

                SchedulingPolicy[] policy = {new SchedulingPolicy(){ DayOfWeek = 2, StartEffectDate=new DateTime(2013, 1, 1), EndEffectDate = new DateTime(2015, 12, 31), Priority = 1},
                     new SchedulingPolicy(){ DayOfWeek = 3, StartEffectDate=new DateTime(2013, 1, 1), EndEffectDate = new DateTime(2015, 12, 31), Priority = 1},
                     new SchedulingPolicy(){ DayOfWeek = 4, StartEffectDate=new DateTime(2013, 1, 1), EndEffectDate = new DateTime(2015, 12, 31), Priority = 1}};

                policy[0].AddSlot(new Slot() { Time = new TimeSpan(9, 0, 0), Length = 60, AppointmentType = appTypes[0] });
                policy[0].AddSlot(new Slot() { Time = new TimeSpan(10, 0, 0), Length = 30, AppointmentType = appTypes[1] });
                policy[0].AddSlot(new Slot() { Time = new TimeSpan(10, 30, 0), Length = 30, AppointmentType = appTypes[1] });
                policy[0].AddSlot(new Slot() { Time = new TimeSpan(11, 0, 0), Length = 60, AppointmentType = appTypes[2] });
                policy[0].AddSlot(new Slot() { Time = new TimeSpan(12, 0, 0), Length = 60, AppointmentType = appTypes[2] });

                policy[1].AddSlot(new Slot() { Time = new TimeSpan(12, 0, 0), Length = 60, AppointmentType = appTypes[0] });
                policy[1].AddSlot(new Slot() { Time = new TimeSpan(13, 0, 0), Length = 60, AppointmentType = appTypes[1] });
                policy[1].AddSlot(new Slot() { Time = new TimeSpan(14, 0, 0), Length = 30, AppointmentType = appTypes[1] });
                policy[1].AddSlot(new Slot() { Time = new TimeSpan(15, 0, 0), Length = 60, AppointmentType = appTypes[2] });
                policy[1].AddSlot(new Slot() { Time = new TimeSpan(16, 0, 0), Length = 60, AppointmentType = appTypes[2] });

                policy[2].AddSlot(new Slot() { Time = new TimeSpan(9, 0, 0), Length = 60, AppointmentType = appTypes[0] });
                policy[2].AddSlot(new Slot() { Time = new TimeSpan(11, 0, 0), Length = 30, AppointmentType = appTypes[1] });
                policy[2].AddSlot(new Slot() { Time = new TimeSpan(12, 30, 0), Length = 30, AppointmentType = appTypes[1] });
                policy[2].AddSlot(new Slot() { Time = new TimeSpan(14, 0, 0), Length = 60, AppointmentType = appTypes[2] });
                policy[2].AddSlot(new Slot() { Time = new TimeSpan(15, 0, 0), Length = 60, AppointmentType = appTypes[2] });

                retrievedResource.AddShedulingPolicy(policy[0]);
                retrievedResource.AddShedulingPolicy(policy[1]);
                retrievedResource.AddShedulingPolicy(policy[2]);

                transaction.Commit();
            }

            service.Update(retrievedResource);

            var capacity = service.GetCapacity(resourceId, new DateTime(2013, 4, 1), new DateTime(2013, 4, 30));



            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(array[0]);
                dbSession.Delete(array[1]);
                dbSession.Delete(section);
                dbSession.Delete(facility);
                dbSession.Delete(site);
            }
        }

        [TestMethod()]
        public void SetResourceInactiveTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetResourceService();

            Assert.IsNotNull(service);

            Site site = Utils.CreateAndSaveSite();
            Facility facility = Utils.CreateAndSaveFacility(site);
            Section section = Utils.CreateAndSaveSection(facility);
            Provider p = new Provider() { Person = new Person() { FirstName = "Henry", LastName = "Jones" } };
            p.AddProviderSite("555", "11298");

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                //dbSession.Save(site);
                //dbSession.Save(facility);
                //dbSession.Save(section);
                dbSession.Clear();


                dbSession.Save(p);

                transaction.Commit();
            }

            int sectionId = section.Id;

            Resource[] array = { new Resource() { Name = "Dr. Jones", Section = section, Type = ResourceType.Provider, FullfillingResourceId = p.Id } };
                                
            Assert.IsTrue(array[0].Id == 0);

            var resource = service.Add(array[0]);
            int resourceId = resource.Id;

            var allResources = factory.GetResourceService().GetAll(sectionId);
            Assert.AreEqual(1, allResources.Count);

            // deactivate the Section
            section.Active = false;
            factory.GetSectionService().Update(section);

            // see if clinic is deactivated in Vista
            MDWSCalls mdws = new MDWSCalls();
            mdws.setConfig(Factory.Configuration);
            mdws.setConnection(factory.GetVistaServiceAccountForSite("555"));

            var fields = mdws.VistaQuery_SpecificRecord("44", resource.VistaClinicIEN, "*");

            Assert.IsTrue(fields.ContainsKey("2505"));
            Assert.IsTrue(!String.IsNullOrEmpty(fields["2505"]));


            // reactive the Section
            section.Active = true;
            factory.GetSectionService().Update(section);

            // see if clinic is deactivated in Vista
            fields = mdws.VistaQuery_SpecificRecord("44", resource.VistaClinicIEN, "*");

            Assert.IsTrue(fields.ContainsKey("2505"));
            Assert.IsTrue(String.IsNullOrEmpty(fields["2505"]));

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(array[0]);
                dbSession.Delete(section);
                dbSession.Delete(facility);
                dbSession.Delete(site);
                transaction.Commit();
            }
        }

    }
}
