﻿using MedRed.Services;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using MedRed.DataAccess;
using NHibernate;
using MedRed.MDWSAccess;
using Shared.Model.Config.MDWS;
using System.Collections.Generic;
using Shared.Model;
using KellermanSoftware.CompareNetObjects;
using MedRed.Services.Utils;
using System.Collections.ObjectModel;

namespace MedRed.Services.Tests
{
    [TestClass()]
    public class ResourceAvailabilityTest
    {   
        private TestContext testContextInstance;

        /// <summary>
        ///Gets or sets the test context which provides
        ///information about and functionality for the current test run.
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        #region Additional test attributes
        // 
        //You can use the following additional attributes as you write your tests:
        //
        //Use ClassInitialize to run code before running the first test in the class
        [ClassInitialize()]
        public static void MyClassInitialize(TestContext testContext)
        {            
        }
        
        //Use ClassCleanup to run code after all tests in a class have run
        [ClassCleanup()]
        public static void MyClassCleanup()
        {
        }
        
        //Use TestInitialize to run code before running each test
        [TestInitialize()]
        public void MyTestInitialize()
        {
            MDWSConfiguration config = ConnectionHelper.GetTestingConfig();
            var factory = Factory.GetFactory(config);

            TestContext.Properties["Factory"] = factory;

            var service = factory.GetResourceService();

            Assert.IsNotNull(service);
            Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
            Facility facility = new Facility() { Name = "Facility 1", Site = site };
            Section section = new Section() { Name = "Section A", Facility = facility };
            Resource resource = new Resource() { Name = "Dr. Jones", Section = section, Type = ResourceType.Provider };
            Resource resource2 = new Resource() { Name = "Room 3", Section = section, Type = ResourceType.Room };

            AppointmentTypeCategory cat = new AppointmentTypeCategory(){ Name = "testabc"};

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(site);
                dbSession.Save(facility);
                dbSession.Save(section);
                dbSession.Save(resource);
                dbSession.Save(resource2);
                dbSession.Save(cat);

                transaction.Commit();
            }

            AppointmentType type = new AppointmentType() { Name = "test", DSSCreditStopCode = "323",
                                                           AppointmentTypeCategory = cat };
            AppointmentType type2 = new AppointmentType() { Name = "test2", DSSCreditStopCode = "434",
                                                            AppointmentTypeCategory = cat };

            factory.GetSectionService().AddOrUpdateAppointmentType(section.Id, type);
            factory.GetSectionService().AddOrUpdateAppointmentType(section.Id, type2);

            section = factory.GetSectionService().Get(section.Id);

            int resourceId = resource.Id;
            Assert.IsTrue(resourceId != 0);

            Resource retrievedResource = service.Get(resourceId);

            TestContext.Properties["site"] = site;
            TestContext.Properties["facility"] = facility;
            TestContext.Properties["section"] = section;
            TestContext.Properties["resource"] = retrievedResource;
            TestContext.Properties["resource2"] = resource2;
            TestContext.Properties["cat"] = cat;


        }
        
        //Use TestCleanup to run code after each test has run
        [TestCleanup()]
        public void MyTestCleanup()
        {
            var site = TestContext.Properties["site"] as Site;
            var facility = TestContext.Properties["facility"] as Facility;
            var section = TestContext.Properties["section"] as Section;
            var resource = TestContext.Properties["resource"] as Resource;
            var resource2 = TestContext.Properties["resource2"] as Resource;
            var cat = TestContext.Properties["cat"] as AppointmentTypeCategory;

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(resource);
                dbSession.Delete(resource2);
                dbSession.Delete(section);
                dbSession.Delete(facility);
                dbSession.Delete(site);
                dbSession.Delete(cat);

                transaction.Commit();
            }
        }
        
        #endregion

        [TestMethod()]
        public void MultipleResourceAvailabilityTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetResourceService();
            var resource1 = TestContext.Properties["resource"] as Resource;
            var resource2 = TestContext.Properties["resource2"] as Resource;

            SchedulingPolicy monday = new SchedulingPolicy(){ DayOfWeek = 1, StartEffectDate=new DateTime(2013, 1, 1), Priority = 1};
            SchedulingHelper.AddSlotsToPolicy(ref monday, new TimeSpan(8, 0, 0), new TimeSpan(10, 0, 0), 60);

            SchedulingPolicy monday2 = new SchedulingPolicy(){ DayOfWeek = 1, StartEffectDate=new DateTime(2013, 1, 1), Priority = 1};
            SchedulingHelper.AddSlotsToPolicy(ref monday2, new TimeSpan(8, 0, 0), new TimeSpan(10, 0, 0), 60);

            resource1.AddShedulingPolicy(monday);
            resource2.AddShedulingPolicy(monday2);

            resource1 = service.Update(resource1);
            resource2 = service.Update(resource2);

            var patient = new Patient()
            {
                ICN = "123",
                Person = new Person()
                {
                    FirstName = "Bob",
                    LastName = "Hope",
                    Sex = "M",
                    DateOfBirth = new DateTime(1970, 1, 1),
                    Email = "BobHope@hotmail.com"
                }
            };

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(patient);
                transaction.Commit();
            }

            Appointment appointment1 = new Appointment()
            {
                Time = new DateTime(2013, 5, 13, 9, 0, 0),
                Length = 60,
                Patient = patient,
                Status = AppointmentStatus.Scheduled,
            };
            appointment1.AddResource(resource1);
            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(appointment1);
                transaction.Commit();
            }

            DateTime apptRequest_Avail = new DateTime(2013, 5, 13, 8, 0, 0);
            DateTime apptRequest_NotAvail = new DateTime(2013, 5, 13, 9, 0, 0);

            Collection<int> resources = new Collection<int>();
            resources.Add(resource1.Id);
            resources.Add(resource2.Id);

            var availability = service.GetAvailability(resources, apptRequest_Avail.Date, apptRequest_Avail.Date);

            Assert.IsTrue(SchedulingHelper.IsTimeAvailable(availability, apptRequest_Avail));
            Assert.IsFalse(SchedulingHelper.IsTimeAvailable(availability, apptRequest_NotAvail));

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(appointment1);
                dbSession.Delete(patient);
                transaction.Commit();
            }

            TestContext.Properties["resource"] = resource1;
            TestContext.Properties["resource2"] = resource2;
        }

        [TestMethod()]
        public void MultipleResourceAvailabilityAppointmentTypeTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetResourceService();
            var resource1 = TestContext.Properties["resource"] as Resource;
            var resource2 = TestContext.Properties["resource2"] as Resource;
            var section =  TestContext.Properties["section"] as Section;

            SchedulingPolicy monday = new SchedulingPolicy() { DayOfWeek = 1, StartEffectDate = new DateTime(2013, 1, 1), Priority = 1 };
            SchedulingHelper.AddSlotsToPolicy(ref monday, new TimeSpan(8, 0, 0), new TimeSpan(10, 0, 0), 60);

            foreach (var slot in monday.Slots)
            {
                slot.AppointmentType = section.AppointmentTypes[0];
            }
            SchedulingPolicy monday2 = new SchedulingPolicy() { DayOfWeek = 1, StartEffectDate = new DateTime(2013, 1, 1), Priority = 1 };
            SchedulingHelper.AddSlotsToPolicy(ref monday2, new TimeSpan(8, 0, 0), new TimeSpan(10, 0, 0), 60);

            resource1.AddShedulingPolicy(monday);
            resource2.AddShedulingPolicy(monday2);

            resource1 = service.Update(resource1);
            resource2 = service.Update(resource2);

            var patient = new Patient()
            {
                ICN = "123",
                Person = new Person()
                {
                    FirstName = "Bob",
                    LastName = "Hope",
                    Sex = "M",
                    DateOfBirth = new DateTime(1970, 1, 1),
                    Email = "BobHope@hotmail.com"
                }
            };

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(patient);
                transaction.Commit();
            }

            //Appointment appointment1 = new Appointment()
            //{
            //    Time = new DateTime(2013, 5, 13, 9, 0, 0),
            //    Length = 60,
            //    Patient = patient,
            //    Status = AppointmentStatus.Scheduled,
            //};
            //appointment1.AddResource(resource1);
            //using (var dbSession = DataAccess.DataAccess.OpenSession())
            //using (var transaction = dbSession.BeginTransaction())
            //{
            //    dbSession.Save(appointment1);
            //    transaction.Commit();
            //}

            DateTime apptRequest_Avail = new DateTime(2013, 5, 13, 8, 0, 0);
            DateTime apptRequest_NotAvail = new DateTime(2013, 5, 13, 9, 0, 0);

            Collection<int> resources = new Collection<int>();
            resources.Add(resource1.Id);
            resources.Add(resource2.Id);

            var availability = service.GetAvailability(resources, apptRequest_Avail.Date, apptRequest_Avail.Date);

            Assert.IsTrue(SchedulingHelper.IsTimeAvailable(availability, apptRequest_Avail));
            Assert.IsTrue(SchedulingHelper.IsTimeAvailable(availability, apptRequest_NotAvail));

            availability = service.GetAvailability(resources, apptRequest_Avail.Date, apptRequest_Avail.Date, section.AppointmentTypes[0]);

            Assert.IsTrue(SchedulingHelper.IsTimeAvailable(availability, apptRequest_Avail));
            Assert.IsTrue(SchedulingHelper.IsTimeAvailable(availability, apptRequest_NotAvail));
            
            availability = service.GetAvailability(resources, apptRequest_Avail.Date, apptRequest_Avail.Date, section.AppointmentTypes[1]);

            Assert.IsFalse(SchedulingHelper.IsTimeAvailable(availability, apptRequest_Avail));
            Assert.IsFalse(SchedulingHelper.IsTimeAvailable(availability, apptRequest_NotAvail));


            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
//                dbSession.Delete(appointment1);
                dbSession.Delete(patient);
                transaction.Commit();
            }

            TestContext.Properties["resource"] = resource1;
            TestContext.Properties["resource2"] = resource2;
        }

    }
}
