﻿using MedRed.Services;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using MedRed.DataAccess;
using System.Linq;
using NHibernate;
using MedRed.MDWSAccess;
using Shared.Model.Config.MDWS;
using System.Collections.Generic;
using Shared.Model;

namespace MedRed.Services.Tests
{
    [TestClass()]
    public class NationalSystemTest
    {   
        private TestContext testContextInstance;

        /// <summary>
        ///Gets or sets the test context which provides
        ///information about and functionality for the current test run.
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        #region Additional test attributes
        // 
        //You can use the following additional attributes as you write your tests:
        //
        //Use ClassInitialize to run code before running the first test in the class
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //Use ClassCleanup to run code after all tests in a class have run
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //Use TestInitialize to run code before running each test
        [TestInitialize()]
        public void MyTestInitialize()
        {
            MDWSConfiguration config = ConnectionHelper.GetTestingConfig();
            var factory = Factory.GetFactory(config);

            TestContext.Properties["Factory"] = factory;
        }

        //Use TestCleanup to run code after each test has run
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion


        [TestMethod()]
        public void getServicesTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetNationalSystemService();

            Assert.IsNotNull(service);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            {
                Service[] services = { new Service() { Name = "TestService 1" }, new Service() { Name = "TestService 2" } };

                dbSession.Save(services[0]);
                dbSession.Save(services[1]);

                var standardServiceList = service.GetServices();

                var serviceList = new List<Service>(standardServiceList);
                Assert.IsTrue(serviceList.Count >= 2);
                Assert.IsTrue(serviceList.Exists(a => a.Name == "TestService 1"));
                Assert.IsTrue(serviceList.Exists(a => a.Name == "TestService 2"));

                dbSession.Delete(services[0]);
                dbSession.Delete(services[1]);
            }
        }

        [TestMethod()]
        public void getAppointmentTypeCategoriesTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetNationalSystemService();

            Assert.IsNotNull(service);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            {
                AppointmentTypeCategory[] apptTypes = { new AppointmentTypeCategory() { Name = "Appt Type Category 1", VistaSiteId="123" }, 
                                                      new AppointmentTypeCategory() { Name = "Appt Type Category 2", VistaSiteId="123" } };
                dbSession.Save(apptTypes[0]);
                dbSession.Save(apptTypes[1]);

                var apptTypeCategories = service.GetAppointmentTypeCategories("123");

                var appTypeCatList = new List<AppointmentTypeCategory>(apptTypeCategories);

                Assert.IsTrue(appTypeCatList.Count >= 2);
                Assert.IsTrue(appTypeCatList.Exists(a => a.Name == "Appt Type Category 1"));
                Assert.IsTrue(appTypeCatList.Exists(a => a.Name == "Appt Type Category 2"));

                dbSession.Delete(apptTypes[0]);
                dbSession.Delete(apptTypes[1]);
            }
        }

        [TestMethod()]
        public void getCancellationReasonsTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetNationalSystemService();

            Assert.IsNotNull(service);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            {
                CancellationReason[] reasons = { new CancellationReason() { Name = "Reason 1", VistaSiteId="123" }, 
                                                      new CancellationReason() { Name = "Reason 2", VistaSiteId="123" } };
                dbSession.Save(reasons[0]);
                dbSession.Save(reasons[1]);

                var cancelReasons = service.GetCancellationReasons("123");

                var reasonList = new List<CancellationReason>(cancelReasons);

                Assert.IsTrue(reasonList.Count >= 2);
                Assert.IsTrue(reasonList.Exists(a => a.Name == "Reason 1"));
                Assert.IsTrue(reasonList.Exists(a => a.Name == "Reason 2"));

                dbSession.Delete(reasons[0]);
                dbSession.Delete(reasons[1]);
            }
        }

        [TestMethod()]
        public void getStopCodeTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetNationalSystemService();

            Assert.IsNotNull(service);

            StopCode[] array = { new StopCode() { Name = "Reason 1", Code = "323", VistaSiteId="123", RestrictionType = RestrictionType.Primary }, 
                                        new StopCode() { Name = "Reason 2", Code = "368", VistaSiteId="123", RestrictionType = RestrictionType.Either } };

            service.AddStopCode(array[0]);
            service.AddStopCode(array[1]);

            var stopCodes = service.GetStopCodes("123");

            var stopCodeList = new List<StopCode>(stopCodes);

            Assert.IsTrue(stopCodeList.Count >= 2);
            Assert.IsTrue(stopCodeList.Exists(a => a.Name == "Reason 1"));
            Assert.IsTrue(stopCodeList.Exists(a => a.Name == "Reason 2"));

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            {
                dbSession.Delete(stopCodeList[0]);
                dbSession.Delete(stopCodeList[1]);
            }
        }


        [TestMethod()]
        public void getSecondaryStopCodeTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetNationalSystemService();

            Assert.IsNotNull(service);

            StopCode[] array = { new StopCode() { Name = "Reason 1", Code = "323", VistaSiteId="123", RestrictionType = RestrictionType.Primary }, 
                                        new StopCode() { Name = "Reason 2", Code = "368", VistaSiteId="123", RestrictionType = RestrictionType.Either } };

            service.AddStopCode(array[0]);
            service.AddStopCode(array[1]);

            var stopCodes = service.GetStopCodes("123", RestrictionType.Secondary);

            var stopCodeList = new List<StopCode>(stopCodes);

            Assert.IsTrue(stopCodeList.Count >= 1);
            Assert.IsTrue(!stopCodeList.Exists(a => a.Name == "Reason 1"));
            Assert.IsTrue(stopCodeList.Exists(a => a.Name == "Reason 2"));

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            {
                dbSession.Delete(stopCodeList[0]);
                dbSession.Delete(stopCodeList[1]);
            }
        }


        [TestMethod()]
        public void getHolidaysTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetNationalSystemService();

            Assert.IsNotNull(service);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            {
                Holiday[] newHolidays = { new Holiday() { Name = "Holiday 1", Date = new DateTime(2012, 1, 1) }, 
                                    new Holiday() { Name = "Holiday 2", Date = new DateTime(2013, 7, 4) } };
                dbSession.Save(newHolidays[0]);
                dbSession.Save(newHolidays[1]);

                var holidays = service.GetHolidays();

                var HolidayList = new List<Holiday>(holidays);

                Assert.IsTrue(HolidayList.Count >= 2);
                Assert.IsTrue(HolidayList.Exists(h => h.Name == "Holiday 1"));
                Assert.IsTrue(HolidayList.Exists(h => h.Name == "Holiday 2"));

                dbSession.Delete(newHolidays[0]);
                dbSession.Delete(newHolidays[1]);
            }
        }

        [TestMethod()]
        public void getPatientPreferenceTypesTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetNationalSystemService();

            PatientPreferenceType ppType = new PatientPreferenceType() { Name = "Morning Appointments" };

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(ppType);
                transaction.Commit();
            }

            var preferenceTypes = service.GetPatientPreferenceTypes();
            Assert.IsTrue(preferenceTypes.Where(p => p.Name == "Morning Appointments").Count() > 0);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(ppType);
                transaction.Commit();
            }
        }

        [TestMethod()]
        public void addPatientPreferenceTypeTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetNationalSystemService();

            PatientPreferenceType ppType = new PatientPreferenceType() { Name = "Morning Appointments" };

            service.AddPatientPreferenceType(ppType);

            var preferenceTypes = service.GetPatientPreferenceTypes();
            Assert.IsTrue(preferenceTypes.Where(p => p.Name == "Morning Appointments").Count() > 0);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(ppType);
                transaction.Commit();
            }
        }

        [TestMethod()]
        public void getSpecialNeedTypesTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetNationalSystemService();

            SpecialNeedType snType = new SpecialNeedType() { Name = "Needs Wheelchair" };

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(snType);
                transaction.Commit();
            }

            var specialNeedTypes = service.GetSpecialNeedTypes();
            Assert.IsTrue(specialNeedTypes.Where(p => p.Name == "Needs Wheelchair").Count() > 0);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(snType);
                transaction.Commit();
            }
        }


        [TestMethod()]
        public void addSpecialNeedTypeTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetNationalSystemService();

            SpecialNeedType snType = new SpecialNeedType() { Name = "Needs Wheelchair" };

            service.AddSpecialNeedType(snType);

            var specialNeedTypes = service.GetSpecialNeedTypes();
            Assert.IsTrue(specialNeedTypes.Where(p => p.Name == "Needs Wheelchair").Count() > 0);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(snType);
                transaction.Commit();
            }
        }
    }
}
