﻿using MedRed.Services;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using MedRed.DataAccess;
using NHibernate;
using MedRed.MDWSAccess;
using Shared.Model.Config.MDWS;
using System.Collections.Generic;
using Shared.Model;
using KellermanSoftware.CompareNetObjects;

namespace MedRed.Services.Tests
{
    [TestClass()]
    public class FacilityServiceTest
    {   
        private TestContext testContextInstance;

        /// <summary>
        ///Gets or sets the test context which provides
        ///information about and functionality for the current test run.
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        #region Additional test attributes
        // 
        //You can use the following additional attributes as you write your tests:
        //
        //Use ClassInitialize to run code before running the first test in the class
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //Use ClassCleanup to run code after all tests in a class have run
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            MDWSConfiguration config = ConnectionHelper.GetTestingConfig();
//            MDWSConnection connection = ConnectionHelper.GetConnectionFor("100");
            var factory = Factory.GetFactory(config);

            TestContext.Properties["Factory"] = factory;
        }
        //Use TestCleanup to run code after each test has run
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion


        [TestMethod()]
        public void getAllFacilitiesTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetFacilityService();

            Assert.IsNotNull(service);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            {
                Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
                dbSession.Save(site);

                int siteId = site.Id;

                Facility[] array = { new Facility() { Name = "Facility 1", Site = site}, 
                                   new Facility() {Name = "Facility 2", Site = site}};

                dbSession.Save(array[0]);
                dbSession.Save(array[1]);

                var facilities = service.GetAll(siteId);
                var facilityList = new List<Facility>(facilities);

                Assert.IsTrue(facilityList.Count >= 2);
                Assert.IsTrue(facilityList.Exists(s => s.Name == "Facility 1"));
                Assert.IsTrue(facilityList.Exists(s => s.Name == "Facility 2"));

                dbSession.Delete(array[0]);
                dbSession.Delete(array[1]);
                dbSession.Delete(site);
            }
        }

        [TestMethod()]
        public void getFacilityTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetFacilityService();

            Assert.IsNotNull(service);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            {
                Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
                dbSession.Save(site);
                int siteId = site.Id;
                Facility[] array = { new Facility() { Name = "Facility 1", Site = site}, 
                                   new Facility() {Name = "Facility 2", Site = site}};
                dbSession.Save(array[0]);
                dbSession.Save(array[1]);

                int facilityId = array[0].Id;

                Assert.IsTrue(facilityId != 0);

                var facility = service.Get(facilityId);

                CompareObjects compareObjects = new CompareObjects();
                Assert.IsTrue(compareObjects.Compare(facility, array[0]), compareObjects.DifferencesString);

                dbSession.Delete(array[0]);
                dbSession.Delete(array[1]);
                dbSession.Delete(site);
            }
        }

        [TestMethod()]
        public void addFacilityTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetFacilityService();

            Assert.IsNotNull(service);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            {
                Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
                dbSession.Save(site);
                int siteId = site.Id;
                Facility[] array = { new Facility() { Name = "Facility 1", Site = site } };

                Assert.IsTrue(array[0].Id == 0);
                var facility = service.Add(array[0]);

                int facilityId = facility.Id;

                Assert.IsTrue(facilityId != 0);

                var retrievedFacility = service.Get(facilityId);

                CompareObjects compareObjects = new CompareObjects();
                Assert.IsTrue(compareObjects.Compare(retrievedFacility, array[0]), compareObjects.DifferencesString);

                dbSession.Delete(array[0]);
                dbSession.Delete(site);
            }
        }

        [TestMethod()]
        public void addCommTemplateFacilityTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetFacilityService();

            Assert.IsNotNull(service);

            int facilityId = 0;
            Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
            Facility[] array = { new Facility() { Name = "Facility 1", Site = site } };

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(site);
                int siteId = site.Id;

                Assert.IsTrue(array[0].Id == 0);
                dbSession.Save(array[0]);

                facilityId = array[0].Id;

                Assert.IsTrue(facilityId != 0);
                transaction.Commit();
            }

            CommunicationTemplate[] commTemplates = 
            {
                new CommunicationTemplate() { Name="Template 1", Type = CommunicationType.PreAppointment, Message = "My message", Recipient = "Patient"},
                new CommunicationTemplate() { Name="Template 2", Type = CommunicationType.Cancellation, Message = "Sorry your appt is cancelled", Recipient = "Patient"},
            };

            service.AddOrUpdateCommunicationTemplate(facilityId, commTemplates[0]);
            service.AddOrUpdateCommunicationTemplate(facilityId, commTemplates[1]);

            var retrievedFacility = service.Get(facilityId);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {

                Assert.AreEqual(retrievedFacility.CommunicationTemplates.Count, 2);

                CompareObjects compareObjects = new CompareObjects();
//                compareObjects.ElementsToIgnore.Add("Id");
                Assert.IsTrue(compareObjects.Compare(retrievedFacility.CommunicationTemplates[0], commTemplates[0]), compareObjects.DifferencesString);
                Assert.IsTrue(compareObjects.Compare(retrievedFacility.CommunicationTemplates[1], commTemplates[1]), compareObjects.DifferencesString);

                dbSession.Delete(retrievedFacility.CommunicationTemplates[0]);
                dbSession.Delete(retrievedFacility.CommunicationTemplates[0]);
                dbSession.Delete(retrievedFacility);
                dbSession.Delete(site);
                transaction.Commit();
            }
        }

        [TestMethod()]
        public void EditCommTemplateFacilityTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetFacilityService();

            Assert.IsNotNull(service);

            int facilityId = 0;
            Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
            Facility facility = new Facility() { Name = "Facility 1", Site = site };

            CommunicationTemplate[] commTemplates = 
            {
                new CommunicationTemplate() { Name="Template 1", Type = CommunicationType.PreAppointment, Message = "My message", Recipient = "Patient"},
                new CommunicationTemplate() { Name="Template 2", Type = CommunicationType.Cancellation, Message = "Sorry your appt is cancelled", Recipient = "Patient"},
            };

           using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(site);
                int siteId = site.Id;

                facility.CommunicationTemplates.Add(commTemplates[0]);
                facility.CommunicationTemplates.Add(commTemplates[1]);

                Assert.IsTrue(facility.Id == 0);
                dbSession.Save(facility);

                facilityId = facility.Id;

                Assert.IsTrue(facilityId != 0);
                transaction.Commit();
            }

            var originalFacility = service.Get(facilityId);

            var communicationToEdit = originalFacility.CommunicationTemplates[1];

            communicationToEdit.Message = "This is my updated message";

            service.AddOrUpdateCommunicationTemplate(facilityId, communicationToEdit);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                var retrievedFacility = service.Get(facilityId);

                Assert.AreEqual(retrievedFacility.CommunicationTemplates.Count, 2);

                Assert.AreEqual(retrievedFacility.CommunicationTemplates[1].Message, "This is my updated message");

                dbSession.Delete(commTemplates[0]);
                dbSession.Delete(commTemplates[1]);
                dbSession.Delete(facility);
                dbSession.Delete(site);
                transaction.Commit();
            }

        }

        [TestMethod()]
        public void DeleteCommTemplateFacilityTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetFacilityService();

            Assert.IsNotNull(service);

            int facilityId = 0;
            Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
            Facility facility = new Facility() { Name = "Facility 1", Site = site };

            CommunicationTemplate[] commTemplates = 
            {
                new CommunicationTemplate() { Name="Template 1", Type = CommunicationType.PreAppointment, Message = "My message", Recipient = "Patient"},
                new CommunicationTemplate() { Name="Template 2", Type = CommunicationType.Cancellation, Message = "Sorry your appt is cancelled", Recipient = "Patient"},
            };

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(site);
                int siteId = site.Id;

                facility.CommunicationTemplates.Add(commTemplates[0]);
                facility.CommunicationTemplates.Add(commTemplates[1]);

                Assert.IsTrue(facility.Id == 0);
                dbSession.Save(facility);

                facilityId = facility.Id;

                Assert.IsTrue(facilityId != 0);
                transaction.Commit();
            }

            var originalFacility = service.Get(facilityId);

            var communicationToDelete = originalFacility.CommunicationTemplates[1];

            service.DeleteCommunicationTemplate(facilityId, communicationToDelete);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                var retrievedFacility = service.Get(facilityId);

                Assert.AreEqual(retrievedFacility.CommunicationTemplates.Count, 1);

                dbSession.Delete(retrievedFacility);
                dbSession.Delete(site);
                transaction.Commit();
            }
        }

        [TestMethod()]
        public void addBigCommTemplateFacilityTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetFacilityService();

            Assert.IsNotNull(service);

            int facilityId = 0;
            Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
            Facility[] array = { new Facility() { Name = "Facility 1", Site = site } };

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(site);
                int siteId = site.Id;

                Assert.IsTrue(array[0].Id == 0);
                dbSession.Save(array[0]);

                facilityId = array[0].Id;

                Assert.IsTrue(facilityId != 0);
                transaction.Commit();
            }

            CommunicationTemplate[] commTemplates = 
            {
                new CommunicationTemplate() { Name="Template 1", Type = CommunicationType.PreAppointment, Message = "My message", Recipient = "Patient"},
                new CommunicationTemplate() { Name="Template 2", Type = CommunicationType.Cancellation, Message = "Sorry your appt is cancelled", Recipient = "Patient"},
            };

            commTemplates[1].Message =
                @"Lorem ipsum dolor sit amet, consectetur adipiscing elit. Sed sit amet quam nisl, quis dignissim neque. Nullam leo nulla, vehicula id rhoncus eu, molestie in turpis. Quisque et velit purus, vel porta tellus. Nullam semper lobortis libero nec rutrum. Integer rutrum mollis consequat. Integer feugiat, risus consectetur facilisis vehicula, sapien felis feugiat lacus, consequat posuere dolor lectus vel risus. Vivamus dapibus eros sit amet leo cursus id elementum justo convallis. Aenean odio eros, accumsan eget placerat id, rhoncus ac urna. Phasellus ut lorem ipsum. Fusce semper tellus auctor nunc lobortis facilisis.

Quisque ut diam at lorem tristique imperdiet ac faucibus mi. Etiam feugiat, augue sagittis gravida vulputate, massa libero ultricies lacus, sit amet iaculis enim ante non orci. Proin pulvinar nibh sit amet dolor interdum auctor. Proin sit amet lectus et augue tincidunt laoreet quis sed nunc. Donec rutrum, ante sed blandit commodo, nibh quam dignissim libero, ut vestibulum tellus nibh nec felis. Aliquam lobortis blandit nunc sit amet porttitor. Ut cursus auctor interdum. Integer rhoncus nisi vel augue venenatis id tincidunt felis tempus. Sed interdum rutrum ipsum a placerat. Suspendisse non nisi id dui varius interdum. Sed vitae enim non arcu eleifend porta. Cras feugiat iaculis nisi, sit amet porta eros mattis ac. Sed rhoncus viverra nisi, at feugiat justo facilisis a. Pellentesque magna ante, pulvinar quis blandit ac, rhoncus at nibh.

Etiam quis elit risus. Aenean pharetra, diam in consequat malesuada, risus tortor laoreet diam, id faucibus magna diam quis ante. Sed a erat purus. Integer ultricies consectetur quam quis congue. Nullam ligula tellus, tempus eu tincidunt a, cursus non ipsum. Integer in turpis vitae odio tincidunt euismod quis sit amet turpis. Ut ac porttitor nulla. Fusce eros metus, vehicula et auctor vitae, rhoncus nec felis. Vestibulum hendrerit varius sapien quis aliquam.

Quisque nulla diam, ultricies non dictum in, tincidunt et sapien. Maecenas molestie lacus quis libero egestas feugiat. Donec ut enim risus. Sed imperdiet, augue nec pulvinar condimentum, dui orci tempor velit, a posuere tellus mi quis ante. Duis et est orci. Proin blandit volutpat tincidunt. Curabitur vitae dolor ligula. Mauris eu nunc in sem pulvinar dapibus. Praesent porttitor, augue sit amet scelerisque facilisis, neque tellus fermentum mauris, at tincidunt orci magna sed magna. Morbi nec nisi ac urna sagittis consequat ut eu justo. Sed volutpat, orci eget placerat vehicula, diam tellus tristique est, at gravida nulla libero varius magna. Morbi arcu ligula, tristique ut aliquet nec, elementum a leo. In gravida, enim quis adipiscing convallis, mi nibh vestibulum velit, non lobortis est neque vitae nisi.

Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas. Suspendisse tempus lobortis arcu id tristique. Cras lacus lectus, porta non dapibus nec, congue vel augue. Aliquam eu leo vitae nibh auctor adipiscing. Phasellus consequat ultricies iaculis. Nullam mollis nulla aliquet risus porttitor vitae suscipit enim imperdiet. Donec convallis placerat ligula, non facilisis diam adipiscing quis.

Nulla cursus nulla a enim tincidunt nec mattis dui pretium. Nam fermentum nisl vel urna iaculis vel feugiat velit vehicula. Ut risus purus, gravida vitae dapibus ac, dignissim eget nibh. Pellentesque ac vestibulum nisi. Nam vulputate arcu et diam convallis vulputate. Curabitur molestie mi sed turpis malesuada vestibulum. Integer dignissim sem sit amet lacus gravida quis pretium libero accumsan.

Etiam a quam magna, sed cursus nibh. Sed sodales tellus ac nulla convallis sagittis. Suspendisse sollicitudin metus eu felis laoreet a fringilla lectus congue. Sed dignissim dolor pretium risus tincidunt dictum. Maecenas at neque enim, et dignissim tortor. Pellentesque neque neque, bibendum a pharetra id, condimentum non metus. Etiam imperdiet viverra rhoncus.

Mauris non nibh sed arcu rutrum aliquam. Donec consectetur, augue eget malesuada varius, massa odio adipiscing erat, vel auctor arcu metus nec erat. Proin quis nibh id nulla imperdiet interdum quis eu libero. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Sed adipiscing lacus eu turpis semper posuere rutrum nisl varius. Praesent eu tincidunt tortor. Morbi sapien massa, facilisis a aliquam sed, feugiat lobortis orci. Nulla blandit mi eget orci elementum aliquam. Etiam lacinia convallis ullamcorper. Class aptent taciti sociosqu ad litora torquent per conubia nostra, per inceptos himenaeos. Praesent eros arcu, porttitor pretium aliquet nec, volutpat et leo.

Morbi ut velit odio. Quisque dapibus est id ipsum semper laoreet. Fusce gravida sem non nisi sodales vel imperdiet nibh dapibus. Aenean semper ipsum ultrices nisl rutrum pretium. Aenean elementum venenatis nulla, eu sollicitudin tortor suscipit eget. Nullam porta nunc sed risus semper elementum. Donec venenatis mattis neque vitae venenatis. Etiam sodales feugiat tortor in vehicula. Nulla facilisi. Fusce quis leo nisl. Duis vehicula augue sed arcu lacinia eu malesuada magna sagittis. Fusce scelerisque feugiat semper.

Donec ultricies eleifend erat, quis venenatis tortor commodo at. Nullam risus neque, tempus vitae tincidunt sit amet, facilisis et nulla. In hac habitasse platea dictumst. Pellentesque eu sem eget orci congue pharetra eget a nibh. Cras venenatis cursus turpis in laoreet. Cras facilisis tortor ut augue tempor sed pharetra libero eleifend. Nulla facilisi. Nullam vitae est quam. Donec diam arcu, placerat eu congue sit amet, sollicitudin sit amet dolor. Nulla egestas sem ac erat sodales malesuada. In volutpat luctus pellentesque.

Aliquam consequat condimentum orci eu lobortis. Suspendisse consectetur posuere urna, sit amet pulvinar sapien volutpat a. Fusce sem ante, bibendum id rhoncus vel, blandit a neque. Vivamus sed ipsum mollis elit pharetra malesuada. Phasellus porttitor est euismod urna imperdiet in dignissim magna ultricies. Aenean sagittis erat ut eros porttitor mattis. Nunc viverra nibh mi.

Morbi leo ante, consequat et cursus vitae, molestie quis mi. Sed ultricies, est ut lacinia pulvinar, metus augue fringilla nunc, et accumsan odio velit ut nunc. Mauris at justo dolor, accumsan gravida felis. Mauris non nulla erat, sed vulputate elit. Lorem ipsum dolor sit amet, consectetur adipiscing elit. Nulla eu sem quis arcu ultrices suscipit facilisis elementum orci. Sed vel quam non eros rutrum hendrerit mattis sodales nunc. Morbi eu diam lectus, nec auctor lorem.

In aliquet, turpis in sollicitudin dictum, massa lorem hendrerit lorem, eu posuere velit risus nec dolor. Sed sagittis feugiat ante in porttitor. Vivamus auctor urna a libero interdum feugiat. Sed eu urna lectus, in mattis purus. Mauris pulvinar mollis mollis. Sed mollis metus nec nunc auctor varius tristique magna venenatis. Nunc sit amet augue a augue fermentum iaculis.

Cras id tortor vel purus ornare scelerisque id eget risus. Pellentesque ac pulvinar dolor. Duis sit amet mi sed leo tincidunt vulputate. Vivamus ut leo vel metus consectetur hendrerit a nec lacus. Aenean ante tortor, cursus ac ultrices non, congue vitae justo. Quisque iaculis orci vitae diam varius sed imperdiet neque posuere. Sed placerat, augue vel hendrerit sagittis, felis arcu pharetra ipsum, at convallis purus nibh nec est. Nullam euismod malesuada urna a tincidunt. Praesent sit amet dolor vitae neque cursus vehicula vitae at erat. Suspendisse varius sollicitudin dolor, sed gravida magna aliquam sit amet. Aenean feugiat eros enim, a placerat quam.

Praesent in ullamcorper erat. Maecenas eget elit nec velit vehicula gravida. Nulla purus lorem, fringilla id luctus et, ultrices quis augue. Sed dui ipsum, dictum sed auctor ac, semper sit amet tortor. Duis ut quam at quam mollis pulvinar. Proin pharetra arcu vel velit rhoncus a vestibulum ipsum iaculis. Class aptent taciti sociosqu ad litora torquent per conubia nostra, per inceptos himenaeos. Praesent in venenatis ligula. Etiam sagittis quam at tellus ornare eleifend. Aenean vestibulum orci quis nibh iaculis vel tempor mi iaculis. Donec luctus auctor nunc, vel commodo justo venenatis a. Sed nec lacus sed enim eleifend ornare eget a ligula.

Nullam eget elit nulla, eget dignissim felis. Donec tincidunt tellus hendrerit leo gravida quis eleifend mauris tempus. Pellentesque enim mauris, rutrum vel porta et, blandit eu justo. Aenean eget ligula lacus. Integer ut urna augue, sed volutpat justo. Maecenas turpis lectus, consequat eu imperdiet ac, facilisis at diam. Sed sed enim sit amet magna semper dignissim. Phasellus lacinia magna auctor dui ullamcorper varius. Praesent sit amet nulla cursus arcu scelerisque adipiscing. Morbi posuere lacus pretium leo venenatis mollis. Pellentesque vulputate blandit dapibus. Suspendisse nec dolor lectus, id ultricies mi.

Etiam non neque eget tortor ultrices gravida id ut quam. Donec in lacus a leo porta ornare et nec urna. Cras vitae nunc eu purus lobortis feugiat. Donec varius rutrum dolor, suscipit egestas nisl consequat a. Aenean scelerisque pulvinar dui quis accumsan. Pellentesque et congue risus. Nullam ac nisl nec lacus tempor dapibus quis a purus. Phasellus orci enim, adipiscing sit amet cursus sit amet, sollicitudin eu turpis. In fringilla convallis elit quis aliquam. Aliquam erat volutpat. Proin mollis est in justo blandit dictum. Aenean non elit a tellus molestie condimentum. Aenean felis sem, iaculis eget posuere ut, dictum malesuada nibh.

Nunc bibendum condimentum libero, vitae pellentesque eros hendrerit ut. Vivamus fermentum, enim eleifend venenatis convallis, enim sem bibendum metus, eu egestas augue odio nec lectus. Etiam lacinia tincidunt massa, quis aliquam ligula dictum non. Duis varius arcu in est iaculis sed ultricies dui sodales. Suspendisse vitae tortor sapien. Cras pellentesque purus vitae felis sagittis porta in a quam. Suspendisse adipiscing augue quis quam ultricies malesuada. Suspendisse potenti.

Suspendisse vitae risus vitae lectus fringilla pretium. Fusce sit amet massa quam, in venenatis risus. Donec facilisis mattis purus, nec eleifend ligula dignissim vel. Vestibulum eu placerat risus. Sed tempor vestibulum dolor eu vulputate. Mauris eget urna augue. In dui tellus, aliquam at adipiscing nec, luctus at dolor. Mauris lobortis tristique blandit. Quisque sed erat ligula. Proin commodo sollicitudin ornare. Class aptent taciti sociosqu ad litora torquent per conubia nostra, per inceptos himenaeos. Quisque nec mi consectetur tellus sagittis ultricies. Aliquam mauris ante, scelerisque vitae scelerisque ut, semper at magna. Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus.

Phasellus volutpat sapien convallis erat semper sagittis. Curabitur non rhoncus felis. Pellentesque a blandit tellus. Cras felis lectus, bibendum ac accumsan sit amet, viverra sed nisi. Quisque a turpis leo, at dignissim tortor. Nam laoreet molestie dui, sed bibendum quam porta vel. Nullam varius orci est. Duis dapibus tempus nisl vitae ullamcorper. Mauris placerat mattis est, vel pellentesque felis consequat sed. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Curabitur tincidunt varius ante iaculis pretium. Donec a ullamcorper ante. Cras eu ligula ipsum.

Cras ac dui est, vel pellentesque magna. Nunc sem lectus, feugiat in commodo ac, congue eu turpis. Ut ligula leo, gravida ut congue ac, auctor sed magna. Vestibulum sed odio vitae nibh consequat dapibus a non sapien. Vivamus laoreet dui eu mauris bibendum ornare. Quisque in quam a mi sagittis mattis eget eu risus. Donec nulla tortor, blandit ac commodo vel, hendrerit et sem. Pellentesque justo libero, euismod faucibus rutrum a, ornare quis libero. Integer sit amet magna mauris, in vestibulum ipsum. Vestibulum leo magna, euismod eget rhoncus sed, sollicitudin at mi. Aliquam ac nunc lorem. Aliquam erat volutpat.

Nunc elementum arcu eu est mollis elementum. Donec aliquam vehicula malesuada. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; In mattis magna quis leo malesuada blandit. Quisque vestibulum eleifend commodo. Proin adipiscing porta arcu at gravida. Cras ultricies ante quis libero feugiat eu posuere ligula convallis. Vestibulum ac metus metus, id dictum neque. Donec accumsan scelerisque nisi, posuere porta elit mattis commodo. Aenean aliquet convallis nunc blandit lobortis.

Praesent eget aliquet arcu. Pellentesque id felis viverra tortor convallis imperdiet id sed ante. Aliquam tempus viverra orci. Duis eu vehicula nisl. Nunc viverra pulvinar enim et fermentum. Suspendisse auctor pharetra vulputate. Sed consectetur iaculis urna, vel venenatis nisi scelerisque ac.

Vivamus eget nisi diam, quis lobortis purus. Integer pulvinar augue non justo interdum at malesuada felis tempor. Suspendisse porta porttitor mauris, vitae malesuada nibh vestibulum vel. Nulla risus est, feugiat ullamcorper blandit quis, sollicitudin vitae quam. Aenean pulvinar, eros vitae tempus vehicula, nisi purus aliquet nulla, in mattis augue tellus non erat. Integer vitae posuere felis. Integer fermentum, arcu eget dictum euismod, turpis libero molestie mi, id eleifend ligula est et augue.

Etiam nibh enim, tempus dictum molestie nec, laoreet quis magna. Nam vel purus lacus, ac scelerisque libero. Nulla mi ante, blandit sit amet interdum at, consequat eget nunc. Pellentesque augue augue, accumsan at dignissim sit amet, mollis non eros. Quisque facilisis volutpat porttitor. Fusce ac dictum lorem. Mauris sed euismod purus. Vivamus leo dui, rutrum nec venenatis eu, porta at velit.

Vivamus at velit id elit adipiscing adipiscing. Donec ligula sem, tempor quis venenatis a, interdum eget tortor. Pellentesque molestie orci et turpis blandit sit amet viverra neque egestas. Proin eros elit, iaculis in facilisis vel, hendrerit aliquam orci. Aenean lobortis lacinia ipsum, non mollis nisi tempor in. Vivamus fringilla facilisis risus, sit amet dictum odio varius a. Ut purus sem, scelerisque eget lacinia sit amet, lacinia vitae augue. Morbi convallis tellus nec ante ultricies ut auctor ipsum varius. Curabitur orci arcu, tincidunt vitae condimentum mattis, malesuada ut arcu. Ut augue felis, mollis a vehicula id, egestas eu metus. Maecenas tortor turpis, suscipit ut consectetur egestas, viverra nec tortor. Morbi dolor massa, lacinia in hendrerit sit amet, tempus quis augue. Quisque porta porttitor lacus a vehicula. Maecenas eget enim eget nisi hendrerit bibendum et commodo felis. Vestibulum dapibus massa at leo viverra aliquam viverra augue tristique. Integer sem sem, fermentum non fermentum vel, dictum nec orci.

Curabitur ut erat lorem, a luctus ligula. Donec ac justo quis nibh auctor molestie in at eros. Donec auctor leo id nibh vulputate nec volutpat nunc viverra. Proin ultricies, nibh quis dapibus pellentesque, neque arcu sodales odio, id mattis ligula diam vitae tellus. Maecenas at libero lectus, sit amet lobortis sem. Morbi nibh nulla, tincidunt eu mollis eu, tristique eu tortor. Duis hendrerit cursus urna ut condimentum. Donec ultricies ipsum eu nibh lacinia mollis. Nam venenatis vehicula laoreet. Pellentesque vulputate, dolor et egestas aliquam, enim elit lacinia nunc, et vehicula odio justo quis odio.

Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Etiam vel enim elementum tortor tempor molestie. Donec porttitor elit vel tellus bibendum eleifend. Morbi bibendum metus dignissim lacus malesuada vel vestibulum quam molestie. Proin laoreet ornare lorem, in feugiat neque facilisis non. Sed eros lacus, adipiscing et rutrum vitae, pretium ac nulla. In hac habitasse platea dictumst. Praesent laoreet molestie feugiat. Praesent eget urna urna, consectetur tristique justo. Phasellus sed odio a odio venenatis tincidunt. Vivamus vitae arcu libero. Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas.

Donec non metus ac risus facilisis commodo sed vel elit. Phasellus accumsan erat sit amet lacus eleifend quis ultricies dui mollis. Pellentesque egestas quam sit amet urna placerat ultricies in in erat. Sed dapibus metus in risus varius lacinia. Curabitur vitae urna dolor. Suspendisse elementum lorem ac ipsum consequat euismod. Ut interdum, velit mattis fringilla interdum, leo urna pellentesque nisl, sit amet dictum purus augue et quam. Mauris aliquam, lacus non laoreet interdum, magna enim sagittis sapien, vitae pharetra erat leo pretium sem. Morbi tincidunt porta orci, non pulvinar ante dignissim quis. Ut ac sem ipsum, ut ullamcorper nunc. Quisque consectetur placerat auctor. Maecenas rhoncus sodales ligula sit amet accumsan. Aliquam erat volutpat. Aliquam condimentum convallis lorem, id fringilla augue faucibus vitae. Pellentesque sagittis augue a risus mattis vitae hendrerit sem tristique.

Pellentesque posuere, nunc ac venenatis ornare, elit erat tristique enim, ultrices tincidunt diam quam sed nunc. Sed bibendum faucibus dolor, a tincidunt nunc commodo in. Nunc pellentesque volutpat sapien a cursus. Morbi metus lorem, venenatis nec aliquet aliquam, vestibulum vitae lectus. Fusce porttitor dictum facilisis. Sed nec ipsum et massa rutrum posuere sit amet id sem. Proin at mi in enim tristique placerat. Quisque mauris augue, scelerisque ut blandit eget, congue in nisl. Morbi aliquam, risus rhoncus tincidunt egestas, erat leo laoreet lectus, sit amet malesuada nibh sem ac metus. Cras eleifend molestie dignissim. Ut tellus nisl, sollicitudin eu scelerisque vel, pellentesque eu risus. Nunc magna lectus, tempus et varius sit amet, interdum vel felis. Fusce sagittis nunc quis justo iaculis feugiat. Curabitur ut lectus eget quam tincidunt pellentesque eu sed eros. Proin pharetra dolor pretium mi posuere eget vestibulum arcu pulvinar. Nam mattis, est vel gravida dapibus, orci mauris fermentum enim, non laoreet est massa a orci.

Aliquam erat volutpat. Fusce eget urna nec nisl ornare semper quis et felis. Praesent gravida hendrerit tempus. Sed eget diam vel leo pellentesque tempor eget in orci. Aenean pretium pellentesque gravida. Suspendisse fringilla, odio at pulvinar ullamcorper, quam arcu ultrices diam, eu elementum ante orci id neque. Fusce id dui erat. Fusce elementum porta vestibulum. Duis nec est at libero dapibus rutrum. Mauris fermentum interdum est quis venenatis. Vestibulum massa justo, lobortis ut ultricies quis, mollis quis justo. Duis faucibus eros nec diam dapibus nec fermentum risus malesuada. Fusce elementum molestie cursus. Quisque sit amet elit risus. Vestibulum dui nisl, pretium quis aliquam accumsan, imperdiet quis justo.

Morbi luctus vehicula urna nec tincidunt. Maecenas venenatis faucibus justo, in iaculis nisl vestibulum at. Morbi quis lectus sit amet velit vestibulum ullamcorper quis vel tortor. Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas. Nam mollis tristique pretium. Nullam laoreet sagittis hendrerit. Nullam ultricies venenatis est in varius. Duis et nulla eget nunc pellentesque volutpat non nec dui. Proin arcu tellus, ullamcorper vitae consectetur id, tincidunt in metus.

Ut sagittis dictum hendrerit. Nulla aliquet, quam a cursus commodo, lacus nisi hendrerit mauris, ac molestie mi tellus ut libero. Nunc auctor, magna quis condimentum aliquam, ipsum libero pulvinar tellus, id venenatis justo massa id nibh. Aliquam erat volutpat. Aenean et placerat nisi. Nullam at est sit amet nibh porta rutrum et nec turpis. Sed odio erat, ultricies tempus blandit volutpat, luctus a libero. Duis non mauris a elit viverra venenatis. Etiam libero justo, venenatis ac condimentum sed, vehicula vel risus. Aliquam ac enim eros, eget aliquet augue. Suspendisse at eros urna. Quisque quis eros ac turpis aliquam tincidunt. Cras pellentesque nibh molestie sem auctor imperdiet. Morbi eu neque eros.

Aliquam a tristique velit. Vivamus eget nisl in augue convallis gravida in eleifend mi. Ut et interdum diam. Donec risus lacus, cursus in ornare et, dapibus ut tortor. Curabitur sit amet dictum mi. Vivamus consequat quam sit amet nulla egestas vel scelerisque ligula aliquam. Nunc sagittis ornare placerat. Morbi ultricies, metus quis iaculis feugiat, odio nisi porta lorem, id aliquet purus felis eget elit.

Vestibulum semper, lacus vitae fermentum ultricies, nisl lectus commodo eros, eget auctor justo turpis sit amet velit. Nunc sodales velit et ante semper ac vulputate libero semper. Phasellus ac ultrices nulla. Sed non massa nibh. Aenean placerat porttitor varius. Morbi diam libero, ullamcorper in imperdiet vel, cursus vel sem. Phasellus rhoncus, erat at mattis convallis, tortor nulla imperdiet urna, nec pulvinar mauris leo at magna. Pellentesque eget mauris massa.

Nullam vel scelerisque nulla. Ut vel arcu nunc, quis fermentum lectus. Aliquam nec nunc libero. Nulla convallis vestibulum molestie. Duis sagittis est ac arcu ultrices pulvinar. Suspendisse in turpis velit. Lorem ipsum dolor sit amet, consectetur adipiscing elit. Nullam placerat, arcu ut lacinia laoreet, magna nulla tincidunt arcu, a venenatis libero dui quis nisi. Proin porta augue sit amet quam varius pellentesque. Pellentesque varius consectetur vehicula. Suspendisse quis ipsum sapien, eu volutpat tortor. Duis vulputate porttitor dolor, id imperdiet dui sodales ut. Suspendisse nec pharetra eros. Vestibulum nec nibh arcu, sit amet tincidunt nibh. Duis ac erat eget dolor congue eleifend malesuada ut erat. Sed scelerisque, tellus a lobortis tristique, leo sapien mattis sem, non iaculis velit lorem vitae dui.

Nunc a urna lectus, vitae porta erat. Pellentesque non eros leo, vitae dictum sem. Donec pellentesque mauris id justo tristique ut adipiscing lacus suscipit. Aenean faucibus sodales sodales. Ut scelerisque dui ut elit pharetra sagittis. In viverra mattis placerat. Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas. Pellentesque iaculis nisi interdum lacus cursus vel porta lectus faucibus. Pellentesque congue interdum eros eget tempor. Proin quis nulla sit amet nisi tincidunt commodo et at augue. Sed pellentesque tortor ac turpis lobortis lacinia. Pellentesque suscipit, risus id auctor imperdiet, odio tortor adipiscing mauris, ut tincidunt lacus sem et massa. Class aptent taciti sociosqu ad litora torquent per conubia nostra, per inceptos himenaeos. Ut ullamcorper orci sit amet tellus bibendum lobortis.

Quisque pharetra faucibus elit eu ullamcorper. Aliquam a tellus vitae arcu sodales ultricies congue et metus. Vestibulum ipsum velit, ultrices sit amet molestie sed, dignissim porta turpis. Mauris bibendum, nisl et auctor venenatis, enim dolor molestie magna, vel tristique magna eros vitae ligula. Ut interdum bibendum purus, sit amet interdum libero iaculis sed. Phasellus quis dignissim nisi. Curabitur vel ipsum et tortor sodales lacinia dignissim id elit. Morbi tristique, magna sed facilisis imperdiet, turpis odio molestie elit, quis euismod purus lectus et risus. Nunc et ante neque, vitae malesuada orci. Aliquam sodales nunc in nulla mattis sed pellentesque nisl auctor. Nunc in mauris in lacus egestas sagittis sit amet in magna. Nulla adipiscing orci ac massa dictum bibendum. Proin nec tempus turpis. Curabitur in velit auctor turpis suscipit vulputate vel at odio. Pellentesque vulputate, nunc elementum gravida commodo, orci nulla ornare lectus, dignissim semper tortor velit non mi. Vivamus sit amet justo a sapien aliquet luctus.

Duis tincidunt vehicula lorem nec mollis. Praesent molestie ligula vel odio tincidunt ut iaculis urna congue. Praesent dapibus, justo sed convallis mollis, nulla nisl mattis diam, sed pulvinar nisi purus sit amet tortor. Maecenas semper, dolor sed blandit imperdiet, mauris sem volutpat risus, et vestibulum quam nibh ac nibh. Etiam lobortis sagittis nisi, quis condimentum ligula interdum et. Nam purus elit, porttitor et luctus vitae, iaculis id dolor. Nulla porttitor molestie leo vitae congue.

Aenean rhoncus odio non lacus mollis venenatis. Integer convallis aliquet laoreet. Vivamus sodales nunc eget mi posuere lobortis. Quisque iaculis accumsan arcu, nec dapibus nibh auctor eu. Praesent sem justo, varius ut auctor non, pharetra vitae odio. Proin consectetur pulvinar est at volutpat. Mauris posuere nunc eu ante vehicula at dapibus lectus elementum. Sed eu bibendum nunc. Nam sed lorem sem. Duis eget nisi at mauris vulputate pulvinar. Nulla nec ultricies erat. Nullam consectetur ullamcorper elit a luctus. Integer id eros augue. Duis at felis id turpis sollicitudin dignissim. Sed rhoncus massa vitae nisl vestibulum eleifend.

Cras consequat velit sit amet mauris viverra cursus. Quisque rhoncus euismod elementum. Sed consectetur interdum urna, vel mollis dui pharetra vel. Sed semper tempus massa tempor tristique. Mauris accumsan fringilla mi, non tempus purus feugiat sit amet. Morbi imperdiet mauris a mi tempus scelerisque. In hac habitasse platea dictumst. Donec eu mauris nibh. Phasellus id nisi libero, nec pulvinar metus. Suspendisse tincidunt gravida purus, et feugiat libero sollicitudin vitae.

Duis ipsum nulla, scelerisque nec scelerisque a, congue sit amet nulla. Integer risus urna, ultrices a tincidunt sit amet, fringilla nec metus. Donec lobortis blandit justo, in volutpat mauris posuere id. Nunc blandit dignissim tempus. Nam iaculis consequat odio, ac venenatis sem commodo vel. In non erat et nisl tempus fringilla. Nullam mattis cursus tortor aliquam viverra. Sed adipiscing, mi id aliquam volutpat, erat felis gravida odio, consectetur rutrum nulla ipsum et velit. Donec felis lorem, fringilla id egestas a, semper quis sapien. In id ligula ac mauris lobortis laoreet rutrum ac erat. Duis venenatis arcu ac sapien porttitor placerat lobortis neque semper. Ut pellentesque dolor non velit cursus condimentum non ut nunc.

Nunc feugiat enim non tellus ullamcorper eget tristique risus pretium. Phasellus nec ante lorem. Ut tincidunt auctor nulla, nec egestas libero tempus eu. Duis dignissim porttitor erat eget laoreet. Praesent convallis, erat nec vestibulum malesuada, mi nisi ultricies nulla, eu faucibus sem felis ut orci. Proin sed orci a mauris vestibulum facilisis. Duis dictum fringilla metus at volutpat. Aenean iaculis, ante non pulvinar dapibus, risus nulla ultricies magna, at adipiscing velit ante eget dui. Proin pharetra lorem vitae massa tincidunt non malesuada lorem lacinia. Sed eget ipsum vel orci feugiat condimentum malesuada in diam. Integer nunc est, dapibus sed sodales in, congue eget velit. Fusce consequat eros vitae est imperdiet aliquet et ornare libero. Suspendisse potenti. Sed nec dictum erat. Class aptent taciti sociosqu ad litora torquent per conubia nostra, per inceptos himenaeos. Nam auctor porta mi in convallis.

Sed tempor bibendum erat ac vulputate. Donec id nisi nec sem tempus congue. Suspendisse pulvinar dolor eu ante auctor tincidunt. Integer adipiscing, nisi vitae lobortis pellentesque, nulla justo pulvinar ante, in pellentesque sem dui eu tellus. Mauris ac lorem ac velit hendrerit euismod sit amet varius risus. Donec tristique, urna non tristique bibendum, augue libero placerat elit, non volutpat elit neque consequat dui. Aenean tempor accumsan mauris, eget luctus leo molestie nec. Etiam et dui ac enim condimentum iaculis. Ut molestie malesuada mauris, nec faucibus sem vehicula vel. Integer mauris purus, facilisis vel sagittis non, hendrerit ut purus. Vestibulum eget enim nec urna convallis pulvinar. Sed molestie nisl vel velit viverra tincidunt.

Praesent tristique egestas magna ac dictum. Ut commodo porta nulla non commodo. Duis tortor risus, euismod id vulputate non, sagittis vel diam. Proin mattis nunc eu enim auctor venenatis a ut eros. Suspendisse convallis, ligula nec euismod imperdiet, dui ligula posuere lectus, in tristique metus ipsum ut justo. Phasellus sollicitudin, odio eu elementum dictum, orci mauris euismod ante, vel posuere odio risus non diam. Morbi sed blandit massa. Nullam adipiscing egestas velit, sed rhoncus enim lobortis ac. Mauris eu sem magna, et elementum justo. Pellentesque nisi quam, ultrices nec molestie sit amet, dignissim mollis felis. Vivamus pharetra mi gravida enim vestibulum in convallis massa suscipit. Praesent luctus lorem vel leo fermentum in luctus justo dignissim.

Aliquam scelerisque lorem id mi pretium sed pharetra massa vulputate. Ut venenatis, ligula vel mattis viverra, odio leo posuere justo, vitae elementum elit nunc in dolor. Aliquam commodo elit quis lorem commodo egestas consequat elit tempus. Suspendisse a ante mi, vitae malesuada tellus. Cras ac dolor quis eros aliquet aliquet ac quis lacus. Nam a risus a sapien gravida placerat et sed nibh. Nulla augue erat, dignissim ac auctor quis, tempor eu dolor. Proin aliquam dignissim vehicula. Fusce eget enim in augue volutpat commodo nec eget libero. Fusce ut est eget ipsum tincidunt scelerisque eu at est. Fusce sagittis leo non ligula mattis vitae hendrerit risus accumsan. Cras non nunc lacus, in congue augue. In posuere diam orci, eget vulputate odio. Sed non dolor quam, quis congue enim.

Nunc pulvinar urna ut felis suscipit eget imperdiet augue interdum. Maecenas odio magna, aliquet eget adipiscing viverra, euismod eget odio. Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus. Quisque vel sem leo, at accumsan mi. Donec at tellus nisl. Praesent bibendum dictum risus, vel egestas nisi aliquet vel. Nam vitae ipsum sit amet lectus dictum congue. Etiam et massa felis. Curabitur id leo vestibulum nisi semper interdum. Nullam quis dui sed ante facilisis dignissim eget id ipsum. Duis aliquam risus sed arcu tempus pellentesque. Donec nec mi eget dui lacinia dapibus non sed nisl. Donec at mi eu lacus ornare feugiat. Praesent vel felis pretium velit vehicula sollicitudin in ac augue. Suspendisse at nunc mauris. Morbi dignissim suscipit nibh nec suscipit.

Nam tempor malesuada diam, non luctus purus aliquet eu. Maecenas tincidunt nisi eget felis ullamcorper vitae tristique odio ultrices. Nunc arcu leo, malesuada quis sed.
";

            service.AddOrUpdateCommunicationTemplate(facilityId, commTemplates[0]);
            service.AddOrUpdateCommunicationTemplate(facilityId, commTemplates[1]);

            var retrievedFacility = service.Get(facilityId);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {

                Assert.AreEqual(retrievedFacility.CommunicationTemplates.Count, 2);

                CompareObjects compareObjects = new CompareObjects();
                //                compareObjects.ElementsToIgnore.Add("Id");
                Assert.IsTrue(compareObjects.Compare(retrievedFacility.CommunicationTemplates[0], commTemplates[0]), compareObjects.DifferencesString);
                Assert.IsTrue(compareObjects.Compare(retrievedFacility.CommunicationTemplates[1], commTemplates[1]), compareObjects.DifferencesString);

                dbSession.Delete(retrievedFacility.CommunicationTemplates[0]);
                dbSession.Delete(retrievedFacility.CommunicationTemplates[0]);
                dbSession.Delete(retrievedFacility);
                dbSession.Delete(site);
                transaction.Commit();
            }
        }

    }
}
