﻿using MedRed.Services;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using MedRed.DataAccess;
using NHibernate;
using MedRed.MDWSAccess;
using Shared.Model.Config.MDWS;
using System.Collections.Generic;
using Shared.Model;
using KellermanSoftware.CompareNetObjects;

namespace MedRed.Services.Tests
{
    [TestClass()]
    public class CommunicationQueueTest
    {   
        private TestContext testContextInstance;

        /// <summary>
        ///Gets or sets the test context which provides
        ///information about and functionality for the current test run.
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        #region Additional test attributes
        // 
        //You can use the following additional attributes as you write your tests:
        //
        //Use ClassInitialize to run code before running the first test in the class
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //Use ClassCleanup to run code after all tests in a class have run
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            MDWSConfiguration config = ConnectionHelper.GetTestingConfig();
//            MDWSConnection connection = ConnectionHelper.GetConnectionFor("100");
            var factory = Factory.GetFactory(config);
            TestContext.Properties["Factory"] = factory;

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
                dbSession.Save(site);
                Facility facility = new Facility() { Name = "Facility 1", Site = site };
                dbSession.Save(facility);
                Section section = new Section() { Name = "Section A", Facility = facility };
                dbSession.Save(section);
                int sectionId = section.Id;

                Resource[] array = {new Resource() { Name = "Dr. Jones", Section = section, Type = ResourceType.Provider},
                                new Resource() { Name = "Room 12", Section = section, Type = ResourceType.Room}};

                dbSession.Save(array[0]);
                dbSession.Save(array[1]);

                TestContext.Properties["Site"] = site;
                TestContext.Properties["Facility"] = facility;
                TestContext.Properties["Section"] = section;
                TestContext.Properties["Resource1"] = array[0];
                TestContext.Properties["Resource2"] = array[1];

                Patient patient = new Patient()
                {
                    ICN = "124",
                    Person = new Person()
                    {
                        FirstName = "Henry",
                        LastName = "Jackson",
                        Sex = "M",
                        DateOfBirth = new DateTime(1980, 1, 1),
                        Email = "Jackson5@hotmail.com"
                    }
                };

                dbSession.Save(patient);

                TestContext.Properties["Patient"] = patient;

                Appointment appointment = new Appointment()
                {
                    Time = DateTime.Now.Date.AddHours(9),
                    Length = 60,
                    Patient = patient,
                    Status = AppointmentStatus.Scheduled,
                    Section = section
                };
                appointment.AddResource(array[0]);
                appointment.AddResource(array[1]);

                dbSession.Save(appointment);

                TestContext.Properties["Appt"] = appointment;

                transaction.Commit();
            }

        }

        //Use TestCleanup to run code after each test has run
        [TestCleanup()]
        public void MyTestCleanup()
        {
            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                var site = TestContext.Properties["Site"] as Site;
                var facility = TestContext.Properties["Facility"] as Facility;
                var section = TestContext.Properties["Section"] as Section;
                var resource1 = TestContext.Properties["Resource1"] as Resource;
                var resource2 = TestContext.Properties["Resource2"] as Resource;
                var patient = TestContext.Properties["Patient"] as Patient;
                var appt = TestContext.Properties["Appt"] as Appointment;

                dbSession.Delete(appt);
                dbSession.Delete(patient);
                dbSession.Delete(resource1);
                dbSession.Delete(resource2);
                dbSession.Delete(section);
                dbSession.Delete(facility);
                dbSession.Delete(site);

                transaction.Commit();
            }
        }

        #endregion


        [TestMethod()]
        public void AddToCommunicationQueueTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var facilityService = factory.GetFacilityService();
            var apptService = factory.GetAppointmentService();

            var facility = TestContext.Properties["Facility"] as Facility;
            var appt = TestContext.Properties["Appt"] as Appointment;

            var ct = new CommunicationTemplate()
            {
                Name = "Template 1",
                Type = CommunicationType.PreAppointment,
                Message = "Dear [PATIENTNAME], please get ready for your appointment",
                Recipient = "Patient"
            };

            facilityService.AddOrUpdateCommunicationTemplate(facility.Id, ct);

            TestContext.Properties["Facility"] = facilityService.Get(facility.Id);

            var newQueueItem = new CommunicationQueueItem()
            {
                Appointment = appt,
                Body = ct.Message,
                Medium = ct.Medium,
                Recipient = ct.Recipient,
                SendBy = DateTime.Today.AddDays(10),
                Subject = ct.Subject,
                Template = ct
            };

            apptService.AddToCommunicationQueue(newQueueItem);

            var communications = apptService.GetAllRelatedCommunications(appt);

            Assert.AreEqual(communications.Count, 1);

            CompareObjects compareObjects = new CompareObjects();
            compareObjects.ElementsToIgnore.Add("Facility");
            Assert.IsTrue(compareObjects.Compare(communications[0], newQueueItem), compareObjects.DifferencesString);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(newQueueItem);

                transaction.Commit();
            }

        }

        [TestMethod()]
        public void AddToCommunicationQueueForRequestTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var facilityService = factory.GetFacilityService();
            var apptService = factory.GetAppointmentService();

            var facility = TestContext.Properties["Facility"] as Facility;
            var patient = TestContext.Properties["Patient"] as Patient;
            var section = TestContext.Properties["Section"] as Section;

            var request = new AppointmentRequest()
            {
                Patient = patient,
                Time = new DateTime(2013, 5, 5, 12, 0, 0),
                DesiredDate = new DateTime(2013, 5, 1, 12, 0, 0),
                Priority = 2,
                Reason = RequestReason.Standard,
                RequestedSection = section,
                RequestMethod = RequestMethod.Email,
                Status = RequestStatus.Pending,
            };

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(request);

                transaction.Commit();
            }

            var ct = new CommunicationTemplate()
            {
                Name = "Template 1",
                Type = CommunicationType.WaitingList,
                Message = "Dear [PATIENT NAME], you have been placed on the waiting list for [CLINIC NAME]",
                Recipient = "Patient"
            };

            facilityService.AddOrUpdateCommunicationTemplate(facility.Id, ct);

            TestContext.Properties["Facility"] = facilityService.Get(facility.Id);

            var newQueueItem = apptService.AddToCommunicationQueue(ct, request);

            Assert.AreEqual(ct.Recipient, newQueueItem.Recipient);
            Assert.IsFalse(newQueueItem.Body.Contains("[PATIENT NAME]"));
            Assert.IsTrue(newQueueItem.Body.Contains("Jackson Henry"));
            Assert.IsFalse(newQueueItem.Body.Contains("[CLINIC NAME]"));
            Assert.IsTrue(newQueueItem.Body.Contains("Section A"));

            var communications = apptService.GetAllRelatedCommunications(request);

            Assert.AreEqual(communications.Count, 1);

            CompareObjects compareObjects = new CompareObjects();
            compareObjects.ElementsToIgnore.Add("Facility");
            Assert.IsTrue(compareObjects.Compare(communications[0], newQueueItem), compareObjects.DifferencesString);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(newQueueItem);
                dbSession.Delete(request);

                transaction.Commit();
            }
        }

        [TestMethod()]
        public void AddToCommunicationQueue2Test()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var facilityService = factory.GetFacilityService();
            var apptService = factory.GetAppointmentService();

            var facility = TestContext.Properties["Facility"] as Facility;
            var appt = TestContext.Properties["Appt"] as Appointment;

            var ct = new CommunicationTemplate()
            {
                Name = "Template 1",
                Type = CommunicationType.PreAppointment,
                Message = "Dear [PATIENT NAME], please get ready for your appointment at [CLINIC NAME]",
                Recipient = "Patient"
            };

            facilityService.AddOrUpdateCommunicationTemplate(facility.Id, ct);

            TestContext.Properties["Facility"] = facilityService.Get(facility.Id);

            var newQueueItem = apptService.AddToCommunicationQueue(ct, appt);

            Assert.AreEqual(ct.Recipient, newQueueItem.Recipient);
            Assert.IsFalse(newQueueItem.Body.Contains("[PATIENT NAME]"));   
            Assert.IsTrue(newQueueItem.Body.Contains("Jackson Henry"));
            Assert.IsFalse(newQueueItem.Body.Contains("[CLINIC NAME]"));
            Assert.IsTrue(newQueueItem.Body.Contains("Section A"));

            var communications = apptService.GetAllRelatedCommunications(appt);

            Assert.AreEqual(communications.Count, 1);

            CompareObjects compareObjects = new CompareObjects();
            compareObjects.ElementsToIgnore.Add("Facility");
            Assert.IsTrue(compareObjects.Compare(communications[0], newQueueItem), compareObjects.DifferencesString);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(newQueueItem);

                transaction.Commit();
            }

        }

    }
}
