﻿using MedRed.Services.ServiceImpl;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using Shared.Model.Config.MDWS;
using Shared.Model;
using KellermanSoftware.CompareNetObjects;
using System.Collections.Generic;

namespace MedRed.Services.Tests
{    
    /// <summary>
    ///This is a test class for AccountServiceTest and is intended
    ///to contain all AccountServiceTest Unit Tests
    ///</summary>
    [TestClass()]
    public class AppointmentServiceTest
    {
        private TestContext testContextInstance;

        /// <summary>
        ///Gets or sets the test context which provides
        ///information about and functionality for the current test run.
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        #region Additional test attributes
        // 
        //You can use the following additional attributes as you write your tests:
        //
        //Use ClassInitialize to run code before running the first test in the class
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //Use ClassCleanup to run code after all tests in a class have run
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //Use TestInitialize to run code before running each test
        [TestInitialize()]
        public void MyTestInitialize()
        {
            MDWSConfiguration config = ConnectionHelper.GetTestingConfig();
            var factory = Factory.GetFactory(config);
            TestContext.Properties["Factory"] = factory;

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
                dbSession.Save(site);
                Facility facility = new Facility() { Name = "Facility 1", Site = site };
                dbSession.Save(facility);
                Section section = new Section() { Name = "Section A", Facility = facility };
                dbSession.Save(section);
                int sectionId = section.Id;

                Resource[] array = {new Resource() { Name = "Dr. Jones", Section = section, Type = ResourceType.Provider},
                                new Resource() { Name = "Room 12", Section = section, Type = ResourceType.Room}};
                
                dbSession.Save(array[0]);
                dbSession.Save(array[1]);

                TestContext.Properties["Site"] = site;
                TestContext.Properties["Facility"] = facility;
                TestContext.Properties["Section"] = section;
                TestContext.Properties["Resource1"] = array[0];
                TestContext.Properties["Resource2"] =array[1];

                Patient patienta = new Patient()
                {
                    ICN = "124",
                    Person = new Person()
                    {
                        FirstName = "Henry",
                        LastName = "Jackson",
                        Sex = "M",
                        DateOfBirth = new DateTime(1980, 1, 1),
                        Email = "Jackson5@hotmail.com"
                    }
                };

                Patient patientb = new Patient()
                {
                    ICN = "124",
                    Person = new Person()
                    {
                        FirstName = "Henry",
                        LastName = "Jackson",
                        Sex = "M",
                        DateOfBirth = new DateTime(1980, 1, 1),
                        Email = "Jackson5@hotmail.com"
                    }
                };

                dbSession.Save(patienta);
                dbSession.Save(patientb);
            
                TestContext.Properties["PatientA"] = patienta;
                TestContext.Properties["PatientB"] = patientb;

                transaction.Commit();
            }
            var hierarchy = (log4net.Repository.Hierarchy.Hierarchy)log4net.LogManager.GetRepository();
            var logger = (log4net.Repository.Hierarchy.Logger)hierarchy.GetLogger("NHibernate.SQL");
            logger.AddAppender(new log4net.Appender.TraceAppender { Layout = new log4net.Layout.SimpleLayout() });
            hierarchy.Configured = true;
        }
        
        //Use TestCleanup to run code after each test has run
        [TestCleanup()]
        public void MyTestCleanup()
        {
            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                var site = TestContext.Properties["Site"] as Site;
                var facility = TestContext.Properties["Facility"] as Facility;
                var section = TestContext.Properties["Section"] as Section;
                var resource1 = TestContext.Properties["Resource1"] as Resource;
                var resource2 = TestContext.Properties["Resource2"] as Resource;
                var patienta = TestContext.Properties["PatientA"] as Patient;
                var patientb = TestContext.Properties["PatientB"] as Patient;

                dbSession.Delete(patienta);
                dbSession.Delete(patientb);
                dbSession.Delete(resource1);
                dbSession.Delete(resource2);
                dbSession.Delete(section);
                dbSession.Delete(facility);
                dbSession.Delete(site);

                transaction.Commit();
            }
        }
        
        #endregion

        [TestMethod()]
        public void GetAppointmentTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();

            var patient = TestContext.Properties["PatientA"] as Patient;
            var resource1 = TestContext.Properties["Resource1"] as Resource;
            var resource2 = TestContext.Properties["Resource2"] as Resource;

            Appointment appointment = new Appointment()
            {
                Time = DateTime.Now.Date.AddHours(9),
                Length = 60,
                Patient = patient,
                Status = AppointmentStatus.Scheduled,
            };
            appointment.AddResource(resource1);
            appointment.AddResource(resource2);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(appointment);
                transaction.Commit();
            }
            int appointmentId = appointment.Id;

            Assert.AreNotEqual(appointmentId, 0);

            var retrieved = service.Get(appointmentId);

            CompareObjects compareObjects = new CompareObjects();
            Assert.IsTrue(compareObjects.Compare(retrieved, appointment), compareObjects.DifferencesString);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(appointment);

                transaction.Commit();
            }
        }

        [TestMethod()]
        public void GetAppointmentsForPatientTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();

            var patienta = TestContext.Properties["PatientA"] as Patient;
            var patientb = TestContext.Properties["PatientB"] as Patient;
            var resource1 = TestContext.Properties["Resource1"] as Resource;
            var resource2 = TestContext.Properties["Resource2"] as Resource;

            Appointment appointment1 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 9, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment1.AddResource(resource1);
            appointment1.AddResource(resource2);

            Appointment appointment2 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 12, 0, 0),
                Length = 60,
                Patient = patientb,
                Status = AppointmentStatus.Scheduled,
            };
            appointment2.AddResource(resource1);
            appointment2.AddResource(resource2);
            
            Appointment appointment3 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 12, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment3.AddResource(resource1);
            appointment3.AddResource(resource2);

            Appointment appointment4 = new Appointment()
            {
                Time = new DateTime(2013, 5, 1, 9, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment4.AddResource(resource1);
            appointment4.AddResource(resource2);


            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(appointment1);
                dbSession.Save(appointment2);
                dbSession.Save(appointment3);
                dbSession.Save(appointment4);
                transaction.Commit();
            }

            var retrieved = service.GetForPatient(patienta.Id, new DateTime(2013, 4, 30), new DateTime(2013, 5, 1));

            Assert.AreEqual(retrieved.Count, 2);
            var retrievedList = new List<Appointment>(retrieved);

            Assert.IsTrue(retrievedList.Exists(x => x.Id == appointment1.Id));
            Assert.IsFalse(retrievedList.Exists(x => x.Id == appointment2.Id));
            Assert.IsTrue(retrievedList.Exists(x => x.Id == appointment3.Id));
            Assert.IsFalse(retrievedList.Exists(x => x.Id == appointment4.Id));

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(appointment1);
                dbSession.Delete(appointment2);
                dbSession.Delete(appointment3);
                dbSession.Delete(appointment4);

                transaction.Commit();
            }
        }

        [TestMethod()]
        public void GetNoShowAppointmentsForPatientTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();

            var patienta = TestContext.Properties["PatientA"] as Patient;
            var patientb = TestContext.Properties["PatientB"] as Patient;
            var resource1 = TestContext.Properties["Resource1"] as Resource;
            var resource2 = TestContext.Properties["Resource2"] as Resource;

            Appointment appointment1 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 9, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment1.AddResource(resource1);
            appointment1.AddResource(resource2);

            Appointment appointment2 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 12, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment2.AddResource(resource1);
            appointment2.AddResource(resource2);

            Appointment appointment3 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 12, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment3.AddResource(resource1);
            appointment3.AddResource(resource2);

            Appointment appointment4 = new Appointment()
            {
                Time = new DateTime(2013, 5, 1, 9, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment4.AddResource(resource1);
            appointment4.AddResource(resource2);

            appointment2.AddActivity(new AppointmentActivity() { ActivityType = ActivityType.NoShow, Time = DateTime.Now });
            appointment3.AddActivity(new AppointmentActivity() { ActivityType = ActivityType.NoShow, Time = DateTime.Now });


            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(appointment1);
                dbSession.Save(appointment2);
                dbSession.Save(appointment3);
                dbSession.Save(appointment4);
                transaction.Commit();
            }

            var retrieved = service.GetNoShowAppointmentsForPatient(patienta.Id, new DateTime(2013, 4, 30), new DateTime(2013, 5, 5));

            Assert.AreEqual(retrieved.Count, 2);
            var retrievedList = new List<Appointment>(retrieved);

            Assert.IsFalse(retrievedList.Exists(x => x.Id == appointment1.Id));
            Assert.IsTrue(retrievedList.Exists(x => x.Id == appointment2.Id));
            Assert.IsTrue(retrievedList.Exists(x => x.Id == appointment3.Id));
            Assert.IsFalse(retrievedList.Exists(x => x.Id == appointment4.Id));

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(appointment1);
                dbSession.Delete(appointment2);
                dbSession.Delete(appointment3);
                dbSession.Delete(appointment4);

                transaction.Commit();
            }
        }

        [TestMethod()]
        public void GetAppointmentsForResourceTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();

            var patienta = TestContext.Properties["PatientA"] as Patient;
            var patientb = TestContext.Properties["PatientB"] as Patient;
            var resource1 = TestContext.Properties["Resource1"] as Resource;
            var resource2 = TestContext.Properties["Resource2"] as Resource;

            Appointment appointment1 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 9, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment1.AddResource(resource1);

            Appointment appointment2 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 12, 0, 0),
                Length = 60,
                Patient = patientb,
                Status = AppointmentStatus.Scheduled,
            };
            appointment2.AddResource(resource1);
            appointment2.AddResource(resource2);

            Appointment appointment3 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 12, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment3.AddResource(resource2);

            Appointment appointment4 = new Appointment()
            {
                Time = new DateTime(2013, 5, 1, 9, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment4.AddResource(resource1);
            appointment4.AddResource(resource2);


            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(appointment1);
                dbSession.Save(appointment2);
                dbSession.Save(appointment3);
                dbSession.Save(appointment4);
                transaction.Commit();
            }

            var retrieved = service.GetForResource(resource1.Id, new DateTime(2013, 4, 1), new DateTime(2013, 5, 30));
            var retrievedList = new List<Appointment>(retrieved);

            Assert.AreEqual(retrieved.Count, 3);
            Assert.IsTrue(retrievedList.Exists(x => x.Id == appointment1.Id));
            Assert.IsTrue(retrievedList.Exists(x => x.Id == appointment2.Id));
            Assert.IsFalse(retrievedList.Exists(x => x.Id == appointment3.Id));
            Assert.IsTrue(retrievedList.Exists(x => x.Id == appointment4.Id));

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(appointment1);
                dbSession.Delete(appointment2);
                dbSession.Delete(appointment3);
                dbSession.Delete(appointment4);

                transaction.Commit();
            }
        }

        [TestMethod()]
        public void GetAppointmentsForResourceNullStartTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();

            var patienta = TestContext.Properties["PatientA"] as Patient;
            var patientb = TestContext.Properties["PatientB"] as Patient;
            var resource1 = TestContext.Properties["Resource1"] as Resource;
            var resource2 = TestContext.Properties["Resource2"] as Resource;

            Appointment appointment1 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 9, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment1.AddResource(resource1);

            Appointment appointment2 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 12, 0, 0),
                Length = 60,
                Patient = patientb,
                Status = AppointmentStatus.Scheduled,
            };
            appointment2.AddResource(resource1);
            appointment2.AddResource(resource2);

            Appointment appointment3 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 12, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment3.AddResource(resource2);

            Appointment appointment4 = new Appointment()
            {
                Time = new DateTime(2013, 5, 1, 9, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment4.AddResource(resource1);
            appointment4.AddResource(resource2);


            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(appointment1);
                dbSession.Save(appointment2);
                dbSession.Save(appointment3);
                dbSession.Save(appointment4);
                transaction.Commit();
            }

            var retrieved = service.GetForResource(resource1.Id, null, new DateTime(2013, 5, 30));
            var retrievedList = new List<Appointment>(retrieved);

            Assert.AreEqual(retrieved.Count, 3);
            Assert.IsTrue(retrievedList.Exists(x => x.Id == appointment1.Id));
            Assert.IsTrue(retrievedList.Exists(x => x.Id == appointment2.Id));
            Assert.IsFalse(retrievedList.Exists(x => x.Id == appointment3.Id));
            Assert.IsTrue(retrievedList.Exists(x => x.Id == appointment4.Id));

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(appointment1);
                dbSession.Delete(appointment2);
                dbSession.Delete(appointment3);
                dbSession.Delete(appointment4);

                transaction.Commit();
            }
        }

        [TestMethod()]
        public void GetAppointmentsForResourceNullEndTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();

            var patienta = TestContext.Properties["PatientA"] as Patient;
            var patientb = TestContext.Properties["PatientB"] as Patient;
            var resource1 = TestContext.Properties["Resource1"] as Resource;
            var resource2 = TestContext.Properties["Resource2"] as Resource;

            Appointment appointment1 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 9, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment1.AddResource(resource1);

            Appointment appointment2 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 12, 0, 0),
                Length = 60,
                Patient = patientb,
                Status = AppointmentStatus.Scheduled,
            };
            appointment2.AddResource(resource1);
            appointment2.AddResource(resource2);

            Appointment appointment3 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 12, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment3.AddResource(resource2);

            Appointment appointment4 = new Appointment()
            {
                Time = new DateTime(2013, 5, 1, 9, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment4.AddResource(resource1);
            appointment4.AddResource(resource2);


            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(appointment1);
                dbSession.Save(appointment2);
                dbSession.Save(appointment3);
                dbSession.Save(appointment4);
                transaction.Commit();
            }

            var retrieved = service.GetForResource(resource1.Id, new DateTime(2013, 4, 1), null);
            var retrievedList = new List<Appointment>(retrieved);

            Assert.AreEqual(retrieved.Count, 3);
            Assert.IsTrue(retrievedList.Exists(x => x.Id == appointment1.Id));
            Assert.IsTrue(retrievedList.Exists(x => x.Id == appointment2.Id));
            Assert.IsFalse(retrievedList.Exists(x => x.Id == appointment3.Id));
            Assert.IsTrue(retrievedList.Exists(x => x.Id == appointment4.Id));

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(appointment1);
                dbSession.Delete(appointment2);
                dbSession.Delete(appointment3);
                dbSession.Delete(appointment4);

                transaction.Commit();
            }
        }

        [TestMethod()]
        public void GetAppointmentsForResourceOnlyScheduledTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();

            var patienta = TestContext.Properties["PatientA"] as Patient;
            var patientb = TestContext.Properties["PatientB"] as Patient;
            var resource1 = TestContext.Properties["Resource1"] as Resource;
            var resource2 = TestContext.Properties["Resource2"] as Resource;

            Appointment appointment1 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 9, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment1.AddResource(resource1);

            Appointment appointment2 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 12, 0, 0),
                Length = 60,
                Patient = patientb,
                Status = AppointmentStatus.Closed,
            };
            appointment2.AddResource(resource1);
            appointment2.AddResource(resource2);

            Appointment appointment3 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 12, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment3.AddResource(resource2);

            Appointment appointment4 = new Appointment()
            {
                Time = new DateTime(2013, 5, 1, 9, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment4.AddResource(resource1);
            appointment4.AddResource(resource2);


            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(appointment1);
                dbSession.Save(appointment2);
                dbSession.Save(appointment3);
                dbSession.Save(appointment4);
                transaction.Commit();
            }

            var retrieved = service.GetForResource(resource1.Id, new DateTime(2013, 4, 1), new DateTime(2013, 5, 30));
            var retrievedList = new List<Appointment>(retrieved);

            var retrievedScheduled = service.GetForResource(resource1.Id, new DateTime(2013, 4, 1), new DateTime(2013, 5, 30), AppointmentStatus.Scheduled);
            var retrievedScheduledList = new List<Appointment>(retrievedScheduled);

            Assert.AreEqual(retrievedScheduled.Count, 2);
            Assert.IsTrue(retrievedScheduledList.Exists(x => x.Id == appointment1.Id));
            Assert.IsFalse(retrievedScheduledList.Exists(x => x.Id == appointment2.Id));
            Assert.IsFalse(retrievedScheduledList.Exists(x => x.Id == appointment3.Id));
            Assert.IsTrue(retrievedScheduledList.Exists(x => x.Id == appointment4.Id));

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(appointment1);
                dbSession.Delete(appointment2);
                dbSession.Delete(appointment3);
                dbSession.Delete(appointment4);

                transaction.Commit();
            }
        }

        [TestMethod()]
        public void UpdateAppointmentTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();

            var patient = TestContext.Properties["PatientA"] as Patient;
            var resource1 = TestContext.Properties["Resource1"] as Resource;
            var resource2 = TestContext.Properties["Resource2"] as Resource;

            Appointment appointment = new Appointment()
            {
                Time = DateTime.Now.Date.AddHours(9),
                Length = 60,
                Patient = patient,
                Status = AppointmentStatus.Scheduled,
            };
            appointment.AddResource(resource1);
            appointment.AddResource(resource2);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(appointment);
                transaction.Commit();
            }
            int appointmentId = appointment.Id;

            Assert.AreNotEqual(appointmentId, 0);

            var original = service.Get(appointmentId);

            original.Time = DateTime.Now.Date.AddHours(10);
            original.Length = 30;

            service.Update(original);

            var retrieved = service.Get(appointmentId);

            Assert.AreEqual(retrieved.Time, DateTime.Now.Date.AddHours(10));
            Assert.AreEqual(retrieved.Length, 30);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(retrieved);

                transaction.Commit();
            }
        }

        [TestMethod()]
        public void DeleteAppointmentDoesntDeletePatientTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();

            var patient = TestContext.Properties["PatientA"] as Patient;
            var resource1 = TestContext.Properties["Resource1"] as Resource;
            var resource2 = TestContext.Properties["Resource2"] as Resource;

            Appointment appointment = new Appointment()
            {
                Time = DateTime.Now.Date.AddHours(9),
                Length = 60,
                Patient = patient,
                Status = AppointmentStatus.Scheduled,
            };
            appointment.AddResource(resource1);
            appointment.AddResource(resource2);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(appointment);
                transaction.Commit();
            }

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(appointment);

                transaction.Commit();
            }

            var retrievedPatient = factory.GetPatientService().Get(patient.Id);

            CompareObjects compareObjects = new CompareObjects();
            Assert.IsTrue(compareObjects.Compare(retrievedPatient, patient), compareObjects.DifferencesString);
        }

        [TestMethod()]
        public void DeleteAppointmentDoesntDeleteResourcesTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();

            var patient = TestContext.Properties["PatientA"] as Patient;
            var resource1 = TestContext.Properties["Resource1"] as Resource;
            var resource2 = TestContext.Properties["Resource2"] as Resource;

            Appointment appointment = new Appointment()
            {
                Time = DateTime.Now.Date.AddHours(9),
                Length = 60,
                Patient = patient,
                Status = AppointmentStatus.Scheduled,
            };
            appointment.AddResource(resource1);
            appointment.AddResource(resource2);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(appointment);
                transaction.Commit();
            }

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(appointment);

                transaction.Commit();
            }

            var retrievedResource = factory.GetResourceService().Get(resource1.Id);

            CompareObjects compareObjects = new CompareObjects();
            Assert.IsTrue(compareObjects.Compare(retrievedResource, resource1), compareObjects.DifferencesString);
        }

        [TestMethod()]
        public void AddRelatedAppointmentTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();

            var patienta = TestContext.Properties["PatientA"] as Patient;
            var patientb = TestContext.Properties["PatientB"] as Patient;
            var resource1 = TestContext.Properties["Resource1"] as Resource;
            var resource2 = TestContext.Properties["Resource2"] as Resource;

            Appointment appointment1 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 9, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment1.AddResource(resource1);
            appointment1.AddResource(resource2);

            Appointment appointment2 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 12, 0, 0),
                Length = 60,
                Patient = patientb,
                Status = AppointmentStatus.Scheduled,
            };
            appointment2.AddResource(resource1);
            appointment2.AddResource(resource2);

            Appointment appointment3 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 12, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment3.AddResource(resource1);
            appointment3.AddResource(resource2);

            Appointment appointment4 = new Appointment()
            {
                Time = new DateTime(2013, 5, 1, 9, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment4.AddResource(resource1);
            appointment4.AddResource(resource2);


            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(appointment1);
                dbSession.Save(appointment2);
                dbSession.Save(appointment3);
                dbSession.Save(appointment4);
                transaction.Commit();
            }

            var appt1 = service.Get(appointment1.Id);
            var appt2 = service.Get(appointment2.Id);
            var appt3 = service.Get(appointment3.Id);
            var appt4 = service.Get(appointment4.Id);

            appt1.AddChildAppointment(appt3, AppointmentRelationType.Followup);
            appt1.AddChildAppointment(appt4, AppointmentRelationType.Followup);

            service.Update(appt1);

            var retrievedAppt = service.Get(appt1.Id);

            Assert.AreEqual(retrievedAppt.ChildAppointments.Count, 2);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(appt1);
                dbSession.Delete(appt2);
                dbSession.Delete(appt3);
                dbSession.Delete(appt4);

                transaction.Commit();
            }
        }

        [TestMethod()]
        public void GetAllRelatedAppointmentsTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();

            var patienta = TestContext.Properties["PatientA"] as Patient;
            var patientb = TestContext.Properties["PatientB"] as Patient;
            var resource1 = TestContext.Properties["Resource1"] as Resource;
            var resource2 = TestContext.Properties["Resource2"] as Resource;

            Appointment appointment1 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 9, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment1.AddResource(resource1);
            appointment1.AddResource(resource2);

            Appointment appointment2 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 12, 0, 0),
                Length = 60,
                Patient = patientb,
                Status = AppointmentStatus.Scheduled,
            };
            appointment2.AddResource(resource1);
            appointment2.AddResource(resource2);

            Appointment appointment3 = new Appointment()
            {
                Time = new DateTime(2013, 4, 30, 12, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment3.AddResource(resource1);
            appointment3.AddResource(resource2);

            Appointment appointment4 = new Appointment()
            {
                Time = new DateTime(2013, 5, 1, 9, 0, 0),
                Length = 60,
                Patient = patienta,
                Status = AppointmentStatus.Scheduled,
            };
            appointment4.AddResource(resource1);
            appointment4.AddResource(resource2);


            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(appointment1);
                dbSession.Save(appointment2);
                dbSession.Save(appointment3);
                dbSession.Save(appointment4);
                transaction.Commit();
            }

            var appt1 = service.Get(appointment1.Id);
            var appt2 = service.Get(appointment2.Id);
            var appt3 = service.Get(appointment3.Id);
            var appt4 = service.Get(appointment4.Id);

            appt1.AddChildAppointment(appt3, AppointmentRelationType.Followup);
            appt1.AddChildAppointment(appt4, AppointmentRelationType.Followup);

            service.Update(appt1);

            var appt1List = service.GetAllRelatedAppointments(appt1);
            Assert.AreEqual(appt1List.Count, 2);

            var appt2List = service.GetAllRelatedAppointments(appt2);
            Assert.AreEqual(appt2List.Count, 0);

            var appt3List = service.GetAllRelatedAppointments(appt3);
            Assert.AreEqual(appt3List.Count, 1);

            var appt4List = service.GetAllRelatedAppointments(appt4);
            Assert.AreEqual(appt4List.Count, 1);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(appt1);
                dbSession.Delete(appt2);
                dbSession.Delete(appt3);
                dbSession.Delete(appt4);

                transaction.Commit();
            }
        }

    }
}
