﻿using MedRed.Services.ServiceImpl;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Linq;
using Shared.Model.Config.MDWS;
using Shared.Model;
using KellermanSoftware.CompareNetObjects;
using MedRed.Services.Interfaces;
using System.Collections.Generic;

namespace MedRed.Services.Tests
{    
    /// <summary>
    ///This is a test class for AccountServiceTest and is intended
    ///to contain all AccountServiceTest Unit Tests
    ///</summary>
    [TestClass()]
    public class AppointmentRequestTest
    {
        private TestContext testContextInstance;

        /// <summary>
        ///Gets or sets the test context which provides
        ///information about and functionality for the current test run.
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        #region Additional test attributes
        // 
        //You can use the following additional attributes as you write your tests:
        //
        //Use ClassInitialize to run code before running the first test in the class
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //Use ClassCleanup to run code after all tests in a class have run
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //Use TestInitialize to run code before running each test
        [TestInitialize()]
        public void MyTestInitialize()
        {
            MDWSConfiguration config = ConnectionHelper.GetTestingConfig();
//            MDWSConnection connection = ConnectionHelper.GetConnectionFor("100");
            var factory = Factory.GetFactory(config);
            TestContext.Properties["Factory"] = factory;

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
                dbSession.Save(site);
                Facility facility = new Facility() { Name = "Facility 1", Site = site };
                dbSession.Save(facility);
                Section section = new Section() { Name = "Section A", Facility = facility };
                dbSession.Save(section);

                Section section2 = new Section() { Name = "Section B", Facility = facility };
                dbSession.Save(section2);

                int sectionId = section.Id;

                Resource[] array = {new Resource() { Name = "Dr. Jones", Section = section, Type = ResourceType.Provider},
                                new Resource() { Name = "Room 12", Section = section, Type = ResourceType.Room}};
                
                dbSession.Save(array[0]);
                dbSession.Save(array[1]);

                TestContext.Properties["Site"] = site;
                TestContext.Properties["Facility"] = facility;
                TestContext.Properties["Section"] = section;
                TestContext.Properties["Section2"] = section2;
                TestContext.Properties["Resource1"] = array[0];
                TestContext.Properties["Resource2"] = array[1];

                Patient patient = new Patient()
                {
                    ICN = "124",
                    Person = new Person()
                    {
                        FirstName = "Henry",
                        LastName = "Jackson",
                        Sex = "M",
                        DateOfBirth = new DateTime(1980, 1, 1),
                        Email = "Jackson5@hotmail.com"
                    }
                };

                dbSession.Save(patient);
            
                TestContext.Properties["Patient"] = patient;

                Appointment appointment = new Appointment()
                {
                    Time = DateTime.Now.Date.AddHours(9),
                    Length = 60,
                    Patient = patient,
                    Status = AppointmentStatus.Scheduled,
                };
                appointment.AddResource(array[0]);
                appointment.AddResource(array[1]);

                dbSession.Save(appointment);

                TestContext.Properties["Appt"] = appointment;

                transaction.Commit();
            }

        }
        
        //Use TestCleanup to run code after each test has run
        [TestCleanup()]
        public void MyTestCleanup()
        {
            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                var site = TestContext.Properties["Site"] as Site;
                var facility = TestContext.Properties["Facility"] as Facility;
                var section = TestContext.Properties["Section"] as Section;
                var section2 = TestContext.Properties["Section2"] as Section;
                var resource1 = TestContext.Properties["Resource1"] as Resource;
                var resource2 = TestContext.Properties["Resource2"] as Resource;
                var patient = TestContext.Properties["Patient"] as Patient;
                var appt = TestContext.Properties["Appt"] as Appointment;

                dbSession.Delete(appt);
                dbSession.Delete(patient);
                dbSession.Delete(resource1);
                dbSession.Delete(resource2);
                dbSession.Delete(section);
                dbSession.Delete(section2);
                dbSession.Delete(facility);
                dbSession.Delete(site);

                transaction.Commit();
            }
        }
        
        #endregion

        [TestMethod()]
        public void AddAppointmentRequestTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();

            var site = TestContext.Properties["Site"] as Site;
            var facility = TestContext.Properties["Facility"] as Facility;
            var section = TestContext.Properties["Section"] as Section;
            var resource1 = TestContext.Properties["Resource1"] as Resource;
            var resource2 = TestContext.Properties["Resource2"] as Resource;
            var patient = TestContext.Properties["Patient"] as Patient;
            var appt = TestContext.Properties["Appt"] as Appointment;

            var request = new AppointmentRequest()
            {
                Patient = patient,
                Time = new DateTime(2013, 4, 30, 10, 9, 8),
                DesiredDate = new DateTime(2013, 6, 1, 10, 0, 0),
                RequestedResource = resource1,
                RequestMethod = RequestMethod.Email,
                Status = RequestStatus.Pending,
                Notes = "Need to have leg looked at"
            };

            service.AddAppointmentRequest(request);

            var retrieved = service.GetAppointmentRequestsForPatient(patient);
            Assert.AreEqual(retrieved.Count, 1);

            CompareObjects compareObjects = new CompareObjects();
            Assert.IsTrue(compareObjects.Compare(retrieved[0], request), compareObjects.DifferencesString);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(retrieved[0]);

                transaction.Commit();
            }
        }

        [TestMethod()]
        public void UpdateAppointmentRequestTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();

            var site = TestContext.Properties["Site"] as Site;
            var facility = TestContext.Properties["Facility"] as Facility;
            var section = TestContext.Properties["Section"] as Section;
            var resource1 = TestContext.Properties["Resource1"] as Resource;
            var resource2 = TestContext.Properties["Resource2"] as Resource;
            var patient = TestContext.Properties["Patient"] as Patient;
            var appt = TestContext.Properties["Appt"] as Appointment;

            var request = new AppointmentRequest()
            {
                Patient = patient,
                Time = DateTime.Now,
                DesiredDate = new DateTime(2013, 6, 1, 10, 0, 0),
                RequestedResource = resource1,
                RequestMethod = RequestMethod.Email,
                Status = RequestStatus.Pending,
                Notes = "Need to have leg looked at"
            };

            service.AddAppointmentRequest(request);

            var retrieved = service.GetAppointmentRequestsForPatient(patient);
            Assert.AreEqual(retrieved.Count, 1);

            retrieved[0].Status = RequestStatus.Complete;
            retrieved[0].ResultingAppointment = appt;

            service.UpdateAppointmentRequest(retrieved[0]);

            var updated = service.GetAppointmentRequestsForPatient(patient);
            Assert.AreEqual(updated.Count, 1);

            Assert.AreEqual(updated[0].Status, RequestStatus.Complete);
            Assert.IsNotNull(updated[0].ResultingAppointment);
            
            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(updated[0]);

                transaction.Commit();
            }
        }

        [TestMethod()]
        public void GetAppointmentRequestsForSectionTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();

            var site = TestContext.Properties["Site"] as Site;
            var facility = TestContext.Properties["Facility"] as Facility;
            var section = TestContext.Properties["Section"] as Section;
            var section2 = TestContext.Properties["Section2"] as Section;
            var resource1 = TestContext.Properties["Resource1"] as Resource;
            var resource2 = TestContext.Properties["Resource2"] as Resource;
            var patient = TestContext.Properties["Patient"] as Patient;
            var appt = TestContext.Properties["Appt"] as Appointment;

            var request = new AppointmentRequest()
            {
                Patient = patient,
                Time = new DateTime(2013, 4, 1, 10, 5, 0),
                DesiredDate = new DateTime(2013, 6, 1, 10, 0, 0),
                RequestedResource = resource1,
                RequestMethod = RequestMethod.Email,
                Status = RequestStatus.Pending,
                Notes = "Need to have leg looked at"
            };

            var request2 = new AppointmentRequest()
            {
                Patient = patient,
                Time = new DateTime(2013, 4, 2, 14, 30, 0),
                DesiredDate = new DateTime(2013, 6, 1, 10, 0, 0),
                RequestedSection = section2,
                RequestMethod = RequestMethod.Email,
                Status = RequestStatus.Pending,
                Notes = "Need to have leg looked at"
            };

            service.AddAppointmentRequest(request);
            service.AddAppointmentRequest(request2);

            var requestsFor1 = service.GetAllOpenAppointmentRequests(section);
            Assert.AreEqual(requestsFor1.Count, 1);

            CompareObjects compareObjects = new CompareObjects();
            Assert.IsTrue(compareObjects.Compare(requestsFor1[0], request), compareObjects.DifferencesString);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(request);
                dbSession.Delete(request2);

                transaction.Commit();
            }

        }

        [TestMethod()]
        public void GetAppointmentRequestsForPatientTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();

            var site = TestContext.Properties["Site"] as Site;
            var facility = TestContext.Properties["Facility"] as Facility;
            var section = TestContext.Properties["Section"] as Section;
            var section2 = TestContext.Properties["Section2"] as Section;
            var resource1 = TestContext.Properties["Resource1"] as Resource;
            var resource2 = TestContext.Properties["Resource2"] as Resource;
            var patient = TestContext.Properties["Patient"] as Patient;
            var appt = TestContext.Properties["Appt"] as Appointment;

            var request = new AppointmentRequest()
            {
                Patient = patient,
                Time = new DateTime(2013, 4, 1, 10, 5, 0),
                DesiredDate = new DateTime(2013, 6, 1, 10, 0, 0),
                RequestedResource = resource1,
                RequestMethod = RequestMethod.Email,
                Status = RequestStatus.Pending,
                Notes = "Need to have leg looked at"
            };

            var request2 = new AppointmentRequest()
            {
                Patient = patient,
                Time = new DateTime(2013, 4, 2, 14, 30, 0),
                DesiredDate = new DateTime(2013, 6, 1, 10, 0, 0),
                RequestedSection = section2,
                RequestMethod = RequestMethod.Email,
                Status = RequestStatus.Pending,
                Notes = "Need to have leg looked at"
            };

            service.AddAppointmentRequest(request);
            service.AddAppointmentRequest(request2);

            var requestsFor1 = service.GetAppointmentRequestsForPatient(patient);
            Assert.AreEqual(requestsFor1.Count, 2);

            CompareObjects compareObjects = new CompareObjects();
            Assert.IsTrue(compareObjects.Compare(requestsFor1[0], request), compareObjects.DifferencesString);
            Assert.IsTrue(compareObjects.Compare(requestsFor1[1], request2), compareObjects.DifferencesString);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(request);
                dbSession.Delete(request2);

                transaction.Commit();
            }

        }

        //[TestMethod()]
        //public void GetWaitListForSectionTest()
        //{
        //    var factory = TestContext.Properties["Factory"] as Factory;
        //    var service = factory.GetAppointmentService();

        //    var site = TestContext.Properties["Site"] as Site;
        //    var facility = TestContext.Properties["Facility"] as Facility;
        //    var section = TestContext.Properties["Section"] as Section;
        //    var section2 = TestContext.Properties["Section2"] as Section;
        //    var resource1 = TestContext.Properties["Resource1"] as Resource;
        //    var resource2 = TestContext.Properties["Resource2"] as Resource;
        //    var patient = TestContext.Properties["Patient"] as Patient;
        //    var appt = TestContext.Properties["Appt"] as Appointment;

        //    var request = new AppointmentRequest()
        //    {
        //        Patient = patient,
        //        Time = new DateTime(2013, 4, 1, 10, 5, 0),
        //        DesiredDate = new DateTime(2013, 6, 1, 10, 0, 0),
        //        RequestedResource = resource1,
        //        RequestMethod = RequestMethod.Email,
        //        Status = RequestStatus.Pending,
        //        Notes = "Need to have leg looked at",
        //        Reason = RequestReason.Standard
        //    };

        //    var request2 = new AppointmentRequest()
        //    {
        //        Patient = patient,
        //        Time = new DateTime(2013, 4, 2, 14, 30, 0),
        //        DesiredDate = new DateTime(2013, 6, 1, 10, 0, 0),
        //        RequestedSection = section2,
        //        RequestMethod = RequestMethod.Email,
        //        Status = RequestStatus.Pending,
        //        Notes = "Need to have leg looked at",
        //        Reason = RequestReason.Standard

        //    };

        //    service.AddAppointmentRequest(request);
        //    service.AddAppointmentRequest(request);

        //    var requestsFor1 = service.GetAllOpenAppointmentRequests(section);
        //    Assert.AreEqual(requestsFor1.Count, 1);

        //    CompareObjects compareObjects = new CompareObjects();
        //    Assert.IsTrue(compareObjects.Compare(requestsFor1[0], request), compareObjects.DifferencesString);

        //    using (var dbSession = DataAccess.DataAccess.OpenSession())
        //    using (var transaction = dbSession.BeginTransaction())
        //    {
        //        dbSession.Delete(request);
        //        dbSession.Delete(request2);

        //        transaction.Commit();
        //    }

        //}

    }
}
