﻿using MedRed.Services.ServiceImpl;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Linq;
using Shared.Model.Config.MDWS;
using Shared.Model;
using KellermanSoftware.CompareNetObjects;
using MedRed.Services.Interfaces;
using System.Collections.Generic;

namespace MedRed.Services.Tests
{    
    /// <summary>
    ///This is a test class for AccountServiceTest and is intended
    ///to contain all AccountServiceTest Unit Tests
    ///</summary>
    [TestClass()]
    public class AppointmentDoActionServiceTest
    {
        private TestContext testContextInstance;

        /// <summary>
        ///Gets or sets the test context which provides
        ///information about and functionality for the current test run.
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        #region Additional test attributes
        // 
        //You can use the following additional attributes as you write your tests:
        //
        //Use ClassInitialize to run code before running the first test in the class
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //Use ClassCleanup to run code after all tests in a class have run
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //Use TestInitialize to run code before running each test
        [TestInitialize()]
        public void MyTestInitialize()
        {
            MDWSConfiguration config = ConnectionHelper.GetTestingConfig();
            var factory = Factory.GetFactory(config);
            TestContext.Properties["Factory"] = factory;

            Site site = Utils.GetExistingSiteforVistaCode(factory, "555");
            Facility facility = Utils.CreateAndSaveFacility(site);
            Section section = Utils.CreateAndSaveSectionWithAppTypes(factory, "555", facility);

            var providers = factory.GetProviderService().GetAllForSite(site.Id);
            Assert.IsTrue(providers.Count > 0);
            var provider = providers.First();

            var providerResource = Utils.GetRealProviderResource(factory, "555", section);
            var provider2Resource = Utils.GetSecondRealProviderResource(factory, "555", section);
            var roomResource = Utils.GetRoomResource(factory, "555", section);

            TestContext.Properties["Site"] = site;
            TestContext.Properties["Facility"] = facility;
            TestContext.Properties["Section"] = section;
            TestContext.Properties["Resource1"] = providerResource;
            TestContext.Properties["Resource2"] = roomResource;
            TestContext.Properties["Provider2Resource"] = provider2Resource;

            Patient patient = Utils.GetFirstPatientFromVista(factory, "555");

            TestContext.Properties["Patient"] = patient;

            Appointment appointment = new Appointment()
            {
                Time = new DateTime(2013,7, 16, 8, 0, 0),
                Length = 1,
                Patient = patient,
                Status = AppointmentStatus.Scheduled,
                Section = section,
                AppointmentType = section.AppointmentTypes[0],

            };
            appointment.AddResource(providerResource);
            appointment.AddResource(roomResource);

            var createdAppt = factory.GetAppointmentService().CreateAppointment(appointment);

            TestContext.Properties["Appt"] = createdAppt;

            var hierarchy = (log4net.Repository.Hierarchy.Hierarchy)log4net.LogManager.GetRepository();
            var logger = (log4net.Repository.Hierarchy.Logger)hierarchy.GetLogger("NHibernate.SQL");
            logger.AddAppender(new log4net.Appender.TraceAppender { Layout = new log4net.Layout.SimpleLayout() });
            hierarchy.Configured = true;

        }
        
        //Use TestCleanup to run code after each test has run
        [TestCleanup()]
        public void MyTestCleanup()
        {
            var appt = TestContext.Properties["Appt"] as Appointment;
            var factory = TestContext.Properties["Factory"] as Factory;
            var deleteAppt = TestContext.Properties["DeleteAppt"] as String;

            if (!String.IsNullOrEmpty(deleteAppt) && String.Equals(Boolean.TrueString, deleteAppt))
            {
                factory.GetAppointmentService().Delete(appt.Id);
            }

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                var site = TestContext.Properties["Site"] as Site;
                var facility = TestContext.Properties["Facility"] as Facility;
                var section = TestContext.Properties["Section"] as Section;
                var resource1 = TestContext.Properties["Resource1"] as Resource;
                var resource2 = TestContext.Properties["Resource2"] as Resource;
                var resource3 = TestContext.Properties["Provider2Resource"] as Resource;
                var patient = TestContext.Properties["Patient"] as Patient;
//                var appt = TestContext.Properties["Appt"] as Appointment;

//                dbSession.Delete(appt);
//                dbSession.Delete(patient);
                dbSession.Delete(resource1);
                dbSession.Delete(resource2);
                dbSession.Delete(resource3);
                dbSession.Delete(section);
                dbSession.Delete(facility);
//                dbSession.Delete(site);

                transaction.Commit();
            }
        }
        
        #endregion

        [TestMethod()]
        public void AppointmentDoCancelActionTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();

            var appt = TestContext.Properties["Appt"] as Appointment;

            int initialApptActions = appt.Activity.Count;

            service.CancelAppointment(appt, "PC", "13", "Patient no longer wants appointment");

            var retrieved = service.Get(appt.Id);

            var activitys = new List<AppointmentActivity>(retrieved.Activity);
            Assert.AreEqual(initialApptActions + 1, retrieved.Activity.Count);
            Assert.AreEqual(AppointmentStatus.Cancelled, retrieved.Status);

            var lastActivity = activitys.OrderByDescending(x => x.Time).First();
            Assert.AreEqual(ActivityType.Cancel, lastActivity.ActivityType);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                //                dbSession.Delete(retrieved);
                dbSession.Delete(retrieved);
                transaction.Commit();
            }
        }

        [TestMethod()]
        [ExpectedException(typeof(ArgumentException))]
        public void AppointmentDoCancelActionRequiresReasonTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();
            var appt = TestContext.Properties["Appt"] as Appointment;

            TestContext.Properties["DeleteAppt"] = Boolean.TrueString;

            service.CancelAppointment(appt, "", "", "");
        }

        [TestMethod()]
        public void AppointmentDoCheckInActionTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();

            var appt = TestContext.Properties["Appt"] as Appointment;

            int initialApptActions = appt.Activity.Count;

            service.CheckInAppointment(appt);

            var retrieved = service.Get(appt.Id);

            var activitys = new List<AppointmentActivity>(retrieved.Activity);
            Assert.AreEqual(initialApptActions + 1, retrieved.Activity.Count);
            Assert.AreEqual(retrieved.Status, AppointmentStatus.CheckedIn);

            var lastActivity = activitys.OrderByDescending(x => x.Time).First();
            Assert.AreEqual(lastActivity.ActivityType, ActivityType.CheckIn);

            TestContext.Properties["DeleteAppt"] = Boolean.TrueString;
        }

        [TestMethod()]
        [ExpectedException(typeof(ArgumentException))]
        public void AppointmentDoCheckOutWithoutCheckinTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();
            TestContext.Properties["DeleteAppt"] = Boolean.TrueString;

            var appt = TestContext.Properties["Appt"] as Appointment;

            int initialApptActions = appt.Activity.Count;

            service.CheckOutAppointment(appt, "Kept");
        }


        // can't run this due to quirk in Vista. If I 'cancel' a checked out appointment, I can't create another appointment
        // in that timeslot again
        //[TestMethod()]
        //public void AppointmentDoCheckOutActionTest()
        //{
        //    var factory = TestContext.Properties["Factory"] as Factory;
        //    var service = factory.GetAppointmentService();

        //    TestContext.Properties["DeleteAppt"] = Boolean.TrueString;

        //    var appt = TestContext.Properties["Appt"] as Appointment;

        //    service.CheckInAppointment(appt);

        //    var retrieved = service.Get(appt.Id);

        //    int initialApptActions = retrieved.Activity.Count;

        //    service.CheckOutAppointment(appt, "Kept");

        //    retrieved = service.Get(appt.Id);

        //    var activitys = new List<AppointmentActivity>(retrieved.Activity);
        //    Assert.AreEqual(initialApptActions + 1, retrieved.Activity.Count);
        //    Assert.AreEqual(retrieved.Status, AppointmentStatus.Closed);

        //    var lastActivity = activitys.OrderByDescending(x => x.Time).First();
        //    Assert.AreEqual(lastActivity.ActivityType, ActivityType.CheckOut);

        //    //using (var dbSession = DataAccess.DataAccess.OpenSession())
        //    //using (var transaction = dbSession.BeginTransaction())
        //    //{
        //    //    dbSession.Delete(retrieved);
        //    //    transaction.Commit();
        //    //}
        //}

        [TestMethod()]
        public void AppointmentDoCreateActionTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();
            TestContext.Properties["DeleteAppt"] = Boolean.TrueString;

//            var patient = TestContext.Properties["Patient"] as Patient;
            var resource = TestContext.Properties["Resource1"] as Resource;
            var section = TestContext.Properties["Section"] as Section;
            
//            var site = factory.GetSiteService().GetFromVistaSiteId("555");

            var patient = Utils.GetFirstPatientFromVista(factory, "555");

            DateTime apptTime = new DateTime(2013, 5, 2, 8, 0, 0);

            var appt = new Appointment()
                {
                    Time = apptTime,
                    Length = 5,
                    Patient = patient,
                    Status = AppointmentStatus.Scheduled,
                    Section = section,
                    AppointmentType = section.AppointmentTypes[0],

                };
            appt.AddResource(resource);

            int initialApptActions = appt.Activity.Count;

            service.CreateAppointment(appt);

            var retrieved = service.Get(appt.Id);

            Assert.AreNotEqual(retrieved.VistaId, MDWSAccess.Utils.ToMDWSAppointmentDateTime(apptTime));

            var activitys = new List<AppointmentActivity>(retrieved.Activity);
            Assert.AreEqual(initialApptActions + 1, retrieved.Activity.Count);
            Assert.AreEqual(retrieved.Status, AppointmentStatus.Scheduled);

            var lastActivity = activitys.OrderByDescending(x => x.Time).First();
            Assert.AreEqual(lastActivity.ActivityType, ActivityType.Create);

            service.Delete(retrieved.Id);

//            using (var dbSession = DataAccess.DataAccess.OpenSession())
//            using (var transaction = dbSession.BeginTransaction())
//            {
////                dbSession.Delete(retrieved);
//                dbSession.Delete(resource);
//                transaction.Commit();
//            }
            
            TestContext.Properties["DeleteAppt"] = Boolean.TrueString;
        }

        [TestMethod()]
        public void AppointmentModifiedTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();
            TestContext.Properties["DeleteAppt"] = Boolean.TrueString;

            var appt = TestContext.Properties["Appt"] as Appointment;

            int initialApptActions = appt.Activity.Count;
            System.Threading.Thread.Sleep(1000);
            service.Update(appt);

            var retrieved = service.Get(appt.Id);

            var activitys = new List<AppointmentActivity>(retrieved.Activity);
            Assert.AreEqual(initialApptActions + 1, retrieved.Activity.Count);

            var lastActivity = activitys.OrderByDescending(x => x.Time).First();
            Assert.AreEqual(ActivityType.Modified, lastActivity.ActivityType);

        }

        [TestMethod()]
        public void AppointmentChangeProviderTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();
            TestContext.Properties["DeleteAppt"] = Boolean.TrueString;

            var appt = TestContext.Properties["Appt"] as Appointment;

            int initialApptActions = appt.Activity.Count;
            System.Threading.Thread.Sleep(1000);

            var origProvider = TestContext.Properties["Resource1"] as Resource;
            var newProvider = TestContext.Properties["Provider2Resource"] as Resource;

            var orig = appt.Resources.FirstOrDefault(r => r.Id == origProvider.Id);
            Assert.IsNotNull(orig);

            var providersAvail = factory.GetResourceService().FindAlternateAvailableProviders(appt);
            Assert.AreEqual(1, providersAvail.Count);

            appt.Resources.Remove(orig);
            appt.Resources.Add(newProvider);

            service.Update(appt);

            var retrieved = service.Get(appt.Id);

            var activitys = new List<AppointmentActivity>(retrieved.Activity);
            Assert.AreEqual(initialApptActions + 2, retrieved.Activity.Count);

            var lastActivity = activitys.OrderByDescending(x => x.Time).First();
            Assert.AreEqual(ActivityType.Modified, lastActivity.ActivityType);
        }


        [TestMethod()]
        public void AppointmentDoNoShowActionTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAppointmentService();

            var appt = TestContext.Properties["Appt"] as Appointment;

            int initialApptActions = appt.Activity.Count;
            System.Threading.Thread.Sleep(1000);
            service.NoShowAppointment(appt);

            var retrieved = service.Get(appt.Id);

            var activitys = new List<AppointmentActivity>(retrieved.Activity);
            Assert.AreEqual(initialApptActions + 1, retrieved.Activity.Count);
            Assert.AreEqual(retrieved.Status, AppointmentStatus.Closed);

            var lastActivity = activitys.OrderByDescending(x => x.Time).First();
            Assert.AreEqual(lastActivity.ActivityType, ActivityType.NoShow);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                //                dbSession.Delete(retrieved);
                dbSession.Delete(retrieved);
                transaction.Commit();
            }
        }

        //[TestMethod()]
        //public void VerifyAppointmentStatus()
        //{
        //    var factory = TestContext.Properties["Factory"] as Factory;
        //    var service = factory.GetAppointmentService();

        //    var appt = TestContext.Properties["Appt"] as Appointment;

        //    service.GetAppointmentVistaStatus(null);

        //    TestContext.Properties["DeleteAppt"] = Boolean.TrueString;


        //}

    }
}
