﻿using MedRed.Services.ServiceImpl;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using Shared.Model.Config.MDWS;
using Shared.Model;
using Shared.Model.UserRoles;
using MedRed.Services.Utils;
using KellermanSoftware.CompareNetObjects;

namespace MedRed.Services.Tests
{
    
    
    /// <summary>
    ///This is a test class for AccountServiceTest and is intended
    ///to contain all AccountServiceTest Unit Tests
    ///</summary>
    [TestClass()]
    public class AccountServiceTest
    {


        private TestContext testContextInstance;

        /// <summary>
        ///Gets or sets the test context which provides
        ///information about and functionality for the current test run.
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        #region Additional test attributes
        // 
        //You can use the following additional attributes as you write your tests:
        //
        //Use ClassInitialize to run code before running the first test in the class
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //Use ClassCleanup to run code after all tests in a class have run
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //Use TestInitialize to run code before running each test
        [TestInitialize()]
        public void MyTestInitialize()
        {
            MDWSConfiguration config = ConnectionHelper.GetTestingConfig();
//            MDWSConnection connection = ConnectionHelper.GetConnectionFor("100");
            var factory = Factory.GetFactory(config);

            TestContext.Properties["Factory"] = factory;

            var hierarchy = (log4net.Repository.Hierarchy.Hierarchy)log4net.LogManager.GetRepository();
            var logger = (log4net.Repository.Hierarchy.Logger)hierarchy.GetLogger("NHibernate.SQL");
            logger.AddAppender(new log4net.Appender.TraceAppender { Layout = new log4net.Layout.SimpleLayout() });
            hierarchy.Configured = true;

        }
        
        //Use TestCleanup to run code after each test has run
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion


        /// <summary>
        ///A test for Login
        ///</summary>
        [TestMethod()]
        public void LoginFailTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAccountService();

            var newUser = new User() { Username = "medred1", Person = new Person(){ FirstName="John", LastName="Doe"}};
            PasswordHelper.SetUserPassword(ref newUser, "mysecretpassword");

            service.AddUser(newUser);

            Assert.IsNull(service.LogOn("medred1", "wrongpassword"));

            service.DeleteUser(newUser.Id);
        }

        /// <summary>
        ///A test for Login
        ///</summary>
        [TestMethod()]
        public void LoginSuccessTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAccountService();

            var newUser = new User() { Username = "medred3", Person = new Person() { FirstName = "John", LastName = "Doe" } };
            PasswordHelper.SetUserPassword(ref newUser, "mysecretpassword");

            service.AddUser(newUser);

            
            var loggedInUser = service.LogOn("medred3", "mysecretpassword");
            Assert.IsNotNull(loggedInUser);

            CompareObjects compareObjects = new CompareObjects();
            Assert.IsTrue(compareObjects.Compare(newUser, loggedInUser), compareObjects.DifferencesString);

            service.DeleteUser(newUser.Id);
        }

        [TestMethod()]
        public void AddUserTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAccountService();

            var newUser = new User() { Username = "medred1", Person = new Person() { FirstName = "John", LastName = "Doe" } };
            PasswordHelper.SetUserPassword(ref newUser, "mysecretpassword");

            service.AddUser(newUser);

            var retrievedUser = service.GetUser(newUser.Id);

            CompareObjects compareObjects = new CompareObjects();
            Assert.IsTrue(compareObjects.Compare(newUser, retrievedUser), compareObjects.DifferencesString);

            service.DeleteUser(newUser.Id);
        }

        [TestMethod()]
        public void UpdateUserTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAccountService();

            var newUser = new User() { Username = "medred1", Person = new Person() { FirstName = "John", LastName = "Doe" } };
            PasswordHelper.SetUserPassword(ref newUser, "mysecretpassword");

            service.AddUser(newUser);
            int userId = newUser.Id;
            Assert.IsTrue(userId != 0);

            var retrievedUser = service.GetUser(userId);

            retrievedUser.Person.LastName = "Jones";
            PasswordHelper.SetUserPassword(ref retrievedUser, "NewPassword");

            service.UpdateUser(retrievedUser);

            var updatedUser = service.GetUser(userId);

            Assert.AreNotEqual(newUser.Password, updatedUser.Password);
            Assert.AreEqual(retrievedUser.Password, updatedUser.Password);
            Assert.AreNotEqual(newUser.Person.LastName, updatedUser.Person.LastName);
            Assert.AreEqual(retrievedUser.Person.LastName, updatedUser.Person.LastName);
            
            service.DeleteUser(userId);
        }

        [TestMethod()]
        public void GetUserFromUsernameTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAccountService();

            var newUser = new User() { Username = "medred9", Person = new Person() { FirstName = "John", LastName = "Doe" } };
            PasswordHelper.SetUserPassword(ref newUser, "mysecretpassword");

            service.AddUser(newUser);
            int userId = newUser.Id;
            Assert.IsTrue(userId != 0);

            var retrievedUser = service.GetUserFromUsername("medred9");

            CompareObjects compareObjects = new CompareObjects();
            Assert.IsTrue(compareObjects.Compare(newUser, retrievedUser), compareObjects.DifferencesString);

            service.DeleteUser(userId);
        }


        [TestMethod()]
        public void AddUserRolesTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAccountService();

            Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };
            
            var newUser = new User() { Username = "medred1", Person = new Person() { FirstName = "John", LastName = "Doe" } };
            PasswordHelper.SetUserPassword(ref newUser, "mysecretpassword");

            service.AddUser(newUser);

            Shared.Model.UserRoles.Action[] Actions = {
                 new Shared.Model.UserRoles.Action(){ Name = "Action 1", Description="This is description 1", Key="Key1"}, 
                 new Shared.Model.UserRoles.Action() { Name = "Action 2", Description="This is description 2", Key = "Key2"}};

            var Role1 = new Role() { Name = "Role 1" };
            var Role2 = new Role() { Name = "Role 2" };

            Role1.AddAction(Actions[0]);
            Role1.AddAction(Actions[1]);

            Role2.AddAction(Actions[1]);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(site);
                dbSession.Save(Actions[0]);
                dbSession.Save(Actions[1]);
                dbSession.Save(Role1);
                dbSession.Save(Role2);
                transaction.Commit();
            }

            newUser.AddRole(new UserRole() { Role = Role1, Site = site });
            newUser.AddRole(new UserRole() { Role = Role2, Site = site });

            service.UpdateUser(newUser);

            var retrievedUser = service.GetUser(newUser.Id);

            CompareObjects compareObjects = new CompareObjects();
            Assert.IsTrue(compareObjects.Compare(newUser, retrievedUser), compareObjects.DifferencesString);

            service.DeleteUser(newUser.Id);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(Role1);
                dbSession.Delete(Role2);
                dbSession.Delete(Actions[0]);
                dbSession.Delete(Actions[1]);
                dbSession.Delete(site);
                transaction.Commit();
            }
        }

        [TestMethod()]
        public void RemoveUserRolesTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAccountService();

            Site site = new Site() { VistaSiteId = "501", Name = "Test Site 2" };

            var newUser = new User() { Username = "medred1", Person = new Person() { FirstName = "John", LastName = "Doe" } };
            PasswordHelper.SetUserPassword(ref newUser, "mysecretpassword");

            service.AddUser(newUser);

            Shared.Model.UserRoles.Action[] Actions = {
                 new Shared.Model.UserRoles.Action(){ Name = "Action 1", Description="This is description 1", Key="Key1"}, 
                 new Shared.Model.UserRoles.Action() { Name = "Action 2", Description="This is description 2", Key = "Key2"}};

            var Role1 = new Role() { Name = "Role 1" };
            var Role2 = new Role() { Name = "Role 2" };

            Role1.AddAction(Actions[0]);
            Role1.AddAction(Actions[1]);

            Role2.AddAction(Actions[1]);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(site);
                dbSession.Save(Actions[0]);
                dbSession.Save(Actions[1]);
                dbSession.Save(Role1);
                dbSession.Save(Role2);
                transaction.Commit();
            }

            newUser.AddRole(new UserRole() { Role = Role1, Site = site });
            newUser.AddRole(new UserRole() { Role = Role2, Site = site });

            service.UpdateUser(newUser);

            var retrievedUser = service.GetUser(newUser.Id);

            Assert.AreEqual(2, retrievedUser.Roles.Count);

            retrievedUser.Roles.Remove(retrievedUser.Roles[0]);

            service.UpdateUser(retrievedUser);
            var roleRemovedUser = service.GetUser(newUser.Id);

            Assert.AreEqual(1, roleRemovedUser.Roles.Count);

            service.DeleteUser(newUser.Id);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(Role1);
                dbSession.Delete(Role2);
                dbSession.Delete(Actions[0]);
                dbSession.Delete(Actions[1]);
                dbSession.Delete(site);
                transaction.Commit();
            }
        }

        [TestMethod()]
        public void GetUsersTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAccountService();

            var newUser = new User() { Username = "medred1", Person = new Person() { FirstName = "John", LastName = "Doe" } };
            PasswordHelper.SetUserPassword(ref newUser, "mysecretpassword");

            service.AddUser(newUser);

            var users = service.GetUsers();

            Assert.IsTrue(users.Count >= 1);


            service.DeleteUser(newUser.Id);
        }

        [TestMethod()]
        public void GetAllRolesTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAccountService();

            Shared.Model.UserRoles.Action[] Actions = {
                 new Shared.Model.UserRoles.Action(){ Name = "Action 1", Description="This is description 1", Key="Key1"}, 
                 new Shared.Model.UserRoles.Action() { Name = "Action 2", Description="This is description 2", Key = "Key2"}};

            var Role1 = new Role() { Name = "Role 1" };
            var Role2 = new Role() { Name = "Role 2" };

            Role1.AddAction(Actions[0]);
            Role1.AddAction(Actions[1]);

            Role2.AddAction(Actions[1]);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(Actions[0]);
                dbSession.Save(Actions[1]);
                dbSession.Save(Role1);
                dbSession.Save(Role2);
                transaction.Commit();
            }

            var roles = service.GetAllRoles();

            Assert.IsTrue(roles.Count >= 2);
            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(Role1);
                dbSession.Delete(Role2);
                dbSession.Delete(Actions[0]);
                dbSession.Delete(Actions[1]);
                transaction.Commit();
            }
        }

        [TestMethod()]
        public void RemoveActionFromRolesTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAccountService();

            Shared.Model.UserRoles.Action[] Actions = {
                 new Shared.Model.UserRoles.Action(){ Name = "Action 1", Description="This is description 1", Key="Key1"}, 
                 new Shared.Model.UserRoles.Action() { Name = "Action 2", Description="This is description 2", Key = "Key2"}};

            var Role1 = new Role() { Name = "Role 1" };
            var Role2 = new Role() { Name = "Role 2" };

            Role1.AddAction(Actions[0]);
            Role1.AddAction(Actions[1]);

            Role2.AddAction(Actions[1]);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(Actions[0]);
                dbSession.Save(Actions[1]);
                dbSession.Save(Role1);
                dbSession.Save(Role2);
                transaction.Commit();
            }

            var role = service.GetRole(Role1.Id);

            Assert.AreEqual(2, role.Actions.Count);

            role.Actions.Remove(role.Actions[0]);

            service.UpdateRole(role);

            var actionRemovedRole = service.GetRole(Role1.Id);

            Assert.AreEqual(1, actionRemovedRole.Actions.Count);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(Role1);
                dbSession.Delete(Role2);
                dbSession.Delete(Actions[0]);
                dbSession.Delete(Actions[1]);
                transaction.Commit();
            }
        }

        [TestMethod()]
        public void GetAllActionsTest()
        {
            var factory = TestContext.Properties["Factory"] as Factory;
            var service = factory.GetAccountService();

            Shared.Model.UserRoles.Action[] Actions = {
                 new Shared.Model.UserRoles.Action(){ Name = "Action 1", Description="This is description 1", Key="Key1"}, 
                 new Shared.Model.UserRoles.Action() { Name = "Action 2", Description="This is description 2", Key = "Key2"}};

            var Role1 = new Role() { Name = "Role 1" };
            var Role2 = new Role() { Name = "Role 2" };

            Role1.AddAction(Actions[0]);
            Role1.AddAction(Actions[1]);

            Role2.AddAction(Actions[1]);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Save(Actions[0]);
                dbSession.Save(Actions[1]);
                dbSession.Save(Role1);
                dbSession.Save(Role2);
                transaction.Commit();
            }

            var actions = service.GetAllActions();

            Assert.IsTrue(actions.Count >= 2);

            using (var dbSession = DataAccess.DataAccess.OpenSession())
            using (var transaction = dbSession.BeginTransaction())
            {
                dbSession.Delete(Role1);
                dbSession.Delete(Role2);
                dbSession.Delete(Actions[0]);
                dbSession.Delete(Actions[1]);
                transaction.Commit();
            }
        }

    }
}
