﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Shared.Model;
using System.Globalization;

namespace MedRed.MDWSAccess
{
    class PatientFormatter
    {
        private const string NscVeteran = "NSC VETERAN";
        private const string ScVeteran = "SC VETERAN";
        private const string M = "M";
        private const string F = "F";

        public static Patient CreatePatientFromMDWSPatient(string vistaSiteId, EmrSvc.PatientTO p_mdwsPatient)
        {
            Patient patient = new Patient();
            patient.Person = new Person();

            DateTime dob;
            if (DateTime.TryParseExact(p_mdwsPatient.dob, "yyyyMMdd",
                                       System.Globalization.CultureInfo.InvariantCulture, DateTimeStyles.None, out dob))
            {
                patient.Person.DateOfBirth = dob;// dob.ToString("MM/dd/yyyy");
            }

            string first, middle, last;
            Utils.SplitName(p_mdwsPatient.name, out first, out middle, out last);
            patient.Person.FirstName = first;
            patient.Person.MiddleName = middle;
            patient.Person.LastName = last;

            if (p_mdwsPatient.homePhone != null)
            {
                patient.Person.Phone = p_mdwsPatient.homePhone.ToString();
            }

            if (p_mdwsPatient.demographics != null && p_mdwsPatient.demographics.Count() > 0)
            {
                //address
                if (p_mdwsPatient.demographics[0].addresses != null && p_mdwsPatient.demographics[0].addresses.Count() > 0)
                {
                    patient.Person.Address = new Address();
                    patient.Person.Address.Street1 = p_mdwsPatient.demographics[0].addresses[0].streetAddress1;
                    patient.Person.Address.Street2 = p_mdwsPatient.demographics[0].addresses[0].city + " " +
                        p_mdwsPatient.demographics[0].addresses[0].state + " " +
                        p_mdwsPatient.demographics[0].addresses[0].zipcode;
//                    patient.State = p_mdwsPatient.demographics[0].addresses[0].state;
//                    patient.ZipCode = p_mdwsPatient.demographics[0].addresses[0].zipcode;
                }

                //phone
                if (p_mdwsPatient.demographics[0].phones != null && p_mdwsPatient.demographics[0].phones.Count() > 0)
                {
                    string areaCode = p_mdwsPatient.demographics[0].phones[0].areaCode;
                    string exchange = p_mdwsPatient.demographics[0].phones[0].exchange;
                    string number = p_mdwsPatient.demographics[0].phones[0].number;
                    patient.Person.Phone = string.Format("{0}{1}{2}", areaCode, exchange, number);
                }

                if (p_mdwsPatient.demographics[0].emailAddresses != null &&
                    p_mdwsPatient.demographics[0].emailAddresses.Count() > 0)
                {
                    patient.Person.Email = p_mdwsPatient.demographics[0].emailAddresses[0];
                }
            }

            patient.Person.Sex = TranslateMdwsGender(p_mdwsPatient.gender);
            patient.Person.SSN = p_mdwsPatient.ssn;
//            patient.DFN = p_mdwsPatient.localPid; 
            if (!String.IsNullOrEmpty(p_mdwsPatient.patientType))
            {
                patient.MilitaryStatus = TranslateMdwsPatientType(p_mdwsPatient.patientType);
            }


            patient.AddPatientSite(vistaSiteId, p_mdwsPatient.localPid);

            patient.ICN = p_mdwsPatient.mpiPid;

            // add patient sites
            if (p_mdwsPatient.sites != null && p_mdwsPatient.sites.count > 0)
            {
                foreach (var site in p_mdwsPatient.sites.sites)
                {
                    // if this is a different site, and there is a pid, add it to the PatientSites
                    if (!String.Equals(site.sitecode, vistaSiteId) && !String.IsNullOrEmpty(site.pid))
                    {
                        patient.AddPatientSite(site.sitecode, site.pid);
                    }
                }
            }

            return patient;
        }

        public static PatientSearchResult GetPatientSearchResultFromMDWSPatient(string vistaSiteId, EmrSvc.PatientTO p_mdwsPatient)
        {
            return GetPatientSearchResultFromMDWSPatientData(vistaSiteId, p_mdwsPatient.localPid, p_mdwsPatient.name, p_mdwsPatient.dob, p_mdwsPatient.gender, p_mdwsPatient.ssn);
        }

        public static PatientSearchResult GetPatientSearchResultFromMDWSPatientData(string vistaSiteId, string dfn, string name, string dob, string sex, string ssn)
        {
            PatientSearchResult result = new PatientSearchResult();

            DateTime? dateOfBirth;
            dateOfBirth = Utils.FromMDWSDate(dob);
            if (dateOfBirth.HasValue)
                result.DateOfBirth = dateOfBirth.Value;
            string first, middle, last;
            Utils.SplitName(name, out first, out middle, out last);
            result.FirstName = first;
            result.MiddleName = middle;
            result.LastName = last;

            result.Sex = TranslateMdwsGender(sex);
            result.SSN = ssn;
            result.VistaSiteId = vistaSiteId;
            result.VistaDFN = dfn;

            return result;
        }

        
        private static string TranslateMdwsGender(string p_mdwsGender)
        {
            switch (p_mdwsGender)
            {
                case M:
                    return "Male";
                case F:
                    return "Female";
                default:
                    return string.Empty;
            }
        }

        private static string TranslateMdwsPatientType(string p_mdwsPatientType)
        {
            switch (p_mdwsPatientType)
            {
                case NscVeteran:
                case ScVeteran:
                    return "Veteran";
                default:
                    return string.Empty;
            }
        }

        public static string ReformatCPRSDate(string p_dateString)
        {
            DateTime date = ParseCPRSDate(p_dateString);
            string newFormat = date.ToString("MMM dd, yyyy HH:mm");
            return newFormat;
        }

        public static DateTime ParseCPRSDate(string p_dateString)
        {
            System.Globalization.CultureInfo provider = System.Globalization.CultureInfo.InvariantCulture;

            //20110125.090000
            //        Date is formatted 20111118 and should be Nov 18, 2011
            DateTime date;
            try
            {
                date = DateTime.ParseExact(p_dateString, @"yyyyMMdd\.HHmmss", provider);

            }
            catch (FormatException)
            {
//                Logger.Log(string.Format(RX_Others._0_is_not_in_the_correct_format, p_dateString));
                return DateTime.MinValue;
            }
            return date;
        }

    }
}
