﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using MedRed.MDWSAccess.QuerySvc;
using MedRed.MDWSAccess.SchedulingService;
using Shared.Model;
using Shared.Model.Config.MDWS;

namespace MedRed.MDWSAccess
{
    public partial class MDWSCalls : IDisposable
    {
        public enum Service { EMR, Scheduling };

        private MDWSConfiguration Configuration { get; set; }
        private MDWSConnection Connection { get; set; }

        private EmrSvc.EmrSvcSoapClient _EmrService;
        private SchedulingService.SchedulingSvcSoapClient _SchedulingService;
        private QuerySvc.QuerySvcSoapClient _QueryService;

        private string Emr_Service_Name = "EmrSvc.asmx";
        private string Scheduling_Service_Name = "SchedulingSvc.asmx";
        private string Query_Service_Name = "QuerySvc.asmx";

        public MDWSCalls()
        {
            _EmrService = null;
            _QueryService = null;
            _SchedulingService = null;
        }

        public bool setConfig(MDWSConfiguration _config)
        {
            Configuration = _config;

            return true;
        }

        public bool setConnection(MDWSConnection _connection)
        {
            Connection = _connection;

            return true;
        }

        private bool prepareSchedulingConnection()
        {
            if (_SchedulingService == null)  {  _prepareSchedulingConnection(); }
            return true;
        }

        private bool prepareEmrConnection()
        {
            if (_EmrService == null) { _prepareEmrConnection(); }
            return true;
        }

        private bool prepareQueryConnection()
        {
            if (_QueryService == null) { _prepareQueryConnection(); }
            return true;
        }

        private bool disconnectScheduling()
        {
            return true;
        }

        private bool disconnectEmr()
        {
            return true;
        }

        private bool disconnectQuery()
        {
            return true;
        }


        private bool _prepareSchedulingConnection()
        {
            _SchedulingService = new SchedulingSvcSoapClient();
            Uri baseMDWSUrl = new Uri(Configuration.BaseMDWS_Url);
            _SchedulingService.Endpoint.Address = new System.ServiceModel.EndpointAddress(new Uri(baseMDWSUrl, Scheduling_Service_Name));

            //// if anything logged in, don't need to log in again
            //if (_EmrService == null && _QueryService == null)
            //{
                var result = _SchedulingService.connect(Connection.SiteID);
                if (result != null && Utils.HasFaultMessage(result))
                {
                    DataSource ds = Configuration.DataSources[Connection.SiteID];
                    if (ds == null)
                    {
                        throw new Exception("Unknown site " + Connection.SiteID);
                    }
                    AddSchedulingDataSource(ds);
                    result = _SchedulingService.connect(Connection.SiteID);
                }
                if (result != null && Utils.HasFaultMessage(result))
                {
                    throw new Exception("Unable to connect to site " + Connection.SiteID + " Message: " + result.fault.message);
                }
                var result2 = _SchedulingService.login(Connection.AccessCode, Connection.VerifyCode, string.Empty);
                if (result2 != null && Utils.HasFaultMessage(result2))
                {
                    throw new Exception("Unable to log in to site " + Connection.SiteID + " Message: " + result2.fault.message);
                }
            //}
            return true;
        }

        private bool _disconnectScheduling()
        {
            var result = _SchedulingService.disconnect();
            if (result != null && Utils.HasFaultMessage(result))
            {
                throw new Exception("Unable to log out of site " + Connection.SiteID + " Message: " + result.fault.message);
            }

            return true;
        }

        private bool _prepareEmrConnection()
        {
            _EmrService = new EmrSvc.EmrSvcSoapClient();
            Uri baseMDWSUrl = new Uri(Configuration.BaseMDWS_Url);
            _EmrService.Endpoint.Address = new System.ServiceModel.EndpointAddress(new Uri(baseMDWSUrl, Emr_Service_Name));

            //// if anything logged in, don't need to log in again
            //if (_SchedulingService == null && _QueryService == null)
            //{
                var result = _EmrService.connect(Connection.SiteID);
                if (result != null && Utils.HasFaultMessage(result))
                {
                    DataSource ds = Configuration.DataSources[Connection.SiteID];
                    if (ds == null)
                    {
                        throw new Exception("Unknown site " + Connection.SiteID);
                    }
                    AddEmrDataSource(ds);
                    result = _EmrService.connect(Connection.SiteID);
                }
                if (result != null && Utils.HasFaultMessage(result))
                {
                    throw new Exception("Unable to connect to site " + Connection.SiteID + " Message: " + result.fault.message);
                }
                var result2 = _EmrService.login(Connection.AccessCode, Connection.VerifyCode, string.Empty);
                if (result2 != null && Utils.HasFaultMessage(result2))
                {
                    throw new Exception("Unable to log in to site " + Connection.SiteID + " Message: " + result2.fault.message);
                }
            //}
            return true;
        }

        private bool _disconnectEmr()
        {
            var result = _EmrService.disconnect();
            if (result != null && Utils.HasFaultMessage(result))
            {
                throw new Exception("Unable to log out of site " + Connection.SiteID + " Message: " + result.fault.message);
            }

            return true;
        }

        private bool _prepareQueryConnection()
        {
            _QueryService = new QuerySvc.QuerySvcSoapClient();
            Uri baseMDWSUrl = new Uri(Configuration.BaseMDWS_Url);
            _QueryService.Endpoint.Address = new System.ServiceModel.EndpointAddress(new Uri(baseMDWSUrl, Query_Service_Name));

            //// if anything logged in, don't need to log in again
            //if (_SchedulingService == null && _EmrService == null)
            //{
                var result = _QueryService.connect(Connection.SiteID);
                if (result != null && Utils.HasFaultMessage(result))
                {
                    DataSource ds = Configuration.DataSources[Connection.SiteID];
                    if (ds == null)
                    {
                        throw new Exception("Unknown site " + Connection.SiteID);
                    }
                    AddQueryDataSource(ds);
                    result = _QueryService.connect(Connection.SiteID);
                }
                if (result != null && Utils.HasFaultMessage(result))
                {
                    throw new Exception("Unable to connect to site " + Connection.SiteID + " Message: " + result.fault.message);
                }
                var result2 = _QueryService.login(Connection.AccessCode, Connection.VerifyCode, string.Empty);
                if (result2 != null && Utils.HasFaultMessage(result2))
                {
                    throw new Exception("Unable to log in to site " + Connection.SiteID + " Message: " + result2.fault.message);
                }
            //}
            return true;
        }

        private bool _disconnectQuery()
        {
            var result = _QueryService.disconnect();
            if (result != null && Utils.HasFaultMessage(result))
            {
                throw new Exception("Unable to log out of site " + Connection.SiteID + " Message: " + result.fault.message);
            }

            return true;
        }

        private bool AddSchedulingDataSource(DataSource ds)
        {
            var siteTo = _SchedulingService.addDataSource(ds.SiteID,
                ds.Server,
                ds.Server,
                ds.Port,
                ds.Modality,
                ds.Protocol,
                ds.Region);
            return true;
        }

        private bool AddEmrDataSource(DataSource ds)
        {
            var siteTo = _EmrService.addDataSource(ds.SiteID,
                ds.Server,
                ds.Server,
                ds.Port,
                ds.Modality,
                ds.Protocol,
                ds.Region);
            return true;
        }

        private bool AddQueryDataSource(DataSource ds)
        {
            var siteTo = _QueryService.addDataSource(ds.SiteID,
                ds.Server,
                ds.Server,
                ds.Port,
                ds.Modality,
                ds.Protocol,
                ds.Region);
            return true;
        }

        //public List<_Clinic> searchForClinics(string criteria)
        //{
        //    string searchString = (criteria == null ? string.Empty : criteria);
        //    prepareSchedulingConnection();
        //    var clinics = _SchedulingService.getClinics(searchString);

        //    List<_Clinic> clinicList = new List<_Clinic>();
        //    if (clinics.count > 0)
        //    {
        //        foreach (var mdwsClinic in clinics.locations)
        //        {
        //            var clinic = ClinicFormatter.CreateClinicFromMDWSClinic(mdwsClinic);
        //            clinicList.Add(clinic);
        //        }
        //    }
        //    disconnectScheduling();
        //    return clinicList;
        //}

        public List<PatientSearchResult> searchForPatients(string criteria)
        {
            string searchString = criteria;
            prepareEmrConnection();

            if (String.IsNullOrEmpty(criteria))
                searchString = "aa";
            else if (criteria.Length == 1)
                searchString = criteria + "a";

            var patients = _EmrService.match(searchString);

            // if search criteria is just text, then remove answers not matching the results
           

            List<PatientSearchResult> patientList = new List<PatientSearchResult>();
            if (patients.count > 0)
            {
                prepareQueryConnection();
                foreach (var patient in patients.arrays[0].patients)
                {
                    if (criteria.Length == 1 || (criteria.Length >= 2 && Char.IsLetter(criteria[1])))
                    {
                        // skip any patients that don't match the criteria
                        if (!patient.name.ToUpper().StartsWith(criteria.ToUpper()))
                            continue;
                    }

                    try
                    {
                        var values = read(patient.localPid, "@;.01;.02;.03;.09", "2");
                        var searchResult = PatientFormatter.GetPatientSearchResultFromMDWSPatientData(Connection.SiteID, patient.localPid,
                            values[".01"],
                            values[".03"],
                            values[".02"],
                            values[".09"]);
                        patientList.Add(searchResult);
                    }
                    catch (Exception)
                    {
                        int i = 0;
                        // MDWS fails for a blank patient - don't know how it got there, but just ignoring when that happens for now
                    }
                }
                disconnectQuery();
            }

            disconnectEmr();
            return patientList;
        }

        public Patient loadPatient(string DFN)
        {
            string DFNStr = (DFN == null ? string.Empty : DFN);
            prepareEmrConnection();
            var patient = _EmrService.select(DFNStr);
            disconnectEmr();

            if (patient == null)
            {
                return null;
            }
            Patient pat = PatientFormatter.CreatePatientFromMDWSPatient(Connection.SiteID, patient);

            return pat;
        }

        public bool GetAllPatientDetails(ref Patient patient)
        {
            string dfn = patient.GetDFNForVistASite(Connection.SiteID);
            if (String.IsNullOrEmpty(dfn))
                return false;
            prepareEmrConnection();
            var mdwsPatient = _EmrService.select(dfn);
            disconnectEmr();

            patient.HasInsurance = mdwsPatient.hasInsurance;
            patient.ActiveInsurance = mdwsPatient.activeInsurance;
            patient.EligibilityCode = mdwsPatient.eligibilityCode;
            patient.CurrentMeansStatus  = mdwsPatient.currentMeansStatus;
            patient.NeedsMeansTest = mdwsPatient.needsMeansTest;
//            patient.MilitaryStatus = mdwsPatient
            patient.Occupation = mdwsPatient.occupation;
            patient.IsVeteran = mdwsPatient.isVeteran;
            patient.ServiceConnected = mdwsPatient.serviceConnected;
            patient.SCPercent = mdwsPatient.scPercent;
            patient.Religion  = mdwsPatient.religion;
            patient.PatientType = mdwsPatient.patientType;

            return true;

        }

        public User login(string accessCode, string verifyCode)
        {
            var result = _EmrService.connect(Connection.SiteID);
            if (result != null && Utils.HasFaultMessage(result))
            {
                DataSource ds = Configuration.DataSources[Connection.SiteID];
                if (ds == null)
                {
                    throw new Exception("Unknown site " + Connection.SiteID);
                }
                AddEmrDataSource(ds);
                result = _EmrService.connect(Connection.SiteID);
            }
            if (result != null && Utils.HasFaultMessage(result))
            {
                throw new Exception("Unable to connect to site " + Connection.SiteID + " Message: " + result.fault.message);
            }

            var mdwsUser = _EmrService.login(accessCode, verifyCode, string.Empty);

            disconnectEmr();
            if (mdwsUser != null && !Utils.HasFaultMessage(mdwsUser))
            {
                Provider provider = ProviderFormatter.CreateFromMDWSUser(mdwsUser);
                User user = new User() {Person = new Person()};
                user.Person.FirstName = provider.Person.FirstName;
                user.Person.LastName = provider.Person.LastName;
                return user;
            }
            return null;
        }

        //public List<Appointment> loadPatientAppointments(string site, string dfn, DateTime start, DateTime stop)
        //{
        //    List<Appointment> appts = new List<Appointment>();

        //    string DFNStr = (dfn == null ? string.Empty : dfn);
        //    prepareEmrConnection();
        //    var patient = _EmrService.select(DFNStr);
        //    var appointmentList = _EmrService.getAppointments();

        //    if (appointmentList.arrays.Count() > 0 && appointmentList.arrays[0].count > 0)
        //    {
        //        foreach (var mdwsAppt in appointmentList.arrays[0].appts)
        //        {
        //            var appointment = AppointmentFormatter.CreateApptFromMDWSAppt(mdwsAppt);
        //            appts.Add(appointment);
        //        }
        //    }
        //    disconnectEmr();

        //    var filteredAppts = appts.Where(a => a.Time >= start && a.Time <= stop);
        //    return filteredAppts.ToList();
        //}

        public TimeZoneInfo getTimeZone(string siteID)
        {
            prepareQueryConnection();
            var results = _QueryService.ddrLister("4.3", "", "@;1;1I", "P", "", "", "", "", "", "");
            if (results.text.Count() > 0)
            {
                string[] values = results.text[0].Split('^');
                var timeZoneDetails = read(values[2], "*", "4.4");
                if (timeZoneDetails != null)
                {
                    string searchString = timeZoneDetails["1"];
                    if (searchString.Length > 10)
                        searchString = searchString.Substring(0, 10);
                    searchString = searchString.ToUpper();

                    var allTimeZones = TimeZoneInfo.GetSystemTimeZones();
                    var timeZones = allTimeZones.Where(t => t.StandardName.ToUpper().StartsWith(searchString));
                    if (timeZones.Count() > 0)
                        return timeZones.FirstOrDefault(t=>t.DisplayName.EndsWith("US & Canada)"));
                    else
                    {
                        // search for daylight names
                        timeZones = allTimeZones.Where(t => t.DaylightName.ToUpper().StartsWith(searchString));
                        if (timeZones.Count() > 0)
                            return timeZones.FirstOrDefault(t => t.DisplayName.EndsWith("US & Canada)"));
                    }
                }
            }
            return null;

            //prepareEmrConnection();
            //var val = _EmrService.getVariableValue("$G(^XMB(1,1,4))");

            //string timezoneOffset = val.text;
            //int timezoneVal;
            //if (!Int32.TryParse(timezoneOffset, out timezoneVal))
            //    timezoneVal = 0;

            //return timezoneVal;
        }

        public Site GetSite(string siteID)
        {
            Site newSite = null;

            prepareQueryConnection();
            var results = _QueryService.ddrLister("40.8", "", "@;.001;.01;.07I;1", "P", "", "", "", "", "", "");

//            var results2 = _QueryService.ddrLister("40.7", "", "@;.01;1;2;3;4;5;6", "P", "", "", "", "", "", "");

            string site_mdc_id = string.Empty;

            foreach (var result in results.text)
            {
                string[] values = result.Split('^');
                if (string.Equals(siteID, values[3]))
                {
                    site_mdc_id = values[2];
                    break;
                }
            }

            if (!String.IsNullOrEmpty(site_mdc_id))
            {
                var institution = read(site_mdc_id, "*", "4");
                newSite = new Site()
                {
                    Name = institution[".01"],
                    VistaSiteId = siteID,                     
                    Address = new Address()
                    {
                        Street1 = institution["1.01"],
                        Street2 = institution["1.02"],
                        City = institution["1.03"],
                        State = GetStateAbbr(institution[".02"]),
                        Zip = institution["1.04"],
                    }                    
                };

                var result2 = _QueryService.ddrLister("4.03", "," + site_mdc_id + ",", "@;.03", "P", "", "", "", "", "", "");

                if (result2 != null && result2.count > 0)
                {
                    string[] contactValues = result2.text[0].Split('^');
                    string phoneNum = contactValues[1];
                    newSite.Phone = phoneNum;                    
                }

            }
            disconnectQuery();
            //prepareEmrConnection();
            //var regions = _EmrService.getVHA();

            //EmrSvc.SiteTO vistaSite = null;

            //if (regions != null && regions.count > 0)
            //{
            //    foreach (var region in regions.regions)
            //    {
            //        var sites = region.sites.sites.Where(s => s.sitecode == siteID);
            //        if (sites != null && sites.Count() > 0)
            //        {
            //            vistaSite = sites.First(); ;
            //            continue;
            //        }
            //    }
            //}

            //Site newSite = null;

            //if (vistaSite != null)
            //{
            //    // create Site and return it
            //    newSite = new Site()
            //    {
            //        Name = vistaSite.name,
            //        VistaSiteId = siteID,
            //        Address = new Address() { Street1 = vistaSite.address },
            //        //                  Phone = vistaSite
            //    };
            //}
            //disconnectEmr();
            return newSite;
        }

        // assuming that there is a medical center division record for each facility we want to load, and that
        // there is one institution record for each medical center division
        public List<Facility> GetFacilities()
        {
            prepareQueryConnection();

            var result = _QueryService.ddrLister("40.8", "", "@;.001;.01;.07I;1", "P", "", "", "", "", "", "");
            var facilities = new List<Facility>();

            string sitePrefix = Connection.SiteID;

            if (result != null && result.text != null)
            {
                foreach (var entry in result.text)
                {
                    string[] values = entry.Split('^');
                    if (values[3].StartsWith(sitePrefix) && values[3].Length > 3)
                    {
                        string vistaFacilityId = values[0];
                        //                    string name = values[1];
                        if (!string.IsNullOrEmpty(values[2]))
                        {
                            Dictionary<String, String> institution = read(values[2], "*", "4");

                            var facility = new Facility()
                            {
                                Name = institution[".01"],
                                VistaId = values[2],
                                ShortName = institution[".05"],
                                StationNumber = institution["99"],
                                Address = new Address()
                                {
                                    Street1 = institution["1.01"],
                                    Street2 = institution["1.02"],
                                    City = institution["1.03"],
                                    State = GetStateAbbr(institution[".02"]),
                                    Zip = institution["1.04"],
                                }
                            };

                            var result2 = _QueryService.ddrLister("4.03", "," + values[2] + ",", "@;.03", "P", "", "", "", "", "", "");

                            if (result2 != null && result2.count > 0)
                            {
                                string[] contactValues = result2.text[0].Split('^');
                                string phoneNum = contactValues[1];
                                facility.Phone = phoneNum;
                            }

                            facilities.Add(facility);
                        }
                    }
                }
            }

            disconnectQuery();
            return facilities;
        }

        private string GetStateAbbr(string vistaStateCode)
        {
            string stateId = vistaStateCode;
            if (!string.IsNullOrEmpty(stateId))
            {
                var details = read(stateId, "*", "5");
                if (details != null && details.ContainsKey("1"))
                {
                    return details["1"];
                }
            }

            return string.Empty;
        }

        public List<Shared.Model.Service> GetServices()
        {
            List<Shared.Model.Service> serviceList = new List<Shared.Model.Service>();

            prepareQueryConnection();
            var result = _QueryService.ddrLister("49", "", "@;.01;1.7", "P", "", "", "", "", "", "");
            disconnectQuery();

            if (result != null && result.text != null)
            {
                foreach (var entry in result.text)
                {
                    string[] values = entry.Split('^');
                    string name = values[1];
                
                    Shared.Model.Service service = new Shared.Model.Service() { Name = name };
                    serviceList.Add(service);
                }
            }

            return serviceList;
        }

        public List<AppointmentTypeCategory> GetAppointmentTypeCategories()
        {
            List<AppointmentTypeCategory> list = new List<AppointmentTypeCategory>();

            // doesn't work on our instance
//            prepareSchedulingConnection();
//            var appTypes = _SchedulingService.getAppointmentTypes("");
            prepareQueryConnection();

            var result = _QueryService.ddrLister("409.1", "", "@;.001;.01;3", "P", "", "", "", "", "", "");

            if (result != null && result.text != null)
            {
                foreach (var entry in result.text)
                {
                    string[] values = entry.Split('^');
                    string vistaCode = values[0];
                    string name = values[1];
                    string inactive = values[2];

                    if (!string.Equals(inactive, "1"))
                    {
                        AppointmentTypeCategory newAppTypeCat = new AppointmentTypeCategory() { Name = name, VistaId = vistaCode, VistaSiteId=Connection.SiteID };
                        list.Add(newAppTypeCat);
                    }

                }
            }

            disconnectQuery();
            return list;
        }

        public List<CancellationReason> getCancellationReasons()
        {
            List<CancellationReason> list = new List<CancellationReason>();

            prepareQueryConnection();

            var result = _QueryService.ddrLister("409.2", "", "@;.01I;.01;2;4", "P", "", "", "", "", "", "");

            if (result != null && result.text != null)
            {
                foreach (var entry in result.text)
                {
                    string[] values = entry.Split('^');
                    string vistaCode = values[0];
                    string name = values[1];
                    string who = values[3];
                    string inactive = values[4];

                    if (!string.Equals(inactive, "1"))
                    {
                        CancellationReason newItem = new CancellationReason() { Name = name, VistaId = vistaCode, Who = who, VistaSiteId=Connection.SiteID };
                        list.Add(newItem);
                    }
                }
            }

            disconnectQuery();
            return list;
        }

        public List<StopCode> getStopCodes()
        {
            List<StopCode> list = new List<StopCode>();

            prepareQueryConnection();

            var result = _QueryService.ddrLister("40.7", "", "@;.01I;.01;2;5", "P", "", "", "", "", "", "");

            if (result != null && result.text != null)
            {
                foreach (var entry in result.text)
                {
                    string[] values = entry.Split('^');
                    string vistaCode = values[0];
                    string name = values[1];
                    string inactiveDateStr = values[3];
                    string restrictionType = values[4];

                    DateTime? inactiveDate = Utils.FromMDWSDate(inactiveDateStr);

                    if (!inactiveDate.HasValue || inactiveDate.Value >= DateTime.Today)
                    {
                        StopCode newItem = new StopCode() { Name = name, Code = vistaCode, VistaSiteId = Connection.SiteID };
                        if (restrictionType.Equals("P") || restrictionType.Equals("Primary"))
                        {
                            newItem.RestrictionType = RestrictionType.Primary;
                        }
                        else if (restrictionType.Equals("S") || restrictionType.Equals("Secondary"))
                        {
                            newItem.RestrictionType = RestrictionType.Secondary;
                        }
                        else if (restrictionType.Equals("E") || restrictionType.Equals("Either"))
                        {
                            newItem.RestrictionType = RestrictionType.Either;
                        }
                        list.Add(newItem);
                    }
                }
            }

            disconnectQuery();
            return list;
        }

        public List<Holiday> GetHolidays()
        {
            prepareQueryConnection();

            var result = _QueryService.ddrLister("40.5", "", "@;.01;2", "P", "", "", "", "", "", "");
            disconnectQuery();
            var holidays = new List<Holiday>();
            DateTime earliestHolidayToAdd = new DateTime(2010, 1,1);

            if (result != null && result.text != null)
            {
                foreach (var entry in result.text)
                {
                    string[] values = entry.Split('^');
                    DateTime? date = Utils.FromMDWSDate(values[0]);
                    string name = values[2];

                    if (date.HasValue && date > earliestHolidayToAdd)
                    {
                        holidays.Add(new Holiday() { Name = name, Date = date.Value });
                    }
                }
            }
            return holidays;
        }

        // returns a list of providers, but only populated with minimal info pulled from Vista
        public List<Provider> GetAllProviders()
        {
            var vistaResults = VistaQuery("200", "", "@;.01;9;41.99");

            var providerList = new List<Provider>();
            foreach (var result in vistaResults)
            {
                string[] values = result.Split('^');
                string duz = values[0];
                string name = values[1];
                string ssn = values[2];
                string npi = values[3];
                string fname, mname, lname;
                Utils.SplitName(name, out fname, out mname, out lname);

                var provider = new Provider() {  NPI = npi, Person = new Person() { FirstName = fname, MiddleName = mname, LastName = lname, SSN = ssn } };
                provider.AddProviderSite(this.Connection.SiteID,duz);

                providerList.Add(provider);
            }

            return providerList;
        }

        public Provider LoadProvider(string duz)
        {
            prepareQueryConnection();
            var fields = VistaQuery_SpecificRecord("200", duz, "*", false);
            string first, middle, last;
            Utils.SplitName(fields[".01"], out first, out middle, out last);
            var provider = new Provider()
            {
                NPI = fields["41.99"],
                Person = new Person()
                {
                    FirstName = first,
                    MiddleName = middle,
                    LastName = last,
                    SSN = fields["9"],
                    Phone = fields[".132"],
                    Email = fields[".151"],
                    Sex = fields["4"],
                    DateOfBirth = Utils.FromMDWSDate(fields["5"]),                     
                    Address = new Address()
                    {
                        Street1 = fields[".111"],
                        Street2 = fields[".112"],
                        Street3 = fields[".113"],
                        City = fields[".114"],
                        State = GetStateAbbr(fields[".115"]),
                        Zip = fields[".116"],
                    }
                }
            };
            disconnectQuery();
            provider.AddProviderSite(Connection.SiteID, duz);
            return provider;
        }

//        public string AddProvider(Provider provider)
//        {
//            prepareQueryConnection();
//  //          var result = _QueryService.ddrLister("200", "", "@;.01", "P", "", "", "", "", "", "SET DIC(0)=\"\" DO EN^DDIOL(DIC(0))");

////            var result = _QueryService.getVariableValue("^VA(200,0)");

//            Dictionary<String, String> fieldsAndValues = new Dictionary<string, string>();
//            fieldsAndValues.Add(".01", "LASTNAME,FIRSTNAME");
//            //fieldsAndValues.Add("1", "FL");
//            //fieldsAndValues.Add("28", "test");

//            String vistaFile = "200";

//            string newId = create(fieldsAndValues, vistaFile, "");

//            disconnectQuery();
//            return newId;
//        }

        public string AddClinic(Resource resource, Provider provider)
        {
            if (resource.Type != ResourceType.Provider)
            {
                throw new ArgumentException("Only provider resources should be passed to AddClinic");
            }
            if (provider == null)
            {
                throw new ArgumentException("Invalid provider passed in to AddClinic");
            }
            string duz = provider.GetDUZForVistASite(resource.Section.Facility.Site.VistaSiteId);
            if (string.IsNullOrEmpty(duz))
            {
                throw new ArgumentException("Provider is not associated with the same site as the resource");
            }
            string clinicName = string.Empty;
            if (!String.IsNullOrEmpty(resource.Section.Facility.ShortName))
            {
                clinicName = resource.Section.Facility.ShortName.Substring(0, 3) + "/";
            }
            if (!String.IsNullOrEmpty(resource.Section.Abbreviation))
            {
                clinicName += resource.Section.Abbreviation + "/";
            }
            clinicName += resource.Name;

            var facility = resource.Section.Facility;

            prepareQueryConnection();
            Dictionary<String, String> fieldsAndValues = new Dictionary<string, string>();
            fieldsAndValues.Add(".01", clinicName);
            fieldsAndValues.Add("2", "C");
            fieldsAndValues.Add("2.1", "CLINIC");
            fieldsAndValues.Add("4", facility.VistaId );
            fieldsAndValues.Add("8", resource.Section.DSSPrimaryStopCode);
            fieldsAndValues.Add("10", resource.Section.Location);
            fieldsAndValues.Add("16", duz);
            // 1920.9	availability flag       need to set this?


            String vistaFile = "44";

            string newId = create(fieldsAndValues, vistaFile, "");

            disconnectQuery();
            return newId;
        }

        public bool SetClinicActive(Resource resource, bool ActiveState, DateTime effectiveDate)
        {
            if (resource.Type != ResourceType.Provider)
            {
                throw new ArgumentException("Only provider resources should be synced to VistA");
            }
                        
            prepareQueryConnection();
            string deactiveDate = Utils.ToMDWSDate((ActiveState ? null : new DateTime?(effectiveDate)));

            Dictionary<String, String> fieldsAndValues = new Dictionary<string, string>();
            fieldsAndValues.Add("2505", deactiveDate);

            String vistaFile = "44";
            update(fieldsAndValues, vistaFile, resource.VistaClinicIEN);
            disconnectQuery();
            return true;
        }

        public bool UpdateClinicDSSCode(Resource resource)
        {
            if (resource.Type != ResourceType.Provider)
            {
                throw new ArgumentException("Only provider resources should be synced to VistA");
            }

            prepareQueryConnection();
            Dictionary<String, String> fieldsAndValues = new Dictionary<string, string>();
            fieldsAndValues.Add("8", resource.Section.DSSPrimaryStopCode);

            String vistaFile = "44";
            update(fieldsAndValues, vistaFile, resource.VistaClinicIEN);
            disconnectQuery();
            return true;
        }


        //public string AddPatient(Patient patient)
        //{
        //    prepareQueryConnection();
        //    Dictionary<String, String> fieldsAndValues = new Dictionary<string, string>();
        //    fieldsAndValues.Add(".01", "LASTNAME,FIRSTNAME");
        //    fieldsAndValues.Add(".02", "M");
        //    fieldsAndValues.Add(".03", "2901010");
        //    fieldsAndValues.Add(".09", "123456789");
        //    fieldsAndValues.Add(".301", "N");
        //    fieldsAndValues.Add("391", "NSC VETERAN");
        //    fieldsAndValues.Add("1901", "Y");

        //    String vistaFile = "2";

        //    string newId = create(fieldsAndValues, vistaFile, "");

        //    disconnectQuery();
        //    return newId;
        //}

        public Dictionary<String, String> read(String iens, String fields, String vistaFile)
        {
            Dictionary<String, String> result = new Dictionary<string, string>();
            StringDictionaryTO sd = _QueryService.read(iens, fields, vistaFile);
            if (sd.fault != null)
            {
                throw new Exception(sd.fault.message);
            }

            for (int i = 0; i < sd.keys.Length; i++)
            {
                result.Add(sd.keys[i], sd.values[i]); // key/values are at same index
            }
            return result;
        }

        public String create(Dictionary<String, String> fieldsAndValues, String vistaFile, String iens)
        {
            prepareQueryConnection();
            // sample serialized Dictionary<String, String>: [{\"Key\":\"Key01\",\"Value\":\"Value01\"},{\"Key\":\"Key02\",\"Value\":\"Value02\"},{\"Key\":\"Key03\",\"Value\":\"Value03\"}]
            string dictString = JsonUtils.Serialize<Dictionary<String, String>>(fieldsAndValues); // to make the API simple, we serialize our dictionary to a string. JSON serialization is widely available in all major programming languages. It would also be trivial to implement your own serializer that passed the format MDWS wants
            MedRed.MDWSAccess.QuerySvc.TextTO result = _QueryService.create(dictString, vistaFile, iens);
            if (result.fault != null)
            {
                throw new Exception(result.fault.message);
            }
            disconnectQuery();
            return result.text; // IEN of new record
        }

        public void update(Dictionary<String, String> fieldsAndValues, String vistaFile, String iens)
        {
            // sample serialized Dictionary<String, String>: [{\"Key\":\"Key01\",\"Value\":\"Value01\"},{\"Key\":\"Key02\",\"Value\":\"Value02\"},{\"Key\":\"Key03\",\"Value\":\"Value03\"}]
            string dictString = JsonUtils.Serialize<Dictionary<String, String>>(fieldsAndValues); // to make the API simple, we serialize our dictionary to a string. JSON serialization is widely available in all major programming languages. It would also be trivial to implement your own serializer that passed the format MDWS wants
            MedRed.MDWSAccess.QuerySvc.TextTO result = _QueryService.update(dictString, iens, vistaFile);
            if (result.fault != null)
            {
                throw new Exception(result.fault.message);
            }
            // success!
            //result.text = "OK"
        }

        public void delete(String vistaFile, String iens)
        {
            prepareQueryConnection(); 
            MedRed.MDWSAccess.QuerySvc.TextTO result = _QueryService.delete(iens, vistaFile);
            if (result.fault != null)
            {
                throw new Exception(result.fault.message);
            }
            // success!
            //result.text = "OK"
            disconnectQuery();
        }

        public List<String> VistaQuery(string file, string iens, string fields, bool initializeConnection = true)
        {
            if (initializeConnection)
            {
                prepareQueryConnection();
            }
            var result = _QueryService.ddrLister(file, iens, fields, "P", "", "", "", "", "", "");
            if (initializeConnection)
            {
                disconnectQuery();
            }

            var resultList = new List<String>();

            if (result != null && result.text != null)
            {
                foreach (var entry in result.text)
                {
                    resultList.Add(entry);
                }
            }

            return resultList;
        }

        public List<String> VistaQuery_FindMatchingRows(string file, string fields, int position, string searchValue, bool initializeConnection = true)
        {
            List<String> list = new List<string>();

            var results = VistaQuery(file, "", fields, initializeConnection);

            foreach (var result in results)
            {
                string[] values = result.Split('^');
                if (position <= values.Length && !String.IsNullOrEmpty(values[position]))
                {
                    if (values[position].Contains(searchValue))
                    {
                        list.Add(result);
                    }
                }
            }
            return list;
        }

        public Dictionary<string, string> VistaQuery_SpecificRecord(string file, string ien, string fields, bool initializeConnection = true)
        {
            if (initializeConnection)
            {
                prepareQueryConnection();
            }
            var result = read(ien, fields, file);
            if (initializeConnection)
            {
                disconnectQuery();
            }
            return result;
        }

        public void Dispose()
        {
            if (_EmrService != null)
                _disconnectEmr();
            if (_QueryService != null)
                _disconnectQuery();
            if (_SchedulingService != null)
                _disconnectScheduling();
        }

    }
}
