﻿using MedRed.CalendarAccess;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using CalDav;
using Shared.Model.Config.Calendar;
using Shared.Model;

namespace MedRed.CalendarAccessTests
{   
    /// <summary>
    ///This is a test class for CalendarAccessImplTest and is intended
    ///to contain all CalendarAccessImplTest Unit Tests
    ///</summary>
    [TestClass()]
    public class CalendarAccessImplTest
    {
        private static Uri calendarServer = new Uri("http://127.0.0.1:8080/ucaldav/user/mtwain/");
        private static string calendarUser = "mtwain";
        private static string calendarPass = "bedework";

        static string ResourceBASE = "R";
        static string PatientBASE = "P";

        private TestContext testContextInstance;

        /// <summary>
        ///Gets or sets the test context which provides
        ///information about and functionality for the current test run.
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        #region Additional test attributes
        // 
        //You can use the following additional attributes as you write your tests:
        //
        //Use ClassInitialize to run code before running the first test in the class
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //Use ClassCleanup to run code after all tests in a class have run
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //Use TestInitialize to run code before running each test
        //[TestInitialize()]
        //public void MyTestInitialize()
        //{
        //}
        //
        //Use TestCleanup to run code after each test has run
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion

        /// <summary>
        ///A test for Creating a new calendar
        ///</summary>
        [TestMethod()]
        public void NewCalendarForResourceTest()
        {
            CalendarConfiguration config = new CalendarConfiguration()
            {
                BaseCalendarUri = calendarServer,
                UserName = calendarUser,
                Password = calendarPass,
            };
            CalendarFactory.Initialize(config);
            ICalendarAccess target = CalendarFactory.GetCalendarAccess();

            var calName = "NewTestCalendar" + DateTime.Now.Ticks.ToString();
            Resource resource = new Resource()
            {
                Id = 1,
                Name = calName,
                Type = ResourceType.Provider,
                Section = new Section() {Id=4, Name = "Section2b" }
            };
            var expectedName = ResourceBASE+"/4/1/";

            var actual = target.GetCalendar(resource);

            Assert.AreEqual(resource.Id.ToString(), actual.Name);
            Assert.AreEqual(new Uri(calendarServer, expectedName).ToString(), actual.Url.AbsoluteUri.ToString());
            // how to delete this calendar now?
        }

        /// <summary>
        ///A test for Creating a new calendar
        ///</summary>
        [TestMethod()]
        public void NewCalendarForPatientTest()
        {
            CalendarConfiguration config = new CalendarConfiguration()
            {
                BaseCalendarUri = calendarServer,
                UserName = calendarUser,
                Password = calendarPass,
            };
            CalendarFactory.Initialize(config);
            ICalendarAccess target = CalendarFactory.GetCalendarAccess();

            var calName = "NewTestCalendar" + DateTime.Now.Ticks.ToString();
            Patient patient = new Patient()
            {
                Id = 3,
                ICN = "123",
                Person = new Person()
                {
                    FirstName = "Bob",
                    LastName = "Hope",
                    Sex = "M",
                    DateOfBirth = new DateTime(1970, 1, 1),
                    Email = "BobHope@hotmail.com"
                }
            };
            var expectedName = PatientBASE+"/3/";

            var actual = target.GetCalendar(patient);

            Assert.AreEqual(patient.Id.ToString(), actual.Name);
            Assert.AreEqual(new Uri(calendarServer, expectedName).ToString(), actual.Url.AbsoluteUri.ToString());
            // how to delete this calendar now?
        }

        /// <summary>
        ///A test for Creating a new calendar
        ///</summary>
//        [TestMethod()]
//        public void DeleteCalendarTest()
//        {
//            CalendarConfiguration config = new CalendarConfiguration()
//            {
//                BaseCalendarUri = calendarServer,
//                UserName = calendarUser,
//                Password = calendarPass,
//            };
//            CalendarFactory.Initialize(config);
//            ICalendarAccess target = CalendarFactory.GetCalendarAccess();

//            var calName = "CalendarToDelete";
//            Resource resource = new Resource()
//            {
//                Id = 2,
//                Name = calName,
//                Type = ResourceType.Provider,
//                Section = new Section() { Id = 5, Name = "Section1c" }
//            };
//            var expectedName = ResourceBASE + "/5/2/";

////            Assert.IsFalse(target.IsCalendarPresent(resource));

//            var actual = target.GetCalendar(resource);

//            Assert.AreEqual(resource.Id.ToString(), actual.Name);
//            Assert.AreEqual(new Uri(calendarServer, expectedName).ToString(), actual.Url.AbsoluteUri.ToString());

//            Assert.IsTrue(target.IsCalendarPresent(resource));
//            target.DeleteCalendar(resource);

//            Assert.IsFalse(target.IsCalendarPresent(resource));
//        }

        /// <summary>
        ///A test for Creating a new calendar
        ///</summary>
        [TestMethod()]
        public void LoadExistingCalendarTest()
        {
            CalendarConfiguration config = new CalendarConfiguration()
            {
                BaseCalendarUri = calendarServer,
                UserName = calendarUser,
                Password = calendarPass,
            };
            CalendarFactory.Initialize(config);
            ICalendarAccess target = CalendarFactory.GetCalendarAccess();

            var calName = "ExistingTestCalendar";
            Resource resource = new Resource()
            {
                Id = 3,
                Name = calName,
                Type = ResourceType.Provider,
                Section = new Section() { Id = 6, Name = "Section3b" }
            };
            var expectedName = ResourceBASE + "/6/3/";

            var actual = target.GetCalendar(resource);

            Assert.AreEqual(resource.Id.ToString(), actual.Name);
            Assert.AreEqual(new Uri(calendarServer, expectedName).ToString(), actual.Url.AbsoluteUri.ToString());
        }


        /// <summary>
        ///A test for FindAppointments
        ///</summary>
        [TestMethod()]
        public void FindAppointmentsEmptyTest()
        {
            CalendarConfiguration config = new CalendarConfiguration()
            {
                BaseCalendarUri = calendarServer,
                UserName = calendarUser,
                Password = calendarPass,
            };
            CalendarFactory.Initialize(config);
            ICalendarAccess target = CalendarFactory.GetCalendarAccess();

            var calName = "EmptyTestCalendar";
            Resource resource = new Resource()
            {
                Id = 4,
                Name = calName,
                Type = ResourceType.Provider,
                Section = new Section() { Id = 7, Name = "Section5" }
            };

            DateTime start = new DateTime(2013, 1, 1);
            DateTime end = new DateTime(2013, 4, 1); 

            var actual = target.FindAppointments(resource, start, end);
            Assert.IsNotNull(actual);
            Assert.AreEqual(actual.Length, 0);
        }

        /// <summary>
        ///A test for FindAppointments
        ///</summary>
        [TestMethod()]
        public void FindAppointmentsInCalendarTest()
        {
            CalendarConfiguration config = new CalendarConfiguration()
            {
                BaseCalendarUri = calendarServer,
                UserName = calendarUser,
                Password = calendarPass,
            };
            CalendarFactory.Initialize(config);
            ICalendarAccess target = CalendarFactory.GetCalendarAccess();

            Event[] events = new Event[]
            {
                new Event() { Description = "Event 1", Summary = "Summary 1", Start = new DateTime(2013, 1, 2, 9, 0, 0), End = new DateTime(2013, 1, 2, 10, 0, 0)},
                new Event() { Description = "Event 2", Summary = "Summary 2", Start = new DateTime(2013, 2, 3, 10, 0, 0), End = new DateTime(2013, 2, 3, 11, 0, 0)},
                new Event() { Description = "Event 3", Summary = "Summary 3", Start = new DateTime(2013, 3, 4, 11, 0, 0), End = new DateTime(2013, 3, 4, 12, 0, 0)},
            };

            var calName = "ExistingTestCalendar";
            Resource resource = new Resource()
            {
                Id = 5,
                Name = calName,
                Type = ResourceType.Provider,
                Section = new Section() { Id = 8, Name = "Section4" }
            };

            DateTime start = new DateTime(2013, 1, 1);
            DateTime end = new DateTime(2013, 4, 1);

            var actual = target.FindAppointments(resource, start, end);
            if (actual != null && actual.Length == 0)
            {
                // we need to add these events if they aren't present. 
                // Not good test design, but it's a quick way to make sure these three events are present in this calendar.
                target.SaveAppointment(resource, events[0]);
                target.SaveAppointment(resource, events[1]);
                target.SaveAppointment(resource, events[2]);

                actual = target.FindAppointments(resource, start, end);
            }

            Assert.IsNotNull(actual);
            Assert.AreEqual(actual.Length, 3);
        }

        /// <summary>
        ///A test for SaveAppointment
        ///</summary>
        [TestMethod()]
        public void SaveAppointmentInCalendarTest()
        {
            CalendarConfiguration config = new CalendarConfiguration()
            {
                BaseCalendarUri = calendarServer,
                UserName = calendarUser,
                Password = calendarPass,
            };
            CalendarFactory.Initialize(config);
            ICalendarAccess target = CalendarFactory.GetCalendarAccess();
                        
            var calName = "CreationTestCalendar";
            Resource resource = new Resource()
            {
                Id = 16,
                Name = calName,
                Type = ResourceType.Provider,
                Section = new Section() { Id= 19, Name = "Section20" }
            };

            Patient patient = new Patient()
            {
                Id = 13,
                ICN = "123",
                Person = new Person()
                {
                    FirstName = "Bob",
                    LastName = "Hope",
                    Sex = "M",
                    DateOfBirth = new DateTime(1970, 1, 1),
                    Email = "BobHope@hotmail.com"
                }
            };

            Appointment appointment = new Appointment()
            {
                AppointmentType = new AppointmentType() { Name = "Blah", PatientInstructions = { } },
                Length = 60,
                Time = new DateTime(2013, 4, 1, 10, 0, 0),
                Patient = patient,
                Status = AppointmentStatus.Scheduled,
            };

            appointment.Resources.Add(resource);

            Assert.IsTrue(target.SaveAppointment(appointment));

            DateTime startTime = appointment.Time;
            DateTime endTime = startTime.AddMinutes(appointment.Length);

            var actual = target.GetAppointment(appointment);
            Assert.IsNotNull(actual);

            var patAppts = target.FindAppointments(appointment.Patient, startTime, endTime);
            Assert.IsTrue(patAppts.Length>=1);

            foreach (var res in appointment.Resources)
            {
                var resAppts = target.FindAppointments(res, startTime, endTime);
                Assert.IsTrue(resAppts.Length>=1);
            }
        }

        ///// <summary>
        /////A test for SaveAppointment
        /////</summary>
        //[TestMethod()]
        //public void SaveMultiResourceAppointmentInCalendarTest()
        //{
        //    CalendarConfiguration config = new CalendarConfiguration()
        //    {
        //        BaseCalendarUri = calendarServer,
        //        UserName = calendarUser,
        //        Password = calendarPass,
        //    };
        //    CalendarFactory.Initialize(config);
        //    ICalendarAccess target = CalendarFactory.GetCalendarAccess();

        //    var calName = "CreationTestCalendar";
        //    Resource resource = new Resource()
        //    {
        //        Id = 6,
        //        Name = calName,
        //        Type = ResourceType.Provider,
        //        Section = new Section() { Id = 9, Name = "Section20" }
        //    };

        //    var startTime = DateTime.Now;
        //    var endTime = DateTime.Now.AddHours(1);
        //    var newEvent = new Event()
        //    {
        //        Description = "Test event created " + DateTime.Now.ToString(),
        //        Summary = "New Test Event",
        //        Start = startTime,
        //        End = endTime,
        //    };
        //    Assert.IsTrue(target.SaveAppointment(resource, newEvent));

        //    var actual = target.FindAppointments(resource, startTime, endTime);
        //    Assert.IsTrue(actual.Length > 0);
        //}

        /// <summary>
        ///A test for SaveAppointment
        ///</summary>
        [TestMethod()]
        public void UpdateAppointmentInCalendarTest()
        {
            CalendarConfiguration config = new CalendarConfiguration()
            {
                BaseCalendarUri = calendarServer,
                UserName = calendarUser,
                Password = calendarPass,
            };
            CalendarFactory.Initialize(config);
            ICalendarAccess target = CalendarFactory.GetCalendarAccess();

            var calName = "CreationTestCalendar";
            Resource resource = new Resource()
            {
                Id = 16,
                Name = calName,
                Type = ResourceType.Provider,
                Section = new Section() { Id = 19, Name = "Section20" }
            };

            Patient patient = new Patient()
            {
                Id = 13,
                ICN = "123",
                Person = new Person()
                {
                    FirstName = "Bob",
                    LastName = "Hope",
                    Sex = "M",
                    DateOfBirth = new DateTime(1970, 1, 1),
                    Email = "BobHope@hotmail.com"
                }
            };

            Appointment appointment = new Appointment()
            {
                AppointmentType = new AppointmentType() { Name = "Blah", PatientInstructions = { } },
                Length = 60,
                Time = new DateTime(2013, 5, 1, 10, 0, 0),
                Patient = patient,
                Status = AppointmentStatus.Scheduled,
            };

            appointment.Resources.Add(resource);

            Assert.IsTrue(target.SaveAppointment(appointment));

            DateTime startTime = appointment.Time;
            DateTime endTime = startTime.AddMinutes(appointment.Length);

            var actual = target.GetAppointment(appointment);
            Assert.IsNotNull(actual);
            
            appointment.Time = DateTime.Now.AddHours(1);

            target.UpdateAppointment(appointment);
        }

        /// <summary>
        ///A test for SaveAppointment
        ///</summary>
        [TestMethod()]
        public void DeleteAppointmentInCalendarTest()
        {
            CalendarConfiguration config = new CalendarConfiguration()
            {
                BaseCalendarUri = calendarServer,
                UserName = calendarUser,
                Password = calendarPass,
            };
            CalendarFactory.Initialize(config);
            ICalendarAccess target = CalendarFactory.GetCalendarAccess();

            var calName = "CreationTestCalendar";
            Resource resource = new Resource()
            {
                Id = 16,
                Name = calName,
                Type = ResourceType.Provider,
                Section = new Section() { Id = 19, Name = "Section20" }
            };

            Patient patient = new Patient()
            {
                Id = 13,
                ICN = "123",
                Person = new Person()
                {
                    FirstName = "Bob",
                    LastName = "Hope",
                    Sex = "M",
                    DateOfBirth = new DateTime(1970, 1, 1),
                    Email = "BobHope@hotmail.com"
                }
            };

            Appointment appointment = new Appointment()
            {
                AppointmentType = new AppointmentType() { Name = "Blah", PatientInstructions = { } },
                Length = 60,
                Time = new DateTime(2013, 5, 1, 10, 0, 0),
                Patient = patient,
                Status = AppointmentStatus.Scheduled,
            };

            appointment.Resources.Add(resource);

            Assert.IsTrue(target.SaveAppointment(appointment));

            target.DeleteAppointment(appointment);
        }
    }
}
