﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using Shared.Model;
using Shared.Model.Config.Calendar;

namespace MedRed.CalendarAccess
{
    class CalendarAccessImpl :  ICalendarAccess
    {
        private CalendarConfiguration Config { get; set; }

        private bool IsInitilized
        {
            get
            {
                return !(Config == null || Config.BaseCalendarUri == null || Config.UserName == null || Config.Password == null);
            }
        }

        internal CalendarAccessImpl(CalendarConfiguration config)
        {
            Config = config;
        }

        public bool IsCaldavEnabled()
        {
            return (Config.BaseCalendarUri != null && !String.IsNullOrEmpty(Config.BaseCalendarUri.ToString()));
        }

        public bool IsCalendarPresent(Resource resource)
        {
            return IsCalendarPresent(GetCalendarFullPath(resource));
        }

        public bool IsCalendarPresent(Patient patient)
        {
            return IsCalendarPresent(GetCalendarFullPath(patient));
        }

        private bool IsCalendarPresent(string fullCalendarPath)
        {
            if (!IsInitilized)
            {
                throw new InvalidOperationException("Call Initialize first");
            }
            var cal = LoadCalendar(fullCalendarPath);
            return (cal != null);
        }

        //public bool IsCalendarPresent(Resource resource)
        //{
        //    if (!IsInitilized)
        //    {
        //        throw new InvalidOperationException("Call Initialize first");
        //    }
        //    var cal = LoadCalendar(fullCalendarPath);
        //    return (cal != null);
        //}

        public CalDav.Client.Calendar GetCalendar(Resource resource)
        {
            if (!IsInitilized)
            {
                throw new InvalidOperationException("Call Initialize first");
            }
            var cal = LoadCalendar(GetCalendarFullPath(resource));
            if (cal == null)
            {
                cal = CreateCalendar(GetCalendarRootPath(resource), GetCalendarName(resource));
            }
            return cal;
        }

        public CalDav.Client.Calendar GetCalendar(Patient patient)
        {
            if (!IsInitilized)
            {
                throw new InvalidOperationException("Call Initialize first");
            }
            var cal = LoadCalendar(GetCalendarFullPath(patient));
            if (cal == null)
            {
                try
                {
                    cal = CreateCalendar(GetCalendarRootPath(patient), GetCalendarName(patient));
                }
                catch (Exception)
                {
                    // try again?
                    System.Threading.Thread.Sleep(1000);
                    cal = CreateCalendar(GetCalendarRootPath(patient), GetCalendarName(patient));
                }
            }
            return cal;
        }

        CalDav.Client.Calendar LoadCalendar(string fullCalendarPath)
        {
            if (!IsInitilized)
            {
                throw new InvalidOperationException("Call Initialize first");
            }
            CalDav.Client.Calendar cal = new CalDav.Client.Calendar()
            {
                Url = new Uri(Config.BaseCalendarUri, fullCalendarPath), //GetCalendarFullPath(resource)),
                Credentials = new System.Net.NetworkCredential(Config.UserName, Config.Password), 
//                Name = resource.Name, 
            };

            try
            {
                cal.Initialize();
            }
            catch (Exception)
            {
                return null;
            }

            return cal;
        }

        CalDav.Client.Calendar CreateCalendar(string calendarRootName, string calendarName)
        {
            if (!IsInitilized)
            {
                throw new InvalidOperationException("Call Initialize first");
            }
            try
            {
                String path = calendarRootName;

                //WebDAVClient webDAVClient = new WebDAVClient();
                //webDAVClient.Server = Config.BaseCalendarUri.Scheme + "://" + Config.BaseCalendarUri.Authority;
                //webDAVClient.BasePath = Config.BaseCalendarUri.AbsolutePath;
                //webDAVClient.User = Config.UserName;
                //webDAVClient.Pass = Config.Password;

                //WebDAVUtils webDavUtils = new WebDAVUtils(webDAVClient);

                WebDAV webdav = new WebDAV(Config.BaseCalendarUri, Config.UserName, Config.Password);

                string[] folders = path.Split('/');
                string currPath = string.Empty;
                bool created = false;
                for (int i=0; i < folders.Count(); i++)
                {
                    if (!string.IsNullOrEmpty(folders[i]))
                    {
                        if (!String.IsNullOrEmpty(currPath))
                        {
                            currPath = currPath+"/";
                        }
                        currPath = currPath + folders[i];
                        try
                        {
                            created = webdav.CreateDirectory(currPath);
                        }
                        catch (Exception e)
                        {
                            created = false;
                        }
                    }
                }

                var server = new CalDav.Client.Server(new Uri(Config.BaseCalendarUri, path), Config.UserName, Config.Password);
                server.CreateCalendar(calendarName);

                var cal = LoadCalendar(calendarRootName+calendarName+"/");
                return cal;                 
            }
            catch (Exception)
            {
                throw;
            }            
        }

        public bool DeleteCalendar(Resource resource)
        {
            return DeleteCalendar(GetCalendarFullPath(resource));
        }

        public bool DeleteCalendar(Patient patient)
        {
            return DeleteCalendar(GetCalendarFullPath(patient));
        }

        private bool DeleteCalendar(string fullCalendarPath)
        {
            if (!IsInitilized)
            {
                throw new InvalidOperationException("Call Initialize first");
            }
            //WebDAVClient webDAVClient = new WebDAVClient();
            //webDAVClient.Server = Config.BaseCalendarUri.Scheme + "://" + Config.BaseCalendarUri.Authority;
            //webDAVClient.BasePath = Config.BaseCalendarUri.AbsolutePath;
            //webDAVClient.User = Config.UserName;
            //webDAVClient.Pass = Config.Password;

            //WebDAVUtils webDavUtils = new WebDAVUtils(webDAVClient);

            //webDavUtils.DeletePath(fullCalendarPath);
            WebDAV webdav = new WebDAV(Config.BaseCalendarUri, Config.UserName, Config.Password);

            webdav.DeletePath(fullCalendarPath);

            return true;
        }

        public CalDav.Event[] FindAppointments(Resource resource, DateTime start, DateTime end)
        {
            if (!IsInitilized)
            {
                throw new InvalidOperationException("Call Initialize first");
            }
            var cal = GetCalendar(resource);
            var events = cal.Search(CalDav.CalendarQuery.SearchEvents(start, end));
            return events.SelectMany(x => x.Events).ToArray();
        }

        public CalDav.Event[] FindAppointments(Patient patient, DateTime start, DateTime end)
        {
            if (!IsInitilized)
            {
                throw new InvalidOperationException("Call Initialize first");
            }
            var cal = GetCalendar(patient);
            var events = cal.Search(CalDav.CalendarQuery.SearchEvents(start, end));
            return events.SelectMany(x => x.Events).ToArray();
        }

        public CalDav.Event GetAppointment(Appointment appointment)
        {
            if (!IsInitilized)
            {
                throw new InvalidOperationException("Call Initialize first");
            }
            var cal = GetCalendar(appointment.Patient);
//            var theEvent = cal.GetAll();

            var theEvent = cal.GetObject(appointment.CaldavId);

            if (theEvent.Count > 0 && theEvent.First().Events.Count > 0)
                return theEvent.First().Events.First();
            else
                return null;
        }
        

        public bool SaveAppointment(Resource resource, CalDav.Event appointment)
        {
            if (!IsInitilized)
            {
                throw new InvalidOperationException("Call Initialize first");
            }
            try
            {
                var cal = GetCalendar(resource);
                cal.Save(appointment);
                return true;
            }
            catch (Exception)
            {
                throw;
            }
        }

        public bool DeleteAppointment(Resource resource, CalDav.Event appointment)
        {
            throw new NotImplementedException();

            if (!IsInitilized)
            {
                throw new InvalidOperationException("Call Initialize first");
            }
            try
            {
                var cal = GetCalendar(resource);
//                cal.(appointment);
                return true;
            }
            catch (Exception)
            {
                throw;
            }
        }

        private string GetCalendarRootPath(Resource resource)
        {
            return "R/"+resource.Section.Id + "/";
        }

        private string GetCalendarName(Resource resource)
        {
            return resource.Id.ToString();
        }

        private string GetCalendarFullPath(Resource resource)
        {
            return GetCalendarRootPath(resource) + GetCalendarName(resource) + "/";
        }

        private string GetCalendarRootPath(Patient patient)
        {
            return "P/";
        }

        private string GetCalendarName(Patient patient)
        {
            return patient.Id.ToString();
        }

        private string GetCalendarFullPath(Patient patient)
        {
            return GetCalendarRootPath(patient) + GetCalendarName(patient) + "/";
        }

        //public bool SaveAppointment(Collection<Resource> resources, CalDav.Event appointment)
        //{
        //    //if (!IsInitilized)
        //    //{
        //    //    throw new InvalidOperationException("Call Initialize first");
        //    //}
        //    //try
        //    //{
        //    //    var cal = GetCalendar(resource);
        //    //    cal.Save(appointment);
        //    //    return true;
        //    //}
        //    //catch (Exception)
        //    //{
        //    //    throw;
        //    //}
        //    return true;
        //}
        
        public bool SaveAppointment(Appointment appointment)
        {
            CalDav.Event calEvent = CreateEvent(appointment);

            try
            {
                var cal = GetCalendar(appointment.Patient);
                cal.Save(calEvent);
                appointment.CaldavId = calEvent.UID;

                foreach (var resource in appointment.Resources)
                {
                    var rescal = GetCalendar(resource);
                    CalDav.Event resEvent = CreateProviderEvent(appointment);
                    rescal.Save(resEvent);
                }
                return true;
            }
            catch (Exception)
            {
                throw;
            }
        }

        // why do i get this error here?
        // Unable to save event: PreconditionFailed
        public bool UpdateAppointment(Appointment appointment)
        {
            DeleteAppointment(appointment);
            return SaveAppointment(appointment);
//            CalDav.Event calEvent = GetAppointment(appointment);
//            if (calEvent == null)
//                return false;
//            UpdateEvent(ref calEvent, appointment);
//            try
//            {
//                var cal = GetCalendar(appointment.Patient);
//                cal.Save(calEvent);
////                appointment.CaldavId = calEvent.UID;

//                foreach (var resource in appointment.Resources)
//                {
//                    var rescal = GetCalendar(resource);
//                    CalDav.Event resEvent = CreateEvent(appointment);
//                    rescal.Save(resEvent);
//                }
//                return true;
//            }
//            catch (Exception)
//            {
//                throw;
//            }
        }

        public bool DeleteAppointment(Appointment appointment)
        {
            CalDav.Event calEvent = GetAppointment(appointment);
            if (calEvent == null)
                return false;
            try
            {
                var cal = GetCalendar(appointment.Patient);
                Uri file = new Uri(cal.Url, calEvent.UID + ".ics");
                //WebDAVClient webDAVClient = new WebDAVClient();
                //webDAVClient.Server = Config.BaseCalendarUri.Scheme + "://" + Config.BaseCalendarUri.Authority;
                //webDAVClient.BasePath = Config.BaseCalendarUri.AbsolutePath;
                //webDAVClient.User = Config.UserName;
                //webDAVClient.Pass = Config.Password;

                //WebDAVUtils webDavUtils = new WebDAVUtils(webDAVClient);
                //webDavUtils.DeletePath(file.AbsoluteUri);

                //foreach (var resource in appointment.Resources)
                //{
                //    var rescal = GetCalendar(resource);
                //    CalDav.Event resEvent = CreateEvent(appointment);
                //    rescal.Save(resEvent);
                //}

                WebDAV webdav = new WebDAV(Config.BaseCalendarUri, Config.UserName, Config.Password);

                webdav.DeletePath(file.AbsolutePath);

                return true;
            }
            catch (Exception)
            {
                throw;
            }
        }

        private CalDav.Event CreateEvent(Appointment appointment)
        {
            CalDav.Event calEvent = new CalDav.Event();
            calEvent.Class = CalDav.Classes.PRIVATE;
            calEvent.Created = DateTime.UtcNow;

            Resource providerResource = appointment.Resources.FirstOrDefault(r => r.Type == ResourceType.Provider);
            if (providerResource != null)
            {
                calEvent.Summary = "Appointment with " + providerResource.Name;
            }
            if (appointment.AppointmentType != null)
            {
                calEvent.Description = "Appointment Type: " + appointment.AppointmentType.Name;

                if (appointment.AppointmentType.PatientInstructions.Count > 0)
                {
                    StringBuilder sb = new StringBuilder();

                    sb.AppendLine(calEvent.Description);
                    sb.AppendLine("Patient Instructions");
                    foreach (var instruction in appointment.AppointmentType.PatientInstructions)
                    {
                        sb.AppendLine(instruction.Instructions);
                    }
                    calEvent.Description = sb.ToString();
                }
            }

            calEvent.Start = appointment.Time;
            calEvent.End = appointment.Time.AddMinutes(appointment.Length);

            calEvent.LastModified = DateTime.UtcNow;

            calEvent.Status = CalDav.Statuses.COMPLETED;
            if (appointment.Status == AppointmentStatus.Cancelled)
                calEvent.Status = CalDav.Statuses.CANCELED;

            return calEvent;
        }

        private CalDav.Event CreateProviderEvent(Appointment appointment)
        {
            CalDav.Event calEvent = CreateEvent(appointment);

            calEvent.Summary = "Appointment with " + appointment.Patient.Person.ToString();

            return calEvent;
        }        

        private bool UpdateEvent(ref CalDav.Event calEvent, Appointment appointment)
        {
            Resource providerResource = appointment.Resources.FirstOrDefault(r => r.Type == ResourceType.Provider);
            if (providerResource != null)
            {
                calEvent.Summary = "Appointment with " + providerResource.Name;
            }
            calEvent.Description = "Appointment Type: " + appointment.AppointmentType.Name;
            if (appointment.AppointmentType.PatientInstructions.Count > 0)
            {
                StringBuilder sb = new StringBuilder();

                sb.AppendLine(calEvent.Description);
                sb.AppendLine("Patient Instructions");
                foreach (var instruction in appointment.AppointmentType.PatientInstructions)
                {
                    sb.AppendLine(instruction.Instructions);
                }
                calEvent.Description = sb.ToString();
            }

            calEvent.Start = appointment.Time;
            calEvent.End = appointment.Time.AddMinutes(appointment.Length);

            calEvent.LastModified = DateTime.UtcNow;

            calEvent.Status = CalDav.Statuses.COMPLETED;

            return true;
        }
    }
}
