﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;

using MedRed.Services;
using Shared.Model.Config.MDWS;
using Shared.Model;
using BT.Health.SchedulingWeb.UISession;
using MedRed.Services.Interfaces;

namespace BT.Health.SchedulingWeb.User_Controls
{
    public partial class wucChangeProvider : System.Web.UI.UserControl
    {
        /// <summary>
        /// Raised after a provider has been changed
        /// </summary>
        public event EventHandler<ProviderChangedArgs> ProviderChanged;

        protected void OnProviderChanged(ProviderChangedArgs e)
        {
            EventHandler<ProviderChangedArgs> eh = ProviderChanged;

            if (eh != null)
            {
                eh(this, e);
            }
        }

        protected void Page_Load(object sender, EventArgs e)
        {
            lblMessage.Text = string.Empty;
        }

        protected void Page_PreRender(object sender, EventArgs e)
        {
            // can only update when a DDL item selected
            btnChangeProvider.Enabled = (ddlNewProvider.SelectedValueAsInt() > 0);
        }

        /// <summary>
        /// Initialise for changing a provider in multiple (group) appointments
        /// </summary>
        /// <param name="appointments"></param>
        public void InitialiseForAppointment(List<Appointment> appointments)
        {
            if (appointments.Count > 0)
            {
                // for a group appointment
                hfIsGroupAppointment.Value = true.ToString();

                System.Text.StringBuilder sb = new System.Text.StringBuilder();

                foreach (var a in appointments)
                {
                    if (sb.Length > 0)
                        sb.Append(",");

                    sb.Append(a.Id.ToString());
                }

                // save a list of appointment ID's
                hfAppointmentId.Value = sb.ToString();

                // show appointment
                mv.ActiveViewIndex = 0;

                lblNewProviderType.Text = "New Group Provider";

                // show details using first appointment
                tdAppointment.InnerHtml = GetAppointmentDetails(appointments[0]);

                // show current provider (select first one found)
                var provider = (from p in appointments[0].Resources
                                where p.Type == ResourceType.Provider
                                select p).FirstOrDefault();

                if (provider != null)
                {
                    lblCurrentProvider.Text = provider.Name;
                    hfCurrentProviderId.Value = provider.Id.ToString();
                }
                else
                {
                    lblCurrentProvider.Text = "There is no current provider";
                    hfCurrentProviderId.Value = string.Empty;
                }

                //----------------------------------------------
                // list alternatives

                ddlNewProvider.DataValueField = "Id";
                ddlNewProvider.DataTextField = "Name";

                var factory = Helper.GetFactory(this.Page);
                var resourceServices = factory.GetResourceService();

                ddlNewProvider.SelectedIndex = -1;

                // alternate providers for groups is different than single appointments
                // group appountments must have providers with capacity greater than one
                // We don't want providers than can be over-booked for use in groups
                ddlNewProvider.DataSource = resourceServices.FindAlternateAvailableGroupProviders(appointments[0]).ToList();
                ddlNewProvider.DataBind();

                if (ddlNewProvider.Items.Count == 0)
                {
                    ddlNewProvider.Items.Add(new ListItem("No available providers", "-1"));
                }

                ddlNewProvider.SelectedIndex = 0;
            }
            else
            {
                // no appointment
                mv.ActiveViewIndex = 1;
            }
        }

        /// <summary>
        /// Initialise for changing a provider in an appointment
        /// </summary>
        /// <param name="a"></param>
        public void InitialiseForAppointment(Appointment a)
        {
            if (a != null)
            {
                // for a single appointment
                hfIsGroupAppointment.Value = false.ToString();

                // show appointment
                mv.ActiveViewIndex = 0;

                lblNewProviderType.Text = "New Provider";

                // save ID for later
                hfAppointmentId.Value = a.Id.ToString();

                // show details
                tdAppointment.InnerHtml = GetAppointmentDetails(a);

                // show current provider (select first one found)
                var provider = (from p in a.Resources
                                where p.Type == ResourceType.Provider
                                select p).FirstOrDefault();

                if (provider != null)
                {
                    lblCurrentProvider.Text = provider.Name;
                    hfCurrentProviderId.Value = provider.Id.ToString();
                }
                else
                {
                    lblCurrentProvider.Text = "There is no current provider";
                    hfCurrentProviderId.Value = string.Empty;
                }

                //----------------------------------------------
                // list alternatives

                ddlNewProvider.DataValueField = "Id";
                ddlNewProvider.DataTextField = "Name";

                var factory = Helper.GetFactory(this.Page);
                var resourceServices = factory.GetResourceService();

                ddlNewProvider.SelectedIndex = -1;
                ddlNewProvider.DataSource = resourceServices.FindAlternateAvailableProviders(a).ToList();
                ddlNewProvider.DataBind();

                if (ddlNewProvider.Items.Count == 0)
                {
                    ddlNewProvider.Items.Add(new ListItem("No available providers", "-1"));
                }

                ddlNewProvider.SelectedIndex = 0;
            }
            else
            {
                // no appointment
                mv.ActiveViewIndex = 1;
            }
        }

        protected void btnChangeProvider_Click(object sender, EventArgs e)
        {
            try
            {
                if (hfIsGroupAppointment.Value == true.ToString())
                {
                    ChangeProviderForGroupAppointment();
                }
                else
                {
                    ChangeProviderForSingleAppointment();
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void ChangeProviderForSingleAppointment()
        {
            int appointmentId = 0;

            if (int.TryParse(hfAppointmentId.Value, out appointmentId))
            {
                // get appointment
                var factory = Helper.GetFactory(this.Page);
                var appointmentService = factory.GetAppointmentService();
                var appointment = appointmentService.Get(appointmentId);

                if (appointment != null)
                {
                    // see if there is a provider to remove
                    int currentProviderId = 0;

                    if (int.TryParse(hfCurrentProviderId.Value, out currentProviderId))
                    {
                        // remove current provider, if there is one
                        var provider = (from p in appointment.Resources
                                        where p.Type == ResourceType.Provider
                                        select p).FirstOrDefault();

                        if (provider != null)
                        {
                            appointment.Resources.Remove(provider);
                        }

                        // get new provider
                        var resourceServices = factory.GetResourceService();
                        var newProvider = resourceServices.Get(ddlNewProvider.SelectedValueAsInt());

                        if (newProvider != null)
                        {
                            appointment.Resources.Add(newProvider);
                            appointmentService.Update(appointment);

                            InitialiseForAppointment(appointment);
                            OnProviderChanged(new ProviderChangedArgs(provider.Id));
                            lblMessage.Text = "Provider Replaced";
                        }
                        else
                        {
                            lblMessage.Text = "New provider not found. ResourceId=" + ddlNewProvider.SelectedValueAsInt().ToString();
                        }
                    }
                }
                else
                {
                    lblMessage.Text = "Appointment not found for id=" + appointmentId.ToString();
                }
            }
            else
            {
                lblMessage.Text = "Cannot find appointment";
            }
        }

        void ChangeProviderForGroupAppointment()
        {
            var factory = Helper.GetFactory(this.Page);
            var appointmentService = factory.GetAppointmentService();

            string[] parts = hfAppointmentId.Value.Split(new char[] { ',' });

            List<Appointment> appointments = new List<Appointment>();

            foreach (string part in parts)
            {
                var appointment = appointmentService.Get(int.Parse(part));

                if (appointment != null)
                {
                    appointments.Add(appointment);
                }
                else
                {
                    throw new ApplicationException("Appointment fot found for Id=" + part);
                }
            }

            // get new provider
            var resourceServices = factory.GetResourceService();
            var newProvider = resourceServices.Get(ddlNewProvider.SelectedValueAsInt());

            if (newProvider == null)
            {
                throw new ApplicationException("Unable to find provider with Id=" + ddlNewProvider.SelectedValueAsInt());
            }

            foreach (var a in appointments)
            {
                // remove current provider, if there is one
                var provider = (from p in a.Resources
                                where p.Type == ResourceType.Provider
                                select p).FirstOrDefault();

                if (provider != null)
                {
                    a.Resources.Remove(provider);
                }

                a.Resources.Add(newProvider);
                appointmentService.Update(a);
            }

            InitialiseForAppointment(appointments);
            OnProviderChanged(new ProviderChangedArgs(newProvider.Id));
            lblMessage.Text = "Provider Replaced";
        }

        protected void ddlNewProvider_SelectedIndexChanged(object sender, EventArgs e)
        {
            // postback only
        }

        string GetAppointmentDetails(Appointment a)
        {
            TimeZoneInfo tz = Helper.GetUserSession(this.Page).CurrentSite.GetTimeZoneInfo();

            // show details
            System.Text.StringBuilder sb = new System.Text.StringBuilder();

            if (a.Section != null)
                sb.Append("Clinic : " + a.Section.Name + "<br />");

            sb.Append("Appointment at " + a.Time.ToMediumFormatFromUtc(tz) + "<br />");
            sb.Append("For " + a.Length.ToString() + " minutes<br />");

            return sb.ToString();
        }
    }

    public class ProviderChangedArgs : EventArgs
    {
        int _providerId = 0;

        public int ProviderId
        {
            get { return _providerId; }
            set { _providerId = value; }
        }

        public ProviderChangedArgs()
        {
        }

        public ProviderChangedArgs(int providerId)
        {
            _providerId = providerId;
        }
    }
}