﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.UI.HtmlControls;

using MedRed.Services;
using Shared.Model.Config.MDWS;
using Shared.Model;
using BT.Health.SchedulingWeb.UISession;

namespace BT.Health.SchedulingWeb.User_Controls
{
    public partial class WucPatientSearch : System.Web.UI.UserControl
    {
        /// <summary>
        /// The selected patient ID used to set the final selected patient before navigating
        /// </summary>
        int _selectedPatientId = 0;

        bool _showExistingAppointments = false;

        protected override void OnInit(EventArgs e)
        {
            Page.RegisterRequiresControlState(this);
            base.OnInit(e);
        }

        protected override object SaveControlState()
        {
            Dictionary<string, object> dic = new Dictionary<string, object>();
            dic.Add("_selectedPatientId", _selectedPatientId);
            dic.Add("_showExistingAppointments", _showExistingAppointments);

            dic.Add("base", base.SaveControlState());

            return dic;
        }

        protected override void LoadControlState(object savedState)
        {
            Dictionary<string, object> dic = savedState as Dictionary<string, object>;

            if (dic != null)
            {
                if (dic["_selectedPatientId"] != null && dic["_selectedPatientId"] is int)
                    _selectedPatientId = (int)dic["_selectedPatientId"];

                if (dic["_showExistingAppointments"] != null && dic["_showExistingAppointments"] is bool)
                    _showExistingAppointments = (bool)dic["_showExistingAppointments"];

                base.LoadControlState(dic["base"]);
            }
            else
            {
                base.LoadControlState(savedState);
            }
        }

        protected void Page_Load(object sender, EventArgs e)
        {
            lblMessage.Text = string.Empty;
        }

        protected void Page_PreRender(object sender, EventArgs e)
        {
            if (!IsPostBack)
            {
                // set text box focus on page load
                Page.SetFocus(tbSearch);
                NavigateText = "Select Patient";

                if (Request.QueryString["redirect"] != null && Request.QueryString["redirect"] != string.Empty)
                {
                    // caller wants us to redirect back to a page
                    // we set the linkbutton manually here
                    NavigateUrl = Request.QueryString["redirect"];
                }
            }
        }

        /// <summary>
        /// True if we are to show a patients existing appointments
        /// </summary>
        public bool ShowExistingAppointments
        {
            get { return _showExistingAppointments; }
            set { _showExistingAppointments = value; }
        }

        /// <summary>
        /// Gets or sets the address to navigate forward to (the next stage after selecting the patient)
        /// </summary>
        public string NavigateUrl
        {
            get
            {
                return lbNavigateForwards.CommandArgument;
            }

            set
            {
                lbNavigateForwards.CommandArgument = value;
            }
        }

        /// <summary>
        /// Gets or sets the Text of the navigate forward link
        /// </summary>
        public string NavigateText
        {
            get
            {
                return lbNavigateForwards.Text;
            }

            set
            {
                lbNavigateForwards.Text = value;
            }
        }

        /// <summary>
        /// Search for patients
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        protected void lbSearch_Click(object sender, EventArgs e)
        {
            try
            {
                // get a factory that does the work - used throughout this DataBind
                Factory factory = Helper.GetFactory(Page);

                // we want the patient service
                var patientService = factory.GetPatientService();

                // get the user session that has the current VISTA Id in it
                UserSession userSession = Helper.GetUserSession(Page);

                // set data source to a list of patients searched for that are listed within the currenr VISTA site
                // we get a set of minimal search result data for the patient
                lvPatients.DataSource = patientService.Search(userSession.CurrentSite.VistaSiteId, tbSearch.Text).ToList();
                lvPatients.DataBind();
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        /// <summary>
        /// Bind rows in the patient table
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        protected void lvPatients_ItemDataBound(object sender, ListViewItemEventArgs e)
        {
            // get bound data item
            Shared.Model.PatientSearchResult p = e.Item.DataItem as Shared.Model.PatientSearchResult;

            // get buttons
            LinkButton lbPatient = e.Item.RecursiveFindControl("lbPatient") as LinkButton;

            HtmlTableCell tdDateOfBirth = e.Item.RecursiveFindControl("tdDateOfBirth") as HtmlTableCell;
            HtmlTableCell tdSex = e.Item.RecursiveFindControl("tdSex") as HtmlTableCell;

            HtmlTableCell tdSSN = e.Item.RecursiveFindControl("tdSSN") as HtmlTableCell;

            if (p != null && lbPatient != null && tdDateOfBirth != null &&
                tdSex != null && tdSSN != null)
            {
                lbPatient.Text = p.GetFullName();
                lbPatient.CommandArgument = p.VistaSiteId + "|" + p.VistaDFN;

                tdDateOfBirth.InnerText = p.DateOfBirth.ToString("dd-MMM-yyyy");
                tdSSN.InnerText = Helper.GetSSN(p.SSN);
                tdSex.InnerText = p.Sex;
            }
        }

        protected void lvPatients_ItemCommand(object sender, ListViewCommandEventArgs e)
        {
            switch (e.CommandName)
            {
                case "VewPatientDetails":
                    ShowPatientDetails(e.CommandArgument.ToString());
                    break;
            }
        }

        void ShowPatientDetails(string vistaSiteAndDfn)
        {
            mvPatients.ActiveViewIndex = 1;
            lbNavigateForwards.Visible = false;

            try
            {
                // get a factory
                Factory factory = Helper.GetFactory(Page);

                // we want the patient service
                var patientService = factory.GetPatientService();

                var idList = vistaSiteAndDfn.Split(new char[] { '|' });


                // set data source to a list of patients searched for that are listed within the currenr VISTA site
                var selectedPatient = patientService.Get(idList[0], idList[1]);


                if (selectedPatient != null)
                {
                    lbNavigateForwards.Visible = true;

                    _selectedPatientId = selectedPatient.Id;

                    var p = selectedPatient.Person;
                    // show details
                    tdName.InnerText = p.GetFullName();

                    tdAddress.InnerHtml = p.Address.Street1 + "<br />" +
                                          p.Address.Street2 + "<br />" +
                                          p.Address.Street3;

                    if (p.DateOfBirth.HasValue)
                        tdDateOfBirth.InnerText = p.DateOfBirth.Value.ToString("dd-MMM-yyyy");
                    else
                        tdDateOfBirth.InnerText = string.Empty;

                    tdEMail.InnerText = p.Email;
                    tdEthnicity.InnerText = p.Ethnicity;
                    tdMaritalStatus.InnerText = p.MaritalStatus;
                    tdPhone.InnerText = Helper.GetTelephone(p.Phone);
                    tdSex.InnerText = p.Sex;
                    tdSSN.InnerText = Helper.GetSSN(p.SSN);
                    tdInsurance.InnerText = selectedPatient.HasInsurance.ToString();
                    tdActiveInsurance.InnerText = selectedPatient.ActiveInsurance;
                    tdEligibilityCode.InnerText = selectedPatient.EligibilityCode;
                    tdCurrentMeansStatus.InnerText = selectedPatient.CurrentMeansStatus;
                    tdNeedsMeansTest.InnerText = selectedPatient.NeedsMeansTest.ToString();
                    tdMilitaryStatus.InnerText = selectedPatient.MilitaryStatus;
                    tdOccupation.InnerText = selectedPatient.Occupation;
                    tdIsVeteran.InnerText = selectedPatient.IsVeteran.ToString();
                    tdServiceConnected.InnerText = selectedPatient.ServiceConnected.ToString();
                    tdSCPercent.InnerText = selectedPatient.SCPercent.ToString();
                    tdReligion.InnerText = selectedPatient.Religion;
                    tdPatientType.InnerText = selectedPatient.PatientType;
                    tdPreferencesOther.InnerText = selectedPatient.PreferencesOther;
                    tdSpecialNeedsOther.InnerText = selectedPatient.SpecialNeedsOther;
                    tdPreferences.InnerText = GetPatientPreferences(selectedPatient);
                    tdSpecialNeeds.InnerText = GetPatientSpecialNeeds(selectedPatient);

                    var primaryCareProvider = patientService.GetPrimaryCareProvider(selectedPatient.Id);
                    string providerName = "None";
                    if (primaryCareProvider != null)
                        providerName = primaryCareProvider.Person.GetFullName();
                    tdPrimaryCareProvider.InnerText = providerName;
                    return;
                }
            }
            catch (System.Exception ex)
            {
                tdAddress.InnerHtml = "<span style='color:red'>" + ex.Message + "</span>";
                ex.Log();
            }

            tdName.InnerHtml = "<span style='color:red'>Patient not found</span>";
        }

        protected string GetPatientPreferences(Shared.Model.Patient p)
        {
            string ret = "";
            foreach (PatientPreference pref in p.Preferences)
            {
                if (ret.Length > 0)
                    ret += ", ";
                ret += pref.PatientPreferenceType.Name;
            }

            return ret;
        }

        protected string GetPatientSpecialNeeds(Shared.Model.Patient p)
        {
            string ret = "";
            foreach (SpecialNeed need in p.SpecialNeeds)
            {
                if (ret.Length > 0)
                    ret += ", ";
                ret += need.SpecialNeedType.Name;
            }

            return ret;
        }

        protected void lbBackToPatients_Click(object sender, EventArgs e)
        {
            mvPatients.ActiveViewIndex = 0;
        }

        protected void lbNavigateForwards_Click(object sender, EventArgs e)
        {
            if (lbNavigateForwards.CommandArgument != string.Empty)
            {
                try
                {
                    if (_selectedPatientId > 0)
                    {
                        // get a factory
                        Factory factory = Helper.GetFactory(Page);

                        // we want the patient service
                        var patientService = factory.GetPatientService();
                        var patient = patientService.Get(_selectedPatientId);

                        if (patient != null)
                        {
                            // select patient by adding to session helper and making the current patient
                            UISession.Helper.GetUserSession(Page).PatientList.Add(patient);
                            UISession.Helper.GetUserSession(Page).PatientList.MakeCurrent(patient);

                            // redirect
                            Response.Redirect(lbNavigateForwards.CommandArgument, false);
                        }
                    }
                }
                catch (System.Exception ex)
                {
                    lblMessage.Text = ex.Message; ex.Log();
                }
            }
        }
    }
}
