﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;

using BT.Health.SchedulingWeb.UISession;
using MedRed.Services;
using MedRed.Services.Interfaces;
using Shared.Model;

namespace BT.Health.SchedulingWeb.User_Controls
{
    public partial class WucPatientAppointmentViewCalendar : System.Web.UI.UserControl
    {
        /// <summary>
        /// Uer selected a date start for the calendar (for a day or week view)
        /// </summary>
        public event EventHandler<DateSelected> DateSelected;

        protected void OnDateSelected(DateTime dt)
        {
            EventHandler<DateSelected> eh = DateSelected;

            if (eh != null)
                eh(this, new DateSelected(dt));
        }

        /// <summary>
        /// Raised when an "event" in the calendar is clicked
        /// </summary>
        public event EventHandler<DayPilot.Web.Ui.Events.EventClickEventArgs> CalendarEventClick;

        protected void OnCalendarEventClick(DayPilot.Web.Ui.Events.EventClickEventArgs e)
        {
            EventHandler<DayPilot.Web.Ui.Events.EventClickEventArgs> eh = CalendarEventClick;

            if (eh != null)
                eh(dpPatientAppointments, e);
        }

        /// <summary>
        /// Raised when the user clicks a time in the calendar
        /// </summary>
        public event EventHandler<DayPilot.Web.Ui.Events.TimeRangeSelectedEventArgs> TimeRangeSelected;

        protected void OnTimeRangeSelected(DayPilot.Web.Ui.Events.TimeRangeSelectedEventArgs e)
        {
            EventHandler<DayPilot.Web.Ui.Events.TimeRangeSelectedEventArgs> eh = TimeRangeSelected;

            if (eh != null)
                eh(dpPatientAppointments, e);
        }

        protected void Page_Load(object sender, EventArgs e)
        {
            if (!IsPostBack)
            {
                dpPatientAppointments.StartDate = DateTime.UtcNow.ToStartOfWeekDate();
                dpPatientAppointments.Days = 7;

                // set default
                dpPatientAppointments.BusinessBeginsHour = 9;
                dpPatientAppointments.BusinessEndsHour = 17;
            }
        }

        protected void Page_PreRender(object sender, EventArgs e)
        {
            if (!IsPostBack)
            {
                tbSelectDate.Text = dpPatientAppointments.StartDate.ToString("dd-MMM-yyyy");
            }
        }

        /// <summary>
        /// SHow calendar for a facility - e.g. business hours
        /// </summary>
        /// <param name="facility"></param>
        public void ShowFacilityProperties(Facility facility)
        {
            if (facility.Hours != null)
            {
                lblBusinessHours.Text = "Business Hours : " + facility.Hours;

                // assume business hours in the form "900-1600"
                var hours = facility.Hours.Split(new char[] { '-' });

                if (hours.GetLength(0) == 2)
                {
                    int startstHour = 0;
                    int.TryParse(hours[0], out startstHour);


                    int endsHour = 0;
                    int.TryParse(hours[1], out endsHour);

                    startstHour = startstHour / 100;
                    endsHour = endsHour / 100;

                    if (startstHour > 0 && endsHour > 0 && endsHour < startstHour && endsHour < 24)
                    {
                        dpPatientAppointments.BusinessBeginsHour = startstHour;
                        dpPatientAppointments.BusinessEndsHour = endsHour;

                        return;
                    }
                }
            }

            lblBusinessHours.Text = "Business hours not found - using default";
            // set default
            dpPatientAppointments.BusinessBeginsHour = 9;
            dpPatientAppointments.BusinessEndsHour = 17;
        }

        /// <summary>
        /// Show a list of appointments in an "Outlook" style calendar
        /// Appointments can be a list from a patient or resource
        /// </summary>
        /// <param name="appointments"></param>
        public void ShowAppointments(List<Appointment> appointments)
        {
            List<CalendarData> list = new List<CalendarData>();

            if (appointments != null)
            {
                TimeZoneInfo tz = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();

                foreach (var a in appointments)
                {
                    CalendarData c = new CalendarData();
                    list.Add(c);

                    c.DataStart = TimeZoneInfo.ConvertTimeFromUtc(a.Time, tz);
                    c.DataEnd = TimeZoneInfo.ConvertTimeFromUtc(a.Time, tz).AddMinutes(a.Length);
                    c.Value = a.Id.ToString();

                    c.Text = "Appointment";

                    if (a.AppointmentType != null)
                        c.Text += " " + a.AppointmentType.Name;

                    if (!string.IsNullOrEmpty(a.TelehealthLocation))
                        c.Text += "Tele-Health : Patient at " + a.TelehealthLocation + ". ";

                    c.Text += " : Status:" + a.Status.ToString();

                    if (a.Patient != null)
                        c.Text += " [Patient: ]" + a.Patient.Person.GetFullName();
                    else
                        c.Text += " [Patient: Not found]";
                }
            }

            BindCalendar(list);
        }

        /// <summary>
        /// Show a list of appointments in an "Outlook" style calendar
        /// for a specified provider and clininc
        /// </summary>
        /// <param name="appointments"></param>
        public void ShowAppointmentsForClinic(int sectionId, int providerId)
        {
            List<CalendarData> list = new List<CalendarData>();

            var factory = Helper.GetFactory(Page);

            // provider can appear as a resource more than once
            var resourceService = factory.GetResourceService();
            var providerResources = (from r in resourceService.GetAllOfType(sectionId, ResourceType.Provider)
                                     where r.FullfillingResourceId == providerId
                                     select r).ToList();

            var appointmentService = factory.GetAppointmentService();

            List<Appointment> appointments = new List<Appointment>();

            foreach (var providerResource in providerResources)
            {
                appointments.AddRange((from a in appointmentService.GetForResource(providerResource.Id, null, null, AppointmentStatus.Scheduled)
                                       where a.Status == AppointmentStatus.Scheduled ||
                                        a.Status == AppointmentStatus.CheckedIn ||
                                         a.Status == AppointmentStatus.Closed
                                       select a).ToList());
            }

            ShowAppointments(appointments);
        }

        /// <summary>
        /// Show appointments for provider at facility
        /// </summary>
        /// <param name="facilityId"></param>
        /// <param name="providerId"></param>
        public void ShowAppointmentsForFacility(int facilityId, int providerId)
        {
            List<Appointment> appointments = new List<Appointment>();
            List<CalendarData> list = new List<CalendarData>();

            var factory = Helper.GetFactory(Page);
            var resourceService = factory.GetResourceService();
            var sectionService = factory.GetSectionService();
            var clinics = sectionService.GetAll(facilityId);

            // provider can appear as a resource more than once
            foreach (var clinic in clinics)
            {
                var providerResources = (from r in resourceService.GetAllOfType(clinic.Id, ResourceType.Provider)
                                         where r.FullfillingResourceId == providerId
                                         select r).ToList();

                var appointmentService = factory.GetAppointmentService();

                foreach (var providerResource in providerResources)
                {
                    appointments.AddRange((from a in appointmentService.GetForResource(providerResource.Id, null, null, AppointmentStatus.Scheduled)
                                           where a.Status == AppointmentStatus.Scheduled ||
                                            a.Status == AppointmentStatus.CheckedIn ||
                                             a.Status == AppointmentStatus.Closed
                                           select a).ToList());
                }
            }

            ShowAppointments(appointments);
        }

        /// <summary>
        /// Show appointments for provider at a site
        /// </summary>
        /// <param name="facilityId"></param>
        /// <param name="providerId"></param>
        public void ShowAppointmentsForSite(int siteId, int providerId)
        {
            List<Appointment> appointments = new List<Appointment>();
            List<CalendarData> list = new List<CalendarData>();

            var factory = Helper.GetFactory(Page);
            var resourceService = factory.GetResourceService();
            var sectionService = factory.GetSectionService();

            var facilityService = factory.GetFacilityService();
            var facilities = facilityService.GetAll(siteId);

            foreach (var facility in facilities)
            {
                var clinics = sectionService.GetAll(facility.Id);

                // provider can appear as a resource more than once
                foreach (var clinic in clinics)
                {
                    var providerResources = (from r in resourceService.GetAllOfType(clinic.Id, ResourceType.Provider)
                                             where r.FullfillingResourceId == providerId
                                             select r).ToList();

                    var appointmentService = factory.GetAppointmentService();

                    foreach (var providerResource in providerResources)
                    {
                        appointments.AddRange((from a in appointmentService.GetForResource(providerResource.Id, null, null, AppointmentStatus.Scheduled)
                                               where a.Status == AppointmentStatus.Scheduled ||
                                                a.Status == AppointmentStatus.CheckedIn ||
                                                 a.Status == AppointmentStatus.Closed
                                               select a).ToList());
                    }
                }
            }

            ShowAppointments(appointments);
        }


        /// <summary>
        /// Gets the week start date for the calendar
        /// </summary>
        public DateTime WeekStartDate
        {
            get
            {
                DateTime dt = DateTime.MinValue;
                DateTime.TryParse(tbSelectDate.Text, out dt);

                return dt;
            }
        }

        /// <summary>
        /// Gets the week end date - this is the 0 second of the next day
        /// test for in week is WeekStartDate [greater or equal than] date [less thav] WeekEndingDate
        /// </summary>
        public DateTime WeekEndingDate
        {
            get
            {
                return WeekStartDate.AddDays(7);
            }
        }

        protected void dpPatientAppointments_EventClick(object sender, DayPilot.Web.Ui.Events.EventClickEventArgs e)
        {
            this.OnCalendarEventClick(e);
        }

        protected void dpPatientAppointments_TimeRangeSelected(object sender, DayPilot.Web.Ui.Events.TimeRangeSelectedEventArgs e)
        {
            this.OnTimeRangeSelected(e);
        }

        void ShowAppointmentsForWeek(DateTime appointmentDate)
        {
            // show week
            dpPatientAppointments.StartDate = appointmentDate.ToStartOfWeekDate();
            dpPatientAppointments.Days = 7;
        }

        public void BindCalendar(List<CalendarData> list)
        {
            // assume 9-5 unless list has times outside of this

            int startHour = 9;
            int endHour = 17;

            if (list.Count > 0)
            {
                int minHour = list.Min(a => a.DataStart).Hour;
                int maxHour = list.Max(a => a.DataEnd).Hour + 1;

                if (minHour < startHour)
                    startHour = minHour;

                if (endHour < maxHour)
                    endHour = maxHour;
            }

            dpPatientAppointments.BusinessBeginsHour = startHour;
            dpPatientAppointments.BusinessEndsHour = endHour;

            dpPatientAppointments.DataStartField = "DataStart";
            dpPatientAppointments.DataEndField = "DataEnd";

            dpPatientAppointments.DataTextField = "Text";
            dpPatientAppointments.DataValueField = "Value";

            dpPatientAppointments.DataSource = list;
            dpPatientAppointments.DataBind();
        }

        protected void btnPreviousWeek_Click(object sender, EventArgs e)
        {
            switch (ddlViewStyle.SelectedValueAsInt())
            {
                case 7:
                    dpPatientAppointments.StartDate = dpPatientAppointments.StartDate.AddDays(-7).ToStartOfWeekDate();
                    tbSelectDate.Text = dpPatientAppointments.StartDate.ToString("dd-MMM-yyyy");
                    OnDateSelected(dpPatientAppointments.StartDate);
                    break;

                case 1:
                    dpPatientAppointments.StartDate = dpPatientAppointments.StartDate.AddDays(-1).Date;
                    tbSelectDate.Text = dpPatientAppointments.StartDate.ToString("dd-MMM-yyyy");
                    OnDateSelected(dpPatientAppointments.StartDate);
                    break;
            }
        }

        protected void btnThisWeek_Click(object sender, EventArgs e)
        {
            switch (ddlViewStyle.SelectedValueAsInt())
            {
                case 7:
                    dpPatientAppointments.StartDate = DateTime.UtcNow.ToStartOfWeekDate();
                    tbSelectDate.Text = dpPatientAppointments.StartDate.ToString("dd-MMM-yyyy");
                    OnDateSelected(dpPatientAppointments.StartDate);
                    break;

                case 1:
                    dpPatientAppointments.StartDate = DateTime.UtcNow.Date;
                    tbSelectDate.Text = dpPatientAppointments.StartDate.ToString("dd-MMM-yyyy");
                    OnDateSelected(dpPatientAppointments.StartDate);
                    break;

            }
        }

        protected void btnNextWeek_Click(object sender, EventArgs e)
        {
            switch (ddlViewStyle.SelectedValueAsInt())
            {
                case 7:
                    dpPatientAppointments.StartDate = dpPatientAppointments.StartDate.AddDays(7).ToStartOfWeekDate();
                    tbSelectDate.Text = dpPatientAppointments.StartDate.ToString("dd-MMM-yyyy");
                    OnDateSelected(dpPatientAppointments.StartDate);
                    break;

                case 1:
                    dpPatientAppointments.StartDate = dpPatientAppointments.StartDate.AddDays(1).Date;
                    tbSelectDate.Text = dpPatientAppointments.StartDate.ToString("dd-MMM-yyyy");
                    OnDateSelected(dpPatientAppointments.StartDate);
                    break;
            }
        }

        protected void tbSelectDate_TextChanged(object sender, EventArgs e)
        {
            DateTime dt = DateTime.MaxValue;

            if (DateTime.TryParse(tbSelectDate.Text, out dt))
                SetCalendarWeekStart(dt);
        }

        /// <summary>
        /// given the date passed - set calendar to shoa the wekk it is in
        /// </summary>
        /// <param name="dt"></param>
        public void SetCalendarWeekStart(DateTime dt)
        {
            switch (ddlViewStyle.SelectedValueAsInt())
            {
                case 1:
                    dpPatientAppointments.StartDate = dt.Date;
                    tbSelectDate.Text = dpPatientAppointments.StartDate.ToString("dd-MMM-yyyy");
                    OnDateSelected(dpPatientAppointments.StartDate);
                    break;

                case 7:
                    dpPatientAppointments.StartDate = dt.ToStartOfWeekDate();
                    tbSelectDate.Text = dpPatientAppointments.StartDate.ToString("dd-MMM-yyyy");
                    OnDateSelected(dpPatientAppointments.StartDate);
                    break;
            }
        }

        protected void ddlViewStyle_SelectedIndexChanged(object sender, EventArgs e)
        {
            switch (ddlViewStyle.SelectedValueAsInt())
            {
                case 1:
                    DisplayDaily();
                    dpPatientAppointments.Days = 1;
                    tbSelectDate.Text = dpPatientAppointments.StartDate.ToString("dd-MMM-yyyy");
                    break;

                case 7:
                    DisplayWeekly();
                    dpPatientAppointments.Days = 7;
                    dpPatientAppointments.StartDate = dpPatientAppointments.StartDate.ToStartOfWeekDate();
                    tbSelectDate.Text = dpPatientAppointments.StartDate.ToString("dd-MMM-yyyy");
                    break;
            }
        }

        void DisplayWeekly()
        {
            btnPreviousWeek.Text = "Previous Week";
            btnThisWeek.Text = "This Week";
            btnNextWeek.Text = "Next Week";
        }

        void DisplayDaily()
        {
            btnPreviousWeek.Text = "Previous Day";
            btnThisWeek.Text = "Today";
            btnNextWeek.Text = "Next Day";
        }
    }

    public class DateSelected : EventArgs
    {
        public DateSelected()
        {
        }

        public DateSelected(DateTime selected)
        {
            _selectedDate = selected;
        }

        DateTime _selectedDate = DateTime.Now;

        public DateTime SelectedDate
        {
            get { return _selectedDate; }
            set { _selectedDate = value; }
        }

    }
}