﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.UI.HtmlControls;

using MedRed.Services;
using Shared.Model.Config.MDWS;
using Shared.Model;
using BT.Health.SchedulingWeb.UISession;
using MedRed.Services.Interfaces;

namespace BT.Health.SchedulingWeb.Patient
{
    public partial class WaitingList : System.Web.UI.Page
    {
        TimeZoneInfo _timeZoneInfo = null;

        protected void Page_Load(object sender, EventArgs e)
        {
            try
            {
                lblMessage.Text = string.Empty;

                if (!IsPostBack)
                {
                    LoadFacilityDropDownListForVistaInstance();
                    bindDataGrid();
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        /// <summary>
        /// Show all facilities for the current VistA site
        /// </summary>
        void LoadFacilityDropDownListForVistaInstance()
        {
            // Data Hierarchy :- VistA Instance -> Facility -> Clinic
            // we have the Vista Site Id, now show all facilities within it
            // We'll sort out which clinic to show later in the page process

            int siteId = UISession.Helper.GetUserSession(Page).CurrentSite.Id;
            Factory factory = Helper.GetFactory(Page);

            var facilityService = factory.GetFacilityService();

            ddlFacility.DataValueField = "Id";
            ddlFacility.DataTextField = "Name";
            ddlFacility.DataSource = (from f in facilityService.GetAll(siteId)
                                      orderby f.Name
                                      select f).ToList();

            ddlFacility.SelectedIndex = -1;

            _timeZoneInfo = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();

            ddlFacility.DataBind();

            if (ddlFacility.Items.Count > 0)
            {
                ddlFacility.SelectedIndex = 0;
                LoadClinicDropDownListForFaclility(ddlFacility.SelectedValueAsInt());
            }
            else
            {
                LoadClinicDropDownListForFaclility(0);
            }
        }

        /// <summary>
        /// Load all of the clinics for a facility
        /// </summary>
        /// <param name="facilityId"></param>
        void LoadClinicDropDownListForFaclility(int facilityId)
        {
            Factory factory = Helper.GetFactory(Page);
            var service = factory.GetSectionService();

            // show active AND inactive clinics
            var clinics = (from f in service.GetAll(facilityId)
                           orderby f.Name
                           select f).ToList();

            if (clinics.Count > 0)
            {
                // One or more clinics to show
                ddlClinic.DataTextField = "Name";
                ddlClinic.DataValueField = "Id";
                ddlClinic.DataSource = clinics;
                ddlClinic.SelectedIndex = -1;

                _timeZoneInfo = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();
                ddlClinic.DataBind();

                if (ddlClinic.Items.Count > 0)
                {
                    ddlClinic.SelectedIndex = 0;
                    return;
                }
            }

            // display "No clinics" to user
            ddlClinic.Items.Clear();
            // ID of zero implies no clinic
            ddlClinic.Items.Add(new ListItem("No Clinics", "0"));
        }

        void LoadAppointmentRequestsForClinic(int clinicId, bool priorityOnly)
        {
            try
            {
                Factory factory = Helper.GetFactory(Page);

                var sectionService = factory.GetSectionService();
                var clinic = sectionService.Get(clinicId);

                if (clinicId != 0)
                {
                    var service = factory.GetAppointmentService();

                    // gets either - only priority, or ALL
                    var appointmentRequests = (from r in service.GetAllOpenAppointmentRequests(clinic)
                                               where (priorityOnly == true && r.Priority == 1) || (priorityOnly == false)
                                               orderby r.Time
                                               select r).ToList();

                    List<WaitingListUI> list = new List<WaitingListUI>();

                    // iterate the requests to see if we can get the data (some might be null)
                    foreach (var r in appointmentRequests)
                    {
                        WaitingListUI u = new WaitingListUI();
                        list.Add(u);

                        // add expected values
                        u.Id = r.Id;

                        u.DesiredTime = r.DesiredDate;

                        u.Reason = r.Reason.ToString();

                        TimeSpan requestAge = DateTime.UtcNow - r.Time;
                        u.RequestAge = requestAge.Days;
                        u.Priority = r.Priority;

                        if (r.Patient != null)
                        {
                            if (r.Patient.Person != null)
                            {
                                u.Patient = r.Patient.Person.GetFullName();
                                u.Veteran = r.Patient.IsVeteran;

                                if (r.Patient.Person.DateOfBirth.HasValue)
                                {
                                    u.DOB = r.Patient.Person.DateOfBirth.Value.ToString("dd-MMM-yyyy");
                                }

                                if (r.Patient.Person.SSN != null)
                                {
                                    u.SSN = Helper.GetSSN(r.Patient.Person.SSN);

                                }

                                u.Sex = r.Patient.Person.Sex;
                            }
                        }
                    }

                    if (list.Count > 0)
                    {
                        lvAppointmentRequests.DataSource = list;
                        _timeZoneInfo = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();

                        lvAppointmentRequests.DataBind();
                    }
                }
                else
                {
                    lblMessage.Text = "Clinic not found";
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void lvAppointmentRequests_ItemDataBound(object sender, ListViewItemEventArgs e)
        {
            if (e.Item.ItemType == ListViewItemType.DataItem)
            {
                WaitingListUI w = e.Item.DataItem as WaitingListUI;

                HtmlTableCell tdTime = e.Item.RecursiveFindControl("tdTime") as HtmlTableCell;

                if (w != null && tdTime != null)
                {
                    if (w.DesiredTime.HasValue)
                    {
                        tdTime.InnerText = w.DesiredTime.Value.ToMediumFormatFromUtc(_timeZoneInfo);
                    }
                }
            }
        }

        protected void lvAppointmentRequests_ItemCommand(object sender, ListViewCommandEventArgs e)
        {
            try
            {
                switch (e.CommandName)
                {
                    case "AddAppointment":
                        // set current patient

                        Factory factory = Helper.GetFactory(Page);
                        var sectionService = factory.GetSectionService();
                        var clinic = sectionService.Get(ddlClinic.SelectedValueAsInt());

                        var service = factory.GetAppointmentService();
                        var appointmentRequest = (from r in service.GetAllOpenAppointmentRequests(clinic)
                                                  where r.Id == int.Parse(e.CommandArgument.ToString())
                                                  select r).FirstOrDefault();

                        if (appointmentRequest != null)
                        {
                            var userSession = Helper.GetUserSession(Page);

                            userSession.PatientList.Add(appointmentRequest.Patient);
                            userSession.PatientList.MakeCurrent(appointmentRequest.Patient);

                            // add appointment
                            Response.Redirect("~/Patient/PatientAppointmentAdd.aspx?AppointmentRequestId=" + e.CommandArgument.ToString(), false);
                        }
                        else
                        {
                            lblMessage.Text = "Unable to find Appointment Request";
                        }
                        break;
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }


        protected void ddlFacility_SelectedIndexChanged(object sender, EventArgs e)
        {
            try
            {
                clearListView();
                LoadClinicDropDownListForFaclility(ddlFacility.SelectedValueAsInt());
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void ddlClinic_SelectedIndexChanged(object sender, EventArgs e)
        {
            try
            {
                // show list for clinic
                bindDataGrid();
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void chkVets_CheckedChanged(object sender, EventArgs e)
        {
            try
            {
                bindDataGrid();
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }


        private void bindDataGrid()
        {
            clearListView();
            LoadAppointmentRequestsForClinic(ddlClinic.SelectedValueAsInt(), chkVets.Checked);
        }

        protected void clearListView()
        {
            lvAppointmentRequests.Items.Clear();
            _timeZoneInfo = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();
            lvAppointmentRequests.DataBind();
        }
    }

    /// <summary>
    /// Class for displaying UI data
    /// </summary>
    public class WaitingListUI
    {
        public int Id { get; set; }
        public DateTime? DesiredTime { get; set; }
        public string Patient { get; set; }
        public bool Veteran { get; set; }
        public string DOB { get; set; }
        public string SSN { get; set; }
        public int Priority { get; set; }
        public string Sex { get; set; }
        public int RequestAge { get; set; }
        public String Reason { get; set; }
    }
}