﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.UI.HtmlControls;

using MedRed.Services;
using Shared.Model.Config.MDWS;
using Shared.Model;
using BT.Health.SchedulingWeb.UISession;
using MedRed.Services.Interfaces;

namespace BT.Health.SchedulingWeb.Patient
{
    public partial class ProviderRoster : System.Web.UI.Page
    {
        bool _refreshAppointmentCalendar = false;

        /// <summary>
        /// If set - make this the displayed dtae (the week that it is on)
        /// </summary>
        DateTime? _calendarDateToDisplay = null;

        /// <summary>
        /// Appointment list for patient bind events
        /// </summary>
        List<Appointment> _appointments = null;

        /// <summary>
        /// site timezone using during list databind
        /// </summary>
        TimeZoneInfo _tz = null;

        protected override void OnInit(EventArgs e)
        {
            base.OnInit(e);

            WucPatientAppointmentViewCalendar1.CalendarEventClick += WucPatientAppointmentViewCalendar1_CalendarEventClick;
        }

        void WucPatientAppointmentViewCalendar1_CalendarEventClick(object sender, DayPilot.Web.Ui.Events.EventClickEventArgs e)
        {
            // calendar eveny clicked - show as patient appointment
            Response.Redirect("~/Patient/PatientAppointmentList.aspx?appointmentId=" + e.Value, false);
        }

        protected void Page_Load(object sender, EventArgs e)
        {
            try
            {
                if (!IsPostBack)
                {
                    // load any facilities
                    LoadFacilitiesDdl();
                    ShowProvidersForSite();
                    // lastly
                    ShowProviderFromQueryString();

                    _refreshAppointmentCalendar = true;
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void Page_PreRender(object sender, EventArgs e)
        {
            try
            {
                if (_refreshAppointmentCalendar)
                {
                    try
                    {
                        if (ddlFacility.SelectedValueAsInt() > 0)
                        {
                            if (ddlClinic.SelectedValueAsInt() > 0)
                            {
                                WucPatientAppointmentViewCalendar1.ShowAppointmentsForClinic(ddlClinic.SelectedValueAsInt(), ddlProvider.SelectedValueAsInt());
                            }
                            else
                            {
                                WucPatientAppointmentViewCalendar1.ShowAppointmentsForFacility(ddlFacility.SelectedValueAsInt(), ddlProvider.SelectedValueAsInt());
                            }
                        }
                        else
                        {
                            WucPatientAppointmentViewCalendar1.ShowAppointmentsForSite(Helper.GetUserSession(Page).CurrentSite.Id, ddlProvider.SelectedValueAsInt());
                        }
                    }
                    catch (System.Exception ex)
                    {
                        lblMessage.Text = ex.Message; ex.Log();
                    }
                }

                if (_calendarDateToDisplay.HasValue)
                {
                    // we need to set this value late on because it is set to the NOW in the WUC page load event
                    WucPatientAppointmentViewCalendar1.SetCalendarWeekStart(_calendarDateToDisplay.Value);

                    // finished with it - let user now choose the date they want
                    _calendarDateToDisplay = null;
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void ShowProviderFromQueryString()
        {
            int appointmentId = 0;
            int providerId = 0;
            int clinicId = 0;

            if (Request.QueryString["ProviderId"] != null)
                int.TryParse(Request.QueryString["ProviderId"], out providerId);

            if (Request.QueryString["ClinicId"] != null)
                int.TryParse(Request.QueryString["ClinicId"], out clinicId);

            if (Request.QueryString["appointmentId"] != null)
                int.TryParse(Request.QueryString["appointmentId"], out appointmentId);

            // if we have an appointment, we can get provider and clinic
            if (appointmentId > 0)
            {
                Factory factory = Helper.GetFactory(Page);
                var appointmentService = factory.GetAppointmentService();
                var appointment = appointmentService.Get(appointmentId);

                if (appointment != null)
                {
                    // set clinic
                    clinicId = appointment.Section.Id;

                    // get first provider
                    var providerResource = (from r in appointment.Resources
                                            where r.Type == ResourceType.Provider
                                            select r).FirstOrDefault();

                    // can set provider ID
                    if (providerResource != null)
                    {
                        providerId = providerResource.FullfillingResourceId;

                        // show calendar week that appointment is in
                        _calendarDateToDisplay = appointment.Time;
                    }
                }
            }

            // we might have the data we need to contine either as direct query sttrings or an appointment look up
            if (providerId > 0 && clinicId > 0)
            {
                string providerIdString = providerId.ToString();
                string clicicIdString = clinicId.ToString();

                for (int index = 0; index < ddlFacility.Items.Count; index += 1)
                {
                    ddlFacility.SelectedIndex = index;

                    ShowClinicsForFacility();

                    ListItem liClinic = ddlClinic.Items.FindByValue(clicicIdString);

                    if (liClinic != null)
                    {
                        ddlClinic.SelectedIndex = -1;
                        liClinic.Selected = true;

                        ListItem liProvider = ddlProvider.Items.FindByValue(providerIdString);

                        if (liProvider != null)
                        {
                            ddlProvider.SelectedIndex = -1;
                            liProvider.Selected = true;

                            // found, so done
                            return;
                        }
                    }
                }
            }
        }

        void LoadFacilitiesDdl()
        {
            try
            {
                Factory factory = Helper.GetFactory(Page);
                var facilityService = factory.GetFacilityService();

                int siteId = Helper.GetUserSession(Page).CurrentSite.Id;

                ddlFacility.DataValueField = "Id";
                ddlFacility.DataTextField = "Name";
                ddlFacility.DataSource = (from f in facilityService.GetAll(siteId)
                                          orderby f.Name
                                          select new
                                          {
                                              Id = f.Id,
                                              Name = f.Name
                                          }).ToList();
                ddlFacility.DataBind();

                // show all facilities
                ddlFacility.Items.Insert(0, new ListItem("All Facilities", "0"));
                // select "all"
                ddlFacility.SelectedIndex = 0;

                ShowClinicsForFacility();
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void ShowClinicsForFacility()
        {
            ddlClinic.Items.Clear();

            if (ddlFacility.SelectedValueAsInt() > 0)
            {
                Factory factory = Helper.GetFactory(Page);
                var sectionService = factory.GetSectionService();

                // Show ONLY active clicnics
                ddlClinic.DataValueField = "Id";
                ddlClinic.DataTextField = "Name";
                ddlClinic.DataSource = (from s in sectionService.GetAll(ddlFacility.SelectedValueAsInt())
                                        where s.Active==true
                                        orderby s.Name
                                        select new
                                        {
                                            Id = s.Id,
                                            Name = s.Name
                                        }).ToList();
                ddlClinic.DataBind();

                // see if there are any clinics
                if (ddlClinic.Items.Count == 0)
                {
                    ddlClinic.Items.Clear();
                    ddlClinic.Items.Add(new ListItem("No clinics", "0"));
                }
                else
                {
                    ddlClinic.Items.Insert(0, new ListItem("All clinics", "0"));
                    ddlClinic.SelectedIndex = 0;
                }

                ddlClinic.SelectedIndex = 0;
            }
            else
            {
                ddlClinic.Items.Add(new ListItem("All clinics", "0"));
                ddlClinic.SelectedIndex = 0;
            }

            _refreshAppointmentCalendar = true;
        }

        void ShowProvidersForSite()
        {
            try
            {
                ddlProvider.Items.Clear();

                int siteId = Helper.GetUserSession(Page).CurrentSite.Id;
                Factory factory = Helper.GetFactory(Page);
                var providerService = factory.GetProviderService();

                var data = (from p in providerService.GetAllForSite(siteId)
                            orderby p.Person.LastName
                            select new
                            {
                                Id = p.Id,
                                Name = p.Person.GetFullName()
                            }).ToList();

                FillDdl(data, ddlProvider);
                FillDdl(data, ddlProviderPatientList);

                _refreshAppointmentCalendar = true;
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void FillDdl(object data, DropDownList ddl)
        {
            ddl.Items.Clear();

            ddl.DataValueField = "Id";
            ddl.DataTextField = "Name";
            ddl.DataSource = data;
            ddl.DataBind();

            if (ddlProvider.Items.Count == 0)
            {
                // show "no providers"
                ddl.Items.Add(new ListItem("No providers", "-1"));
            }
        }

        protected void ddlFacility_SelectedIndexChanged(object sender, EventArgs e)
        {
            try
            {
                ShowClinicsForFacility();
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void ddlClinic_SelectedIndexChanged(object sender, EventArgs e)
        {
            _refreshAppointmentCalendar = true;
        }

        protected void ddlProvider_SelectedIndexChanged(object sender, EventArgs e)
        {
            // keep second ddl in sync
            ddlProviderPatientList.SelectedIndex = ddlProvider.SelectedIndex;
            _refreshAppointmentCalendar = true;
        }

        protected void btnViewProviderPatientList_Click(object sender, EventArgs e)
        {
            mv.ActiveViewIndex = 1;
            ShowProviderPatients();
        }

        protected void btnViewRoster_Click(object sender, EventArgs e)
        {
            mv.ActiveViewIndex = 0;
        }

        protected void ddlProviderPatientList_SelectedIndexChanged(object sender, EventArgs e)
        {
            // keep second ddl in sync
            ddlProvider.SelectedIndex = ddlProviderPatientList.SelectedIndex;
            ShowProviderPatients();
        }

        void ShowProviderPatients()
        {
            try
            {
                int providerId = ddlProviderPatientList.SelectedValueAsInt();
                _appointments = new List<Appointment>();

                if (providerId > 0)
                {
                    int siteId = Helper.GetUserSession(Page).CurrentSite.Id;
                    Factory factory = Helper.GetFactory(Page);
                    var providerService = factory.GetProviderService();

                    // provider can appear as a resource more than once
                    var resourceService = factory.GetResourceService();

                    // clinics live in here
                    var sectionService = factory.GetSectionService();

                    // get facilities for site
                    var facilityService = factory.GetFacilityService();
                    var facilities = facilityService.GetAll(siteId);

                    foreach (var facility in facilities)
                    {
                        // get clinics
                        var clinics = sectionService.GetAll(facility.Id);

                        foreach (var clinic in clinics)
                        {
                            var providerResources = (from r in resourceService.GetAllOfType(clinic.Id, ResourceType.Provider)
                                                     where r.FullfillingResourceId == providerId
                                                     select r).ToList();

                            var appointmentService = factory.GetAppointmentService();

                            foreach (var providerResource in providerResources)
                            {
                                // get only scheduled appointments
                                foreach (var appt in appointmentService.GetForResource(providerResource.Id, null, null, AppointmentStatus.Scheduled))
                                {
                                    // only add appointment if not already in the list
                                    if (!_appointments.Contains(appt))
                                        _appointments.Add(appt);
                                }
                            }
                        }
                    }

                    // show patients in list

                    // get unique patients from one or ore appointments
                    lvPatients.DataSource = (from a in _appointments
                                             group a by a.Patient.Id into g
                                             select g.First().Patient).ToList();
                }
                else
                {
                    lvPatients.Items.Clear();
                    lvPatients.DataSource = null;
                }

                _tz = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();
                lvPatients.DataBind();
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void lvPatients_ItemDataBound(object sender, ListViewItemEventArgs e)
        {
            if (e.Item.ItemType == ListViewItemType.DataItem)
            {
                // only data is patient
                Shared.Model.Patient patient = e.Item.DataItem as Shared.Model.Patient;
                HtmlTableCell tdPatientName = e.Item.RecursiveFindControl("tdPatientName") as HtmlTableCell;
                HtmlTableCell tdAppointments = e.Item.RecursiveFindControl("tdAppointments") as HtmlTableCell;

                if (patient != null && tdPatientName != null && tdAppointments != null)
                {
                    tdPatientName.InnerText = patient.Person.GetFullName();

                    var appts = (from a in _appointments
                                 where a.Patient.Id == patient.Id
                                 orderby a.Time
                                 select a).ToList();

                    foreach (var appt in appts)
                    {
                        HyperLink h = new HyperLink();
                        h.NavigateUrl = "~/Patient/PatientAppointmentList.aspx?appointmentId=" + appt.Id.ToString();
                        h.Text = GetAppointmentText(appt);
                        h.CssClass = "linkWIthSmallSpace";
                        h.ToolTip = "View appointment in patient appointment list";
                        tdAppointments.Controls.Add(h);
                    }
                }
            }
        }

        string GetAppointmentText(Shared.Model.Appointment a)
        {
            System.Text.StringBuilder sb = new System.Text.StringBuilder();

            sb.Append(a.Time.ToMediumFormatFromUtc(_tz));

            sb.Append(" (" + a.Length + " mins)");

            if (a.Section != null)
                sb.Append(" at " + a.Section.Name);

            if (a.TelehealthLocation != null && a.TelehealthLocation != string.Empty)
                sb.Append(" [Telehealth: " + a.TelehealthLocation);

            return sb.ToString();
        }
    }
}