﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.UI.HtmlControls;

using MedRed.Services;
using Shared.Model.Config.MDWS;
using Shared.Model;
using BT.Health.SchedulingWeb.UISession;
using MedRed.Services.Interfaces;
namespace BT.Health.SchedulingWeb.Patient
{
    public partial class PatientCheckIn : System.Web.UI.Page
    {
        TimeZoneInfo _timeZoneInfo = null;

        /// <summary>
        /// A list of parent/child appointments
        /// </summary>
        List<Pair> _relatedAppointments = null;

        protected void Page_Load(object sender, EventArgs e)
        {
            lblMessage.Text = string.Empty;

            if (!IsPostBack)
            {
                // must have a patient to proceed
                UserSession userSession = Helper.GetUserSession(Page);

                if (userSession.PatientList.GetCurrentPatient() != null)
                {
                    // show "NOW" for time zone of site
                    tbDateFrom.Text = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo().GetNowForTimeZone().ToString("dd-MMM-yyyy");
                    LoadData();
                }
            }
        }

        void LoadData()
        {
            try
            {
                _timeZoneInfo = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();

                DateTime dtFrom = DateTime.UtcNow;
                if (DateTime.TryParse(tbDateFrom.Text, out dtFrom))
                {
                    // convert to UTC
                    dtFrom = TimeZoneInfo.ConvertTimeToUtc(dtFrom, _timeZoneInfo);
                }

                int patientId = Helper.GetCurrentPatient(Page).Id;

                Factory factory = Helper.GetFactory(Page);
                var appointmentService = factory.GetAppointmentService();

                var allAppointments = (from a in appointmentService.GetForPatient(patientId, null, null)
                                       select a).ToList();

                _relatedAppointments = new List<Pair>();

                foreach (var appt in allAppointments)
                {
                    foreach (var childAppt in appt.ChildAppointments)
                        _relatedAppointments.Add(new Pair(appt, childAppt));
                }

                // must be scheduled ot checked in
                lv.DataSource = (from a in appointmentService.GetForPatient(patientId, dtFrom, null)
                                 where a.Time > dtFrom.Date &&
                                 a.Status == AppointmentStatus.Scheduled ||
                                 a.Status == AppointmentStatus.CheckedIn
                                 orderby a.Time
                                 select a).ToList();

                lv.DataBind();
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void lv_ItemDataBound(object sender, ListViewItemEventArgs e)
        {
            if (e.Item.ItemType == ListViewItemType.DataItem)
            {
                Appointment a = e.Item.DataItem as Appointment;
                HtmlTableCell tdTime = e.Item.RecursiveFindControl("tdTime") as HtmlTableCell;
                HtmlTableCell tdAppointment = e.Item.RecursiveFindControl("tdAppointment") as HtmlTableCell;
                HtmlTableCell tdStatus = e.Item.RecursiveFindControl("tdStatus") as HtmlTableCell;
                HtmlTableCell tdChildAppointments = e.Item.RecursiveFindControl("tdChildAppointments") as HtmlTableCell;

                LinkButton lbCheckIn = e.Item.RecursiveFindControl("lbCheckIn") as LinkButton;
                LinkButton lbAppointmentKept = e.Item.RecursiveFindControl("lbAppointmentKept") as LinkButton;
                LinkButton lpAppointmentNoShow = e.Item.RecursiveFindControl("lpAppointmentNoShow") as LinkButton;
                LinkButton lbAppointmentLeftWithoutBeingSeen = e.Item.RecursiveFindControl("lbAppointmentLeftWithoutBeingSeen") as LinkButton;

                if (a != null && tdTime != null && tdAppointment != null && tdStatus != null &&
                    tdChildAppointments != null && lbCheckIn != null &&
                    lbAppointmentKept != null && lpAppointmentNoShow != null &&
                    lbAppointmentLeftWithoutBeingSeen != null)
                {
                    tdTime.InnerText = a.Time.ToMediumFormatFromUtc(_timeZoneInfo);
                    tdStatus.InnerText = a.Status.ToString();

                    System.Text.StringBuilder sb = new System.Text.StringBuilder();

                    if (a.AppointmentType != null)
                    {
                        sb.Append(a.AppointmentType.Name + "<br />");
                    }

                    if (a.Section != null)
                        sb.Append(a.Length + " minutes at " + a.Section.Name + "<br />");

                    foreach (var r in a.Resources)
                        sb.Append(r.Name + "<br />");

                    if (a.HighPriorityReschedule)
                        sb.Append("High Priority Reschedule");
                    else
                        sb.Append("Normal Priority");

                    var parentAppointment = (from p in _relatedAppointments
                                             where ((AppointmentRelation)p.Second).ChildAppointment.Id == a.Id
                                             select p).FirstOrDefault();

                    // the type of relationship of this appointment to the parent - if there is one
                    if (parentAppointment != null)
                    {
                        var relation = ((AppointmentRelation)parentAppointment.Second);
                        if (relation != null)
                            sb.Append("<br />" + AppointmentRelation.DisplayText(relation.RelationType));
                    }

                    // show text
                    if (!string.IsNullOrEmpty(a.TelehealthLocation))
                        sb.Append("Tele-Health : Patent at " + a.TelehealthLocation + "<br />");

                    tdAppointment.InnerHtml = sb.ToString();

                    sb = new System.Text.StringBuilder();
                    foreach (var linkedAppointment in a.ChildAppointments)
                        sb.Append(linkedAppointment.ChildAppointment.Time.ToMediumFormatFromUtc(_timeZoneInfo) + "<br />");

                    tdChildAppointments.InnerHtml = sb.ToString();

                    //-----------------------------------------
                    lbCheckIn.Visible = false;
                    lbAppointmentKept.Visible = false;
                    lpAppointmentNoShow.Visible = false;
                    lbAppointmentLeftWithoutBeingSeen.Visible = false;

                    switch (a.Status)
                    {
                        case AppointmentStatus.Scheduled:
                            // only these states can be set when scheduled
                            lbCheckIn.Visible = true;
                            lpAppointmentNoShow.Visible = true;
                            break;

                        case AppointmentStatus.CheckedIn:
                            // Once check in, these states can be set
                            lbAppointmentKept.Visible = true;
                            lbAppointmentLeftWithoutBeingSeen.Visible = true;
                            break;
                    }
                }
            }
        }

        protected void lv_ItemCommand(object sender, ListViewCommandEventArgs e)
        {
            try
            {
                switch (e.CommandName)
                {
                    case "CheckIn":
                        CheckIn(int.Parse(e.CommandArgument.ToString()));
                        break;

                    case "CheckOutAsAppointmentKept":
                        CheckOut(int.Parse(e.CommandArgument.ToString()), "Appointment Kept");
                        break;

                    case "CheckOutAsLeftWithoutBeingSeen":
                        CheckOut(int.Parse(e.CommandArgument.ToString()), "Left without being seen");
                        break;

                    case "CheckOutAsNoShow":
                        NoShow(int.Parse(e.CommandArgument.ToString()));
                        break;

                    case "ChangeProvider":
                        ChangeProvider(int.Parse(e.CommandArgument.ToString()));
                        break;
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }


        void ChangeProvider(int appointmentId)
        {
            Factory factory = Helper.GetFactory(Page);
            var appointmentService = factory.GetAppointmentService();

            var appointment = appointmentService.Get(appointmentId);

            if (appointment != null)
            {
                wucChangeProvider1.InitialiseForAppointment(appointment);
                mv.ActiveViewIndex = 1;
            }
            else
            {
                lblMessage.Text = "Appointment not found with id=" + appointmentId.ToString();
            }
        }

        void CheckIn(int appointmentId)
        {
            Factory factory = Helper.GetFactory(Page);
            var appointmentService = factory.GetAppointmentService();

            var appointment = appointmentService.Get(appointmentId);

            if (appointment != null)
            {
                if (appointment.Status == AppointmentStatus.Scheduled)
                {
                    appointmentService.CheckInAppointment(appointment);
                    lblMessage.Text = "Appointment Checked-In";
                    LoadData();
                }
                else
                {
                    lblMessage.Text = "Appointment must be in the [scheduled] state before it can be checked in";
                }
            }
        }

        void CheckOut(int appointmentId, string disposition)
        {
            Factory factory = Helper.GetFactory(Page);
            var appointmentService = factory.GetAppointmentService();

            var appointment = appointmentService.Get(appointmentId);

            if (appointment != null)
            {
                if (appointment.Status == AppointmentStatus.CheckedIn)
                {
                    appointmentService.CheckOutAppointment(appointment, disposition);
                    if (disposition == "Appointment Kept")
                    {
                        ExtensionMethods.AddACommunicationTemplate(appointment, CommunicationType.PostAppointment, this.Page);
                        lblMessage.Text = "Appointment Checked-Out. (Post Appointment queue item added)";
                    }
                       LoadData();
                }
                else
                {
                    lblMessage.Text = "Appointment must be in the [Checked-In] state before it can be checked out";
                }
            }
        }

        void NoShow(int appointmentId)
        {
            Factory factory = Helper.GetFactory(Page);
            var appointmentService = factory.GetAppointmentService();

            var appointment = appointmentService.Get(appointmentId);

            if (appointment != null)
            {
                if (appointment.Status == AppointmentStatus.Scheduled)
                {
                    appointmentService.NoShowAppointment(appointment);
                    lblMessage.Text = "Appointment Checked-Out as No-Show";
                    LoadData();
                }
                else
                {
                    lblMessage.Text = "Appointment must be in the [Checked-In] state before it can be checked out (no-show)";
                }
            }
        }

        protected void tbDateFrom_TextChanged(object sender, EventArgs e)
        {
            LoadData();
        }

        protected void btnBackToAPpointments_Click(object sender, EventArgs e)
        {
            mv.ActiveViewIndex = 0;
            LoadData();
        }
    }
}