﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;

using MedRed.Services;
using Shared.Model.Config.MDWS;
using Shared.Model;
using BT.Health.SchedulingWeb.UISession;
using MedRed.Services.Interfaces;

namespace BT.Health.SchedulingWeb.Patient
{
    public partial class PatientAppointmentReschedule : System.Web.UI.Page
    {
        //----------------------------------------------------

        /// <summary>
        /// The appointment ID to reschedule
        /// </summary>
        int _appointmentIdToReschedule = 0;

        //----------------------------------------------------

        #region Page and control events

        protected override void OnInit(EventArgs e)
        {
            Page.RegisterRequiresControlState(this);
            base.OnInit(e);
        }

        protected override object SaveControlState()
        {
            return new Pair(_appointmentIdToReschedule, base.SaveControlState());
        }

        protected override void LoadControlState(object savedState)
        {
            Pair p = savedState as Pair;

            if (p != null)
            {
                if (p.First != null && p.First is int)
                    _appointmentIdToReschedule = (int)p.First;

                base.LoadControlState(p.Second);
            }
            else
            {
                base.LoadControlState(savedState);
            }
        }

        protected void Page_Load(object sender, EventArgs e)
        {
            try
            {
                if (!IsPostBack)
                {
                    lblMessage.Text = string.Empty;

                    // must have a patient to proceed
                    UserSession userSession = Helper.GetUserSession(Page);

                    if (userSession.PatientList.GetCurrentPatient() != null)
                    {
                        LoadFacilityDropDownListForVistaInstance();
                        LoadAppointmentRelationTypeDDL();

                        ShowAppoinment();
                    }
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void Page_PreRender(object sender, EventArgs e)
        {
            try
            {
                // perform actions based on wizard view displayed
                switch (mvWizard.ActiveViewIndex)
                {
                    case 0:
                        break;

                    case 1:
                        FillStep2Table(tblStep2);
                        ShowExistingAppointments();
                        break;

                    case 2:
                        FillStep3Table(tblStep3);
                        break;

                    case 3:
                        break;
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void ddlFacility_SelectedIndexChanged(object sender, EventArgs e)
        {
            try
            {
                LoadClinicDropDownListForFaclility(ddlFacility.SelectedValueAsInt());
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void ddlClinic_SelectedIndexChanged(object sender, EventArgs e)
        {
            try
            {
                Factory factory = Helper.GetFactory(Page);
                LoadResourcesDropDownLists(factory, ddlClinic.SelectedValueAsInt());
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        #endregion

        //----------------------------------------------------

        void ShowExistingAppointments()
        {
            int patientId = Helper.GetCurrentPatient(Page).Id;

            Factory factory = Helper.GetFactory(Page);
            var service = factory.GetAppointmentService();

            TimeZoneInfo tz = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();

            // Previous No shows
            var noShowAppointments = (from n in service.GetNoShowAppointmentsForPatient(patientId, null, DateTime.UtcNow)
                                      orderby n.Time descending
                                      select new
                                      {
                                          Id = n.Id,
                                          Time = n.Time.ToMediumFormatFromUtc(tz)
                                      }).ToList();

            blPreviousNoShows.DataValueField = "Id";
            blPreviousNoShows.DataTextField = "Time";
            blPreviousNoShows.DataSource = noShowAppointments;
            blPreviousNoShows.DataBind();

            // existing appointments that are not closed or checked-in
            var appointmentTimeFromUtcs = (from a in service.GetForPatient(patientId, null, null)
                                           where a.Time > DateTime.UtcNow.Date &&
                                           a.Status == AppointmentStatus.Scheduled || a.Status == AppointmentStatus.Unknown
                                           select new
                                           {
                                               Id = a.Id,
                                               Time = a.Time.ToMediumFormatFromUtc(tz)
                                           }).ToList();


            if (appointmentTimeFromUtcs.Count > 0)
            {
                // show times of appointment
                blExistingAppointments.DataTextField = "Time";
                blExistingAppointments.DataSource = appointmentTimeFromUtcs;
                blExistingAppointments.DataBind();

                //----------------------------------------------
                // Linked appointments
                // show times of appointment
                ddlLinkedAppointment.DataTextField = "Time";
                ddlLinkedAppointment.DataValueField = "Id";
                ddlLinkedAppointment.DataSource = appointmentTimeFromUtcs;
                ddlLinkedAppointment.DataBind();
            }
            else
            {
                blExistingAppointments.Items.Clear();
                ddlLinkedAppointment.Items.Clear();
            }

            // add an entry in the linked appointment drop down for NO appointment to link to
            ddlLinkedAppointment.Items.Insert(0, new ListItem("Not linked", "0"));
        }

        /// <summary>
        /// Gets the appointment times displayed in the existing appointments list
        /// </summary>
        /// <returns></returns>
        List<DateTime> GetExistingAppointmentTimes()
        {
            List<DateTime> list = new List<DateTime>();

            foreach (ListItem li in blExistingAppointments.Items)
            {
                list.Add(Convert.ToDateTime(li.Text));
            }

            return list;
        }

        void LoadAppointmentRelationTypeDDL()
        {
            //            ExtensionMethods.FillDropDownListWithEnumValues(typeof(AppointmentRelationType), ddlAppointmentRelationship);
            ExtensionMethods.FillWithRecheduleOptions(ddlAppointmentRelationship);
        }


        /// <summary>
        /// Show all facilities for the current VistA site
        /// </summary>
        void LoadFacilityDropDownListForVistaInstance()
        {
            // Data Hierarchy :- VistA Instance -> Facility -> Clinic
            // we have the Vista Site Id, now show all facilities within it
            // We'll sort out which clinic to show later in the page process

            int siteId = UISession.Helper.GetUserSession(Page).CurrentSite.Id;
            Factory factory = Helper.GetFactory(Page);

            var facilityService = factory.GetFacilityService();

            ddlFacility.DataValueField = "Id";
            ddlFacility.DataTextField = "Name";
            ddlFacility.DataSource = (from f in facilityService.GetAll(siteId)
                                      orderby f.Name
                                      select f).ToList();

            ddlFacility.SelectedIndex = -1;
            ddlFacility.DataBind();

            if (ddlFacility.Items.Count > 0)
            {
                ddlFacility.SelectedIndex = 0;
                LoadClinicDropDownListForFaclility(ddlFacility.SelectedValueAsInt());
            }
            else
            {
                LoadClinicDropDownListForFaclility(0);
            }
        }

        /// <summary>
        /// Load all of the clinics for a facility
        /// </summary>
        /// <param name="facilityId"></param>
        void LoadClinicDropDownListForFaclility(int facilityId)
        {
            Factory factory = Helper.GetFactory(Page);
            var service = factory.GetSectionService();

            // Show ONLY active clicnics
            var clinics = (from c in service.GetAll(facilityId)
                           where c.Active == true
                           orderby c.Name
                           select c).ToList();

            if (clinics.Count > 0)
            {
                // One or more clinics to show
                ddlClinic.DataTextField = "Name";
                ddlClinic.DataValueField = "Id";
                ddlClinic.DataSource = clinics;
                ddlClinic.SelectedIndex = -1;
                ddlClinic.DataBind();

                if (ddlClinic.Items.Count > 0)
                {
                    ddlClinic.SelectedIndex = 0;
                    LoadResourcesDropDownLists(factory, ddlClinic.SelectedValueAsInt());
                    return;
                }
            }

            // display "No clinics" to user
            ddlClinic.Items.Clear();
            // ID of zero implies no clinic
            ddlClinic.Items.Add(new ListItem("No Clinics", "0"));

            LoadResourcesDropDownLists(factory, 0);
        }

        void LoadResourcesDropDownLists(Factory factory, int clinicId)
        {
            ddlProviders.Items.Clear();
            ddlRooms.Items.Clear();
            ddlEquipment.Items.Clear();
            ddlAppointmentType.Items.Clear();

            if (clinicId > 0)
            {
                UserSession userSession = Helper.GetUserSession(Page);
                int siteId = userSession.CurrentSite.Id;

                var sectionService = factory.GetSectionService();
                var clinic = sectionService.Get(clinicId);

                // get available resources for the clinic
                var resourceService = factory.GetResourceService();
                var resources = resourceService.GetAll(clinicId);

                //-----------------------------------------------
                var providerService = factory.GetProviderService();
                var providers = providerService.GetAllForSite(siteId).ToList();

                // get providers who are resources at the clinic
                var resourceProviders = (from r in resources
                                         join p in providers
                                         on r.FullfillingResourceId equals p.Id
                                         where r.Type == ResourceType.Provider
                                         orderby p.Person.LastName
                                         select new BasicResourceData
                                         {
                                             ResourceId = r.Id,
                                             ResourceType = Shared.Model.ResourceType.Provider,
                                             ResourceName = r.Name + " (" + p.Person.GetFullName() + ")"
                                         }).ToList();

                ddlProviders.DataValueField = "ResourceId";
                ddlProviders.DataTextField = "ResourceName";
                ddlProviders.DataSource = resourceProviders;
                ddlProviders.DataBind();

                //-----------------------------------------------
                var roomService = factory.GetRoomService();
                var allRooms = roomService.GetAllForSite(siteId).ToList();
                var allRoomResources = resourceService.GetAllOfType(clinicId, ResourceType.Room);

                var rooms = (from r in allRoomResources
                             join a in allRooms
                             on r.FullfillingResourceId equals a.Id
                             orderby r.Name
                             select new BasicResourceData
                             {
                                 ResourceId = r.Id,
                                 ResourceName = r.Name + " (" + a.Name + " - floor + " + a.Floor + ")"
                             }).ToList();

                ddlRooms.DataValueField = "ResourceId";
                ddlRooms.DataTextField = "ResourceName";
                ddlRooms.DataSource = rooms;
                ddlRooms.DataBind();

                // rooms are optional - prepend an empty selection
                ddlRooms.Items.Insert(0, new ListItem("No room required", "0"));
                ddlRooms.SelectedIndex = 0;

                //-----------------------------------------------
                var equipmentService = factory.GetEquipmentService();
                var allEquipment = equipmentService.GetAllForSite(siteId);
                var allEquipmentResources = resourceService.GetAllOfType(clinicId, ResourceType.Equipment);

                var equipment = (from r in allEquipmentResources
                                 join e in allEquipment
                                 on r.FullfillingResourceId equals e.Id
                                 orderby r.Name
                                 select new BasicResourceData
                                 {
                                     ResourceId = r.Id,
                                     ResourceName = r.Name + " (" + e.Name + " - type " + e.Type + ")"
                                 }).ToList();

                ddlEquipment.DataValueField = "ResourceId";
                ddlEquipment.DataTextField = "ResourceName";
                ddlEquipment.DataSource = equipment;
                ddlEquipment.DataBind();

                // equipment optional
                ddlEquipment.Items.Insert(0, new ListItem("No equipment required", "0"));
                ddlEquipment.SelectedIndex = 0;

                //-----------------------------------------------
                var appointmentTypes = (from t in clinic.AppointmentTypes
                                        orderby t.Name
                                        select new BasicResourceData
                                        {
                                            ResourceId = t.Id,
                                            ResourceName = t.Name
                                        }).ToList();

                ddlAppointmentType.DataValueField = "ResourceId";
                ddlAppointmentType.DataTextField = "ResourceName";
                ddlAppointmentType.DataSource = appointmentTypes;
                ddlAppointmentType.DataBind();
            }

            if (ddlProviders.Items.Count == 0)
                ddlProviders.Items.Add(new ListItem("No Providers", "0"));

            if (ddlRooms.Items.Count == 0)
                ddlRooms.Items.Add(new ListItem("No Rooms", "0"));

            if (ddlEquipment.Items.Count == 0)
                ddlEquipment.Items.Add(new ListItem("No Equipment", "0"));

            if (ddlAppointmentType.Items.Count == 0)
                ddlAppointmentType.Items.Add(new ListItem("No Appointment Types", "0"));
        }



        protected void ddlSlotDays_SelectedIndexChanged(object sender, EventArgs e)
        {
            try
            {
                SelectSlotsForDay(Convert.ToDateTime(ddlSlotDays.SelectedValue));
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }


        void SelectSlotsForDay(DateTime dt)
        {
            lbSlots.Items.Clear();

            if (ddlSlotDays.SelectedIndex >= 0)
            {
                if (Session["BT.Health.SchedulingWeb.Patient.PatientAppointmentSearch"] != null &&
                    Session["BT.Health.SchedulingWeb.Patient.PatientAppointmentSearch"] is Dictionary<DateTime, List<SlotInstance>>)
                {
                    var slots = Session["BT.Health.SchedulingWeb.Patient.PatientAppointmentSearch"] as Dictionary<DateTime, List<SlotInstance>>;

                    List<string> slotTimesFromUtc = new List<string>();

                    TimeZoneInfo tz = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();

                    foreach (var slotTime in slots[dt])
                    {
                        DateTime dtTimeZoneTime = TimeZoneInfo.ConvertTimeFromUtc(slotTime.Time, tz);

                        ListItem li = new ListItem();

                        // set value to local timezone date/time, a | separator and the remaining slots
                        li.Value = dtTimeZoneTime.ToMediumFormat() + "|" + slotTime.Remaining.ToString();

                        string slotText = dtTimeZoneTime.ToString("HH:mm");

                        if (slotTime.Remaining <= 0)
                        {
                            if (slotTime.Remaining == 0)
                                li.Text = slotText + " [Booked to capacity]";
                            else
                                li.Text = slotText + " [Over-booked by " + (Math.Abs(slotTime.Remaining)).ToString() + "]";

                            // simple class to set red text
                            li.Attributes.Add("class", "slotOverBooked");
                        }
                        else
                        {
                            li.Text = slotText;
                        }

                        lbSlots.Items.Add(li);
                    }
                }
            }

            if (lbSlots.Items.Count == 0)
            {
                lbSlots.Items.Add(new ListItem("No slots found", string.Empty));
            }
        }

        /// <summary>
        /// gets the selected slot time in the local timezone (as displayed) (or null if no slot selected)
        /// </summary>
        DateTime? SelectedSlotTimeLocalTimeZone
        {
            get
            {
                // need selected item
                if (lbSlots.SelectedIndex >= 0)
                {
                    // expecting a date, separator and remaining slots
                    string[] parts = lbSlots.SelectedItem.Value.Split(new char[] { '|' });

                    // must have two parts
                    if (parts.GetLength(0) == 2)
                    {
                        // see if valid date
                        DateTime dt = DateTime.Now;
                        if (DateTime.TryParse(parts[0], out dt))
                            return dt;
                    }
                }

                return null;
            }
        }

        /// <summary>
        /// gets the selected slot time remaining capacity, or null if not found or selected
        /// </summary>
        int? SelectedSlotRemaining
        {
            get
            {
                // need selected item
                if (lbSlots.SelectedIndex >= 0)
                {
                    // expecting a date, separator and remaining slots
                    string[] parts = lbSlots.SelectedItem.Value.Split(new char[] { '|' });

                    // must have two parts
                    if (parts.GetLength(0) == 2)
                    {
                        // see if valid date
                        int r = 0;
                        if (int.TryParse(parts[1], out r))
                            return r;
                    }
                }

                return null;
            }
        }
        void ShowSlots(Dictionary<DateTime, List<SlotInstance>> slots)
        {
            ddlSlotDays.DataTextFormatString = "{0:dd-MMM-yyyy}";
            ddlSlotDays.DataSource = slots.Keys;
            ddlSlotDays.DataBind();
            ddlSlotDays.SelectedIndex = 0;
            SelectSlotsForDay(Convert.ToDateTime(ddlSlotDays.SelectedItem.Text));
        }

        /// <summary>
        /// Search for free slots
        /// Get the slots for all selected resources then check each has the same slot time free
        /// </summary>
        /// <param name="dtSearchStartDate"></param>
        /// <param name="dtSearchEndDate"></param>
        Dictionary<DateTime, List<SlotInstance>> SearchForSlots(DateTime dtSearchStartDate, DateTime dtSearchEndDate)
        {
            // to search for free slots we must get the slots for all selected resources then check each has the same slots free
            // get the resource Id's selected - we must have at leat a provider
            int providerResourceId = ddlProviders.SelectedValueAsInt();
            int roomResourceId = ddlRooms.SelectedValueAsInt();
            int equipmentResourceId = ddlEquipment.SelectedValueAsInt();

            int siteId = UISession.Helper.GetUserSession(Page).CurrentSite.Id;
            Factory factory = Helper.GetFactory(Page);
            var resourceService = factory.GetResourceService();

            // a provider is required
            System.Collections.ObjectModel.Collection<int> resources = new System.Collections.ObjectModel.Collection<int>();
            resources.Add(providerResourceId);

            // optional - rooms
            if (roomResourceId > 0)
                resources.Add(roomResourceId);

            // optional - equipment
            if (equipmentResourceId > 0)
                resources.Add(equipmentResourceId);

            var slotDays = resourceService.GetAvailability(resources, dtSearchStartDate, dtSearchEndDate);

            // only return days that have available slots
            Dictionary<DateTime, List<SlotInstance>> daysWithSlots = new Dictionary<DateTime, List<SlotInstance>>();

            foreach (var day in slotDays)
            {
                if (slotDays[day.Key].Count > 0)
                    daysWithSlots.Add(day.Key, day.Value);
            }

            return daysWithSlots;
        }

        void Stage1_SearchForSlots()
        {
            DateTime dtSearchStart = DateTime.UtcNow;

            if (WucDateTimePicker1.SelectedDateTimeUTC != null)
            {
                dtSearchStart = WucDateTimePicker1.SelectedDateTimeUTC.Value;
                int daysToSearch = 0;

                if (int.TryParse(tbDaysToSearch.Text, out daysToSearch))
                {
                    // check dates (all now in UTC) make sense
                    DateTime dtSearchStartDate = dtSearchStart.Date;
                    DateTime now = DateTime.UtcNow.Date;

                    if (dtSearchStart >= now)
                    {
                        if (daysToSearch >= 0)
                        {
                            DateTime dtSearchEndDate = dtSearchStart.Date.AddDays(daysToSearch).Date;

                            // get the free slots
                            var slots = SearchForSlots(dtSearchStartDate, dtSearchEndDate);

                            if (slots.Count > 0)
                            {
                                // show slots
                                mvWizard.ActiveViewIndex = 1;
                                Session.Add("BT.Health.SchedulingWeb.Patient.PatientAppointmentSearch", slots);
                                ShowSlots(slots);
                            }
                            else
                            {
                                lblMessage.Text = "No slots found for the requested period and resources";
                            }
                        }
                        else
                        {
                            lblMessage.Text = "Days to search must be zero or greater";
                        }
                    }
                    else
                    {
                        lblMessage.Text = "Your search cannot start prior to today";
                    }
                }
                else
                {
                    lblMessage.Text = "Days to search is invalid";
                }
            }
            else
            {
                lblMessage.Text = "Preferred date is invalid";
            }
        }

        protected void btnSearchForSlots_Click(object sender, EventArgs e)
        {
            try
            {
                if (ddlFacility.SelectedValueAsInt() > 0)
                {
                    if (ddlClinic.SelectedValueAsInt() > 0)
                    {
                        if (ddlProviders.SelectedValueAsInt() > 0)
                        {
                            if (ddlAppointmentType.SelectedValueAsInt() > 0)
                            {
                                Stage1_SearchForSlots();
                            }
                            else
                            {
                                lblMessage.Text = "Please select an appointment type";
                            }
                        }
                        else
                        {
                            lblMessage.Text = "Please select a provider";
                        }
                    }
                    else
                    {
                        lblMessage.Text = "Please select a clinic";
                    }
                }
                else
                {
                    lblMessage.Text = "Please select a facility";
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void btnGotoStep1_Click(object sender, EventArgs e)
        {
            mvWizard.ActiveViewIndex = 0;
        }

        protected void btnGotoStep3_Click(object sender, EventArgs e)
        {
            try
            {
                trOverbooking.Visible = false;

                if (SelectedSlotTimeLocalTimeZone.HasValue)
                {
                    // check patient doesn't already have an appointment at the time of the slot selected
                    List<DateTime> existingAppointments = GetExistingAppointmentTimes();

                    if (existingAppointments.Contains(SelectedSlotTimeLocalTimeZone.Value))
                    {
                        lblMessage.Text = "The patient already has an appointment at this date and time";
                    }
                    else
                    {
                        // overbooking this slot...
                        if (SelectedSlotRemaining <= 0)
                        {
                            trOverbooking.Visible = true;

                            if (Helper.GetUserSession(Page).UserActions.HasAction("action|CanOverbookAppointment"))
                            {
                                mvWizard.ActiveViewIndex = 2;
                            }
                            else
                            {
                                lblMessage.Text = "You are not authorised to overbook this appointment";
                            }
                        }
                        else
                        {
                            mvWizard.ActiveViewIndex = 2;
                        }
                    }
                }
                else
                {
                    lblMessage.Text = "Select a slot for the appointment";
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void btnGotoStep2_Click(object sender, EventArgs e)
        {
            mvWizard.ActiveViewIndex = 1;
        }

        protected void btnGotoStep4_Click(object sender, EventArgs e)
        {
            try
            {
                TimeZoneInfo tz = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();

                var a = CreateAppointment();

                if (a != null)
                {
                    mvWizard.ActiveViewIndex = 3;

                    // show appointment data
                    AddTableRow(tblStep4, "Appointment Time", a.Time.ToMediumFormatFromUtc(tz));
                    AddTableRow(tblStep4, "Length", a.Length.ToString() + " minutes");

                    if (a.AppointmentType != null)
                        AddTableRow(tblStep4, "Appointment Type", a.AppointmentType.Name);

                    AddTableRow(tblStep4, "Status", a.Status.ToString());

                    foreach (var r in a.Resources)
                        AddTableRow(tblStep4, r.Type.ToString(), r.Name);

                    //--------------------------------------------------
                    // check if its a child appointment and show parent
                    int patientId = Helper.GetCurrentPatient(Page).Id;

                    Factory factory = Helper.GetFactory(Page);
                    var service = factory.GetAppointmentService();

                    // possible parent appointments...
                    var parentAppointments = (from p in service.GetForPatient(patientId, null, null)
                                              orderby p.Time
                                              select p).ToList();

                    foreach (var parentAppointment in parentAppointments)
                    {
                        var childAppointmentIds = (from c in parentAppointment.ChildAppointments
                                                   where c.ChildAppointment.Id == a.Id
                                                   select c).ToList();

                        if (childAppointmentIds.Count > 0)
                        {
                            // our new on is a child
                            AddTableRow(tblStep4, "Linked to (" + parentAppointment.AppointmentType.ToString() + ") on", parentAppointment.Time.ToMediumFormatFromUtc(tz));
                        }
                    }
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void FillStep2Table(Table tbl)
        {
            tbl.Rows.Clear();
            AddTableRow(tbl, "Facility", ddlFacility.SelectedItem.Text);
            AddTableRow(tbl, "Clinic", ddlClinic.SelectedItem.Text);
            AddTableRow(tbl, "Provider", ddlProviders.SelectedItem.Text);

            if (ddlRooms.SelectedValueAsInt() > 0)
                AddTableRow(tbl, "Rooms", ddlRooms.SelectedItem.Text);

            if (ddlEquipment.SelectedValueAsInt() > 0)
                AddTableRow(tbl, "Equipment", ddlEquipment.SelectedItem.Text);
        }

        void FillStep3Table(Table tbl)
        {
            // show all data in step 2
            FillStep2Table(tbl);

            AddTableRow(tbl, "Slot", SelectedSlotTimeLocalTimeZone.Value.ToMediumFormat());
        }

        void AddTableRow(Table tbl, string itemName, string itemValue)
        {
            TableRow tr = new TableRow();
            tbl.Rows.Add(tr);

            TableCell tdName = new TableCell();
            tr.Cells.Add(tdName);
            tdName.Text = itemName;

            TableCell tdValue = new TableCell();
            tr.Cells.Add(tdValue);
            tdValue.Text = itemValue;
        }

        Shared.Model.Appointment CreateAppointment()
        {
            try
            {
                // save
                string vistaSiteId = UISession.Helper.GetUserSession(Page).CurrentSite.VistaSiteId;

                Shared.Model.User user = UISession.Helper.GetCurrentUserAccount(this);
                var patient = UISession.Helper.GetCurrentPatient(this);

                //-----------------------------------------
                Factory factory = Helper.GetFactory(Page);
                //-----------------------------------------
                var siteService = factory.GetSectionService();
                var clinic = siteService.Get(int.Parse(ddlClinic.SelectedValue));

                //-----------------------------------------
                var resourceService = factory.GetResourceService();
                var providerResource = resourceService.Get(ddlProviders.SelectedValueAsInt());

                //-----------------------------------------
                var service = factory.GetAppointmentService();

                var nationalSystemService = factory.GetNationalSystemService();

                //-----------------------------------------
                // new pair of request/appointment
                Shared.Model.AppointmentRequest r = null;
                Shared.Model.Appointment a = null;

                DateTime? desirerdDateTimeUtc = WucDateTimePicker1.SelectedDateTimeUTC;

                if (desirerdDateTimeUtc == null)
                    throw new ApplicationException("Desired date is not valid");

                TimeZoneInfo tz = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();

                //UTC time
                DateTime appointmentDateTimeUtc = TimeZoneInfo.ConvertTimeToUtc(SelectedSlotTimeLocalTimeZone.Value, tz);

                int appointmentLength = GetAppointmentLengthMinutes(appointmentDateTimeUtc);

                if (r == null)
                {
                    // create new request to pair with appointment
                    r = new Shared.Model.AppointmentRequest();
                    r.CreatingUser = user;
                    r.Patient = patient;
                    r.RequestMethod = Shared.Model.RequestMethod.Automatic;
                    r.RequestedSection = clinic;
                    r.DesiredDate = desirerdDateTimeUtc;
                    r.Time = DateTime.UtcNow;
                    r.RequestedResource = providerResource;
                    r.RequestedSection = clinic;

                    service.AddAppointmentRequest(r);
                }

                a = new Shared.Model.Appointment();
                a.Patient = patient;
                a.VistaId = vistaSiteId;
                a.Section = clinic;

                r.Status = Shared.Model.RequestStatus.Complete;
                a.Status = Shared.Model.AppointmentStatus.Scheduled;

                r.Time = appointmentDateTimeUtc;

                a.Time = appointmentDateTimeUtc;
                a.Length = appointmentLength;

                //if (ddlUrgency.SelectedValue != null && ddlUrgency.SelectedValue == "1")
                //    r.Priority = 1;
                //else
                //    r.Priority = 0;

                //----------------------------------------
                // unless specifically NOT selected - assume high priority
                //if (ddlIsRescheduleHighPriority.SelectedIndex == 0)
                //    a.HighPriorityReschedule = false;
                //else
                //    a.HighPriorityReschedule = true;

                string trimmedNotes = tbNotes.Text.Trim();

                //----------------------------------------
                var apptType = (from t in clinic.AppointmentTypes
                                where t.Id == ddlAppointmentType.SelectedValueAsInt()
                                select t).FirstOrDefault();

                a.AppointmentType = apptType;

                //AppointmentTypeCategory atc
                if (trimmedNotes != string.Empty)
                {
                    PatientInstructions i = new PatientInstructions();
                    i.Instructions = trimmedNotes;
                    i.Type = InstructionType.PreAppointment;
                    apptType.AddPatientInstructions(i);
                }

                //----------------------------------------
                // resources
                a.Resources.Add(providerResource);

                // add Room resource (if there is one)
                if (ddlRooms.SelectedValueAsInt() > 0)
                    a.Resources.Add(resourceService.Get(ddlRooms.SelectedValueAsInt()));

                // add Equipment resource (if there is one)
                if (ddlEquipment.SelectedValueAsInt() > 0)
                    a.Resources.Add(resourceService.Get(ddlEquipment.SelectedValueAsInt()));

                //----------------------------------------

                // get as ID
                int origAppointmentId = 0;
                if (int.TryParse(Request.QueryString["appointmentId"], out origAppointmentId))
                {
                    var origAppt = service.Get(origAppointmentId);

                    a.AddChildAppointment(origAppt, AppointmentRelationType.RescheduleOf);
                }

                // standard appointment
                service.CreateAppointment(a);

                // add
                r.ResultingAppointment = a;
                //Add Pre appointment Communication queue item
                //AddPreCommunicationTemplate(a);
                ExtensionMethods.AddACommunicationTemplate(a, CommunicationType.PreAppointment, this.Page);
                lblMessage.Text += " (A pre appointment queue item has been added)";
                // update the request
                service.UpdateAppointmentRequest(r);

                //----------------------------------------
                // Linked appointments
                if (ddlLinkedAppointment.SelectedIndex > 0)
                {
                    // 1..n implies a real appointment selected to link to
                    var parentAppointment = service.Get(int.Parse(ddlLinkedAppointment.SelectedValue));

                    if (parentAppointment != null)
                    {
                        AppointmentRelationType art = (AppointmentRelationType)ddlAppointmentRelationship.SelectedValueAsInt();
                        parentAppointment.AddChildAppointment(a, art);
                        service.Update(parentAppointment);
                    }
                    else
                    {
                        throw new ApplicationException("Unable to find selected appointment linked to");
                    }
                }

                //-----------------------------------------
                // cancel original
                var originalAppointment = service.Get(_appointmentIdToReschedule);

                //AddCancellationCommunicationTemplate(originalAppointment);
                ExtensionMethods.AddACommunicationTemplate(originalAppointment, CommunicationType.Cancellation, this.Page);
                lblMessage.Text += " (A cancellation Queue item has been added)";
                MedRed.Services.Utils.AppointmentHelper.PerformAutomaticAppointmentCancelActions(factory, originalAppointment, "Rescheduled");

                // save the provider ID in link button to allow user to look at the appointment list
                hProviderRoster.NavigateUrl = "~/Patient/ProviderRoster.aspx?appointmentId=" + a.Id.ToString();

                return a;

            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }

            return null;
        }

        //private void AddPreCommunicationTemplate(Appointment r)
        //{

        //    //UserSession userSession = Helper.GetUserSession(Page);
        //    Factory factory = Helper.GetFactory(Page);
        //    MedRed.Services.Interfaces.IFacilityService facilityService = factory.GetFacilityService();
        //    MedRed.Services.Interfaces.IAppointmentService appointmentService = factory.GetAppointmentService();

        //    MedRed.Services.Interfaces.ISectionService sectionService = factory.GetSectionService();
        //    IList<CommunicationTemplate> templateList = null;
        //    Section sec = sectionService.Get(r.Section.Id);
        //    templateList = sec.CommunicationTemplates;

        //    CommunicationTemplate t = null;
        //    foreach (CommunicationTemplate template in templateList)
        //    {
        //        if (template.Type == CommunicationType.PreAppointment)
        //        {
        //            t = template;
        //        }
        //    }

        //    if (t != null)
        //    {    //A matching communication template exists so create a queue item
        //        CommunicationQueueItem item = new CommunicationQueueItem();
        //        appointmentService.AddToCommunicationQueue(t, r);
        //        //lblQueueMessage.Text = "Pre Appointment Queue item added";
        //    }

        //}

        //private void AddCancellationCommunicationTemplate(Appointment r)
        //{
        //    //UserSession userSession = Helper.GetUserSession(Page);
        //    Factory factory = Helper.GetFactory(Page);
        //    MedRed.Services.Interfaces.IFacilityService facilityService = factory.GetFacilityService();
        //    MedRed.Services.Interfaces.IAppointmentService appointmentService = factory.GetAppointmentService();

        //    MedRed.Services.Interfaces.ISectionService sectionService = factory.GetSectionService();
        //    IList<CommunicationTemplate> templateList = null;
        //    Section sec = sectionService.Get(r.Section.Id);
        //    templateList = sec.CommunicationTemplates;

        //    CommunicationTemplate t = null;
        //    foreach (CommunicationTemplate template in templateList)
        //    {
        //        if (template.Type == CommunicationType.Cancellation)
        //        {
        //            t = template;
        //        }
        //    }

        //    if (t != null)
        //    {    //A matching communication template exists so create a queue item
        //        CommunicationQueueItem item = new CommunicationQueueItem();
        //        appointmentService.AddToCommunicationQueue(t, r);
        //        lblMessage.Text += " (A cancellation Queue item has been added)";
        //    }

        //}

        protected void ddlLinkedAppointment_SelectedIndexChanged(object sender, EventArgs e)
        {
            // show appointment relationship when required
            int parentApointmentId = ddlLinkedAppointment.SelectedValueAsInt();

            trAppointmentRelationship.Visible = (parentApointmentId > 0);
        }

        void ShowAppoinment()
        {
            try
            {
                // verify if the appointment id is valid
                if (Request.QueryString["appointmentId"] != null)
                {
                    // get as ID
                    int appointmentId = 0;
                    int.TryParse(Request.QueryString["appointmentId"], out appointmentId);

                    if (appointmentId > 0)
                    {
                        // check it exists for patient
                        Factory factory = Helper.GetFactory(Page);

                        var appointmentService = factory.GetAppointmentService();

                        var appointment = appointmentService.Get(appointmentId);

                        if (appointment != null)
                        {
                            if (appointment.Patient.Id == Helper.GetCurrentPatient(Page).Id)
                            {
                                if (appointment.Status == AppointmentStatus.Scheduled)
                                {
                                    //var teleHealthResource = (from t in appointment.Resources
                                    //                          where t.Type == ResourceType.TeleHealth
                                    //                          select t).FirstOrDefault();

                                    //if (teleHealthResource != null)
                                    //{
                                    // show appointment to cancel
                                    tdASection.InnerText = appointment.Section.Name;
                                    var tz = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo();

                                    tdATime.InnerText = appointment.Time.ToMediumFormatFromUtc(tz) + " [" + appointment.Length.ToString() + " minutes]";

                                    if (appointment.AppointmentType != null)
                                        tdAType.InnerText = appointment.AppointmentType.Name;

                                    // show message if is high priority
                                    //tdAPriority.InnerText = appointment.HighPriorityReschedule ? "Yes" : "No";

                                    foreach (var resource in appointment.Resources)
                                    {
                                        if (tdAResources.InnerHtml != string.Empty)
                                            tdAResources.InnerHtml += "<br />";

                                        tdAResources.InnerHtml += resource.Name;
                                    }

                                    // assume a week ahead is the starter
                                    WucDateTimePicker1.SelectedDateTimeUTC = appointment.Time.AddDays(7);

                                    SetAppointmentControlsToExistingAppointment(appointment);

                                    // keep high priority
                                    if (appointment.HighPriorityReschedule)
                                        ddlIsRescheduleHighPriority.SelectedIndex = 1;
                                    else
                                        ddlIsRescheduleHighPriority.SelectedIndex = 0;

                                    _appointmentIdToReschedule = appointment.Id;
                                    mvWizard.ActiveViewIndex = 0;
                                    //}
                                    //else
                                    //{
                                    //    SetBadAppointmentMessage("Cannot reschedule a Telehealth appointment with the current system");
                                    //}
                                }
                                else
                                {
                                    SetBadAppointmentMessage("You cannot cancel an appointment with a status of " + appointment.Status.ToString());
                                }
                            }
                            else
                            {
                                SetBadAppointmentMessage("The appointment selected for cancellation does not belong to this patient");
                            }
                        }
                        else
                        {
                            SetBadAppointmentMessage("The appointment selected for cancellation could not be found");
                        }
                    }
                    else
                    {
                        SetBadAppointmentMessage("The appointment selected for cancellation is invalid");
                    }
                }
                else
                {
                    SetBadAppointmentMessage("You must select an appointment to cancel");
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void SetBadAppointmentMessage(string msg)
        {
            lblBadAppointmentReason.Text = msg;
            mvWizard.ActiveViewIndex = 4;
        }

        /// <summary>
        /// Set the controls in the wizard steps to that of the existing appointment
        /// </summary>
        /// <param name="a"></param>
        void SetAppointmentControlsToExistingAppointment(Appointment a)
        {
            Factory factory = Helper.GetFactory(Page);

            ListItem liFacility = ddlFacility.Items.FindByValue(a.Section.Facility.Id.ToString());

            if (liFacility != null)
            {
                ddlFacility.SelectedIndex = -1;
                liFacility.Selected = true;

                LoadClinicDropDownListForFaclility(ddlFacility.SelectedValueAsInt());
            }

            ListItem liClinic = ddlClinic.Items.FindByValue(a.Section.Id.ToString());

            if (liClinic != null)
            {
                ddlClinic.SelectedIndex = -1;
                liClinic.Selected = true;
                LoadResourcesDropDownLists(factory, ddlClinic.SelectedValueAsInt());
            }

            foreach (var resource in a.Resources)
            {
                switch (resource.Type)
                {
                    case ResourceType.Provider:
                        SelectDropDownItemFromValue(ddlProviders, resource.Id.ToString());
                        break;

                    case ResourceType.Room:
                        SelectDropDownItemFromValue(ddlRooms, resource.Id.ToString());
                        break;

                    case ResourceType.Equipment:
                        SelectDropDownItemFromValue(ddlEquipment, resource.Id.ToString());
                        break;

                    //case ResourceType.TeleHealth:
                    //    break;
                }
            }

            // appointment type
            SelectDropDownItemFromValue(ddlAppointmentType, a.AppointmentType.Id.ToString());
        }

        void SelectDropDownItemFromValue(DropDownList ddl, string value)
        {
            ListItem li = ddl.Items.FindByValue(value);

            if (li != null)
            {
                ddl.SelectedIndex = -1;
                li.Selected = true;
            }
        }

        protected int GetAppointmentLengthMinutes(DateTime slotTimeUtc)
        {
            var slots = Session["BT.Health.SchedulingWeb.Patient.PatientAppointmentSearch"] as Dictionary<DateTime, List<SlotInstance>>;

            DateTime slotDay = slotTimeUtc.Date;

            var slot = (from s in slots[slotDay]
                        where s.Time == slotTimeUtc
                        select s).FirstOrDefault();

            if (slot != null)
                return slot.Length;
            else
                throw new ApplicationException("Unable to find slot length");
        }
    }
}
