﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.UI.HtmlControls;

using MedRed.Services;
using Shared.Model.Config.MDWS;
using Shared.Model;
using BT.Health.SchedulingWeb.UISession;
using MedRed.Services.Interfaces;

namespace BT.Health.SchedulingWeb.Patient
{
    public partial class PatientAppointmentList : System.Web.UI.Page
    {
        /// <summary>
        /// A list of parent/child appointments
        /// </summary>
        List<Pair> _relatedAppointments = null;

        // used during a data bind
        TimeZoneInfo _tzInfo = null;

        // whether we meed to refresh the current view
        bool _refreshData = false;

        protected void Page_Load(object sender, EventArgs e)
        {
            try
            {
                lblMessage.Text = string.Empty;

                if (!IsPostBack)
                {
                    // if page called with an appointment ID, load patient
                    LoadPatientFromAppointmentQquery();

                    // must have a patient to proceed
                    UserSession userSession = Helper.GetUserSession(Page);

                    if (userSession.PatientList.GetCurrentPatient() != null)
                    {
                        // show NOW for the time zone
                        tbDateFrom.Text = Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo().GetNowForTimeZone().ToString("dd-MMM-yyyy");
                        tbHistoryFrom.Text = tbDateFrom.Text;
                        tbHistoryTo.Text = (Helper.GetUserSession(Page).CurrentSite.GetTimeZoneInfo().GetNowForTimeZone().AddDays(7)).ToString("dd-MMM-yyyy");

                        LoadAppointmentTypeDDL();
                        _refreshData = true;
                    }
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void Page_PreRender(object sender, EventArgs e)
        {
            if (_refreshData)
            {
                // show "tab" buttons to view different items
                switch (mvAppointments.ActiveViewIndex)
                {
                    case 0:
                        btnViewCurrentAppointments.Visible = false;
                        btnViewAppointmentRequests.Visible = true;
                        btnViewAppointmentHistory.Visible = true;
                        LoadAppointmentData();
                        break;

                    case 1:
                        btnViewCurrentAppointments.Visible = true;
                        btnViewAppointmentRequests.Visible = false;
                        btnViewAppointmentHistory.Visible = true;
                        LoadAppointmentRequestData();
                        break;

                    case 2:
                        btnViewCurrentAppointments.Visible = true;
                        btnViewAppointmentRequests.Visible = true;
                        btnViewAppointmentHistory.Visible = false;
                        LoadDataAppointmentHistory();
                        break;

                    case 3:
                        btnViewCurrentAppointments.Visible = true;
                        btnViewAppointmentRequests.Visible = true;
                        btnViewAppointmentHistory.Visible = true;
                        break;
                }
            }
        }

        void LoadPatientFromAppointmentQquery()
        {
            if (Request.QueryString["appointmentId"] != null)
            {
                int appointmentId = 0;
                int.TryParse(Request.QueryString["appointmentId"], out appointmentId);

                if (appointmentId > 0)
                {
                    Factory factory = Helper.GetFactory(Page);
                    var appointmentService = factory.GetAppointmentService();

                    var appointment = appointmentService.Get(appointmentId);

                    if (appointment != null)
                    {
                        UserSession userSession = Helper.GetUserSession(Page);
                        userSession.PatientList.Add(appointment.Patient);
                        userSession.PatientList.MakeCurrent(appointment.Patient);
                    }
                }
            }
        }


        void LoadAppointmentData()
        {
            try
            {
                // needed to convert dates
                _tzInfo = TimeZoneInfo.FindSystemTimeZoneById(Helper.GetUserSession(Page).CurrentSite.TimeZoneId);

                // we work in UTC when not displaying dates
                DateTime dtFrom = DateTime.UtcNow;

                if (DateTime.TryParse(tbDateFrom.Text, out dtFrom))
                {
                    // conversion from displayed date in text to a date time was OK, so convert to UTC for internal working
                    dtFrom = TimeZoneInfo.ConvertTimeToUtc(dtFrom, _tzInfo);
                }

                List<Appointment> appointments = null;

                int patientId = Helper.GetCurrentPatient(Page).Id;

                Factory factory = Helper.GetFactory(Page);
                var appointmentService = factory.GetAppointmentService();

                var allAppointments = (from a in appointmentService.GetForPatient(patientId, null, null)
                                       select a).ToList();


                _relatedAppointments = new List<Pair>();

                foreach (var appt in allAppointments)
                {
                    foreach (var childAppt in appt.ChildAppointments)
                        _relatedAppointments.Add(new Pair(appt, childAppt));
                }

                int statusId = ddlAppoinmentStatus.SelectedValueAsInt();
                if (statusId >= 0)
                {
                    AppointmentStatus status = (AppointmentStatus)statusId;

                    appointments = (from a in allAppointments
                                    where a.Time > dtFrom.Date &&
                                    a.Status == status
                                    orderby a.Time
                                    select a).ToList();
                }
                else
                {
                    appointments = (from a in allAppointments
                                    where a.Time > dtFrom.Date
                                    orderby a.Time
                                    select a).ToList();
                }

                lv.DataSource = appointments;
                lv.DataBind();
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        void LoadAppointmentRequestData()
        {
            try
            {
                // needed to convert dates
                _tzInfo = TimeZoneInfo.FindSystemTimeZoneById(Helper.GetUserSession(Page).CurrentSite.TimeZoneId);

                Factory factory = Helper.GetFactory(Page);
                var appointmentService = factory.GetAppointmentService();

                var appointmentRequests = (from r in appointmentService.GetAppointmentRequestsForPatient(Helper.GetCurrentPatient(Page))
                                           where r.ResultingAppointment == null
                                           orderby r.Time
                                           select r).ToList();

                if (ddlRequestSitesDisplayed.SelectedValueAsInt() == 1)
                {
                    lvppointmentRequests.DataSource = appointmentRequests;
                }
                else
                {
                    // filter by current site
                    int siteId = Helper.GetUserSession(Page).CurrentSite.Id;

                    lvppointmentRequests.DataSource = (from a in appointmentRequests
                                                       where a.RequestedSection != null &&
                                                       a.RequestedSection.Facility.Site.Id == siteId
                                                       select a).ToList();
                }

                lvHistory.DataBind();


                lvppointmentRequests.DataBind();

                foreach (var item in lvppointmentRequests.Items)
                {
                    HtmlTableCell tdAppReason = item.RecursiveFindControl("tdAppRequestReason") as HtmlTableCell;
                    if (tdAppReason.InnerText == "WaitList")
                        tdAppReason.InnerText = "Wait List";
                    continue;
                }

            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void lv_ItemDataBound(object sender, ListViewItemEventArgs e)
        {
            if (e.Item.ItemType == ListViewItemType.DataItem)
            {
                Appointment a = e.Item.DataItem as Appointment;
                HtmlTableCell tdAppointment = e.Item.RecursiveFindControl("tdAppointment") as HtmlTableCell;
                HtmlTableCell tdParentAppointment = e.Item.RecursiveFindControl("tdParentAppointment") as HtmlTableCell;
                HtmlTableCell tdChildAppointments = e.Item.RecursiveFindControl("tdChildAppointments") as HtmlTableCell;
                HtmlTableCell tdIsTeleHealth = e.Item.RecursiveFindControl("tdIsTeleHealth") as HtmlTableCell;

                LinkButton lbChangeProvider = e.Item.RecursiveFindControl("lbChangeProvider") as LinkButton;

                HyperLink hCancelAppointment = e.Item.RecursiveFindControl("hCancelAppointment") as HyperLink;
                HyperLink hRescheduleAppointment = e.Item.RecursiveFindControl("hRescheduleAppointment") as HyperLink;
                HyperLink hViewInProviderRoster = e.Item.RecursiveFindControl("hViewInProviderRoster") as HyperLink;

                if (a != null && tdAppointment != null && hCancelAppointment != null && hRescheduleAppointment != null &&
                    tdParentAppointment != null && tdChildAppointments != null && hViewInProviderRoster != null &&
                    tdIsTeleHealth != null && lbChangeProvider != null)
                {
                    //-------------------------------------------------
                    lbChangeProvider.CommandName = "ChangeProvider";
                    lbChangeProvider.CommandArgument = a.Id.ToString();

                    //-------------------------------------------------
                    var parentAppointment = (from p in _relatedAppointments
                                             where ((AppointmentRelation)p.Second).ChildAppointment.Id == a.Id
                                             select p).FirstOrDefault();

                    // parent appointment--------------------------------------------------------
                    if (parentAppointment != null)
                    {
                        tdParentAppointment.InnerHtml = ((Appointment)parentAppointment.First).Time.ToMediumFormatFromUtc(_tzInfo);
                        tdAppointment.InnerHtml = GetAppointmentInfo(a, AppointmentRelation.DisplayText(((AppointmentRelation)parentAppointment.Second).RelationType));
                    }
                    else
                    {
                        tdAppointment.InnerHtml = GetAppointmentInfo(a, string.Empty);
                    }
                    // --------------------------------------------------------------------------

                    System.Text.StringBuilder sb = new System.Text.StringBuilder();

                    foreach (var linkedAppointment in a.ChildAppointments)
                    {
                        if (sb.ToString() == string.Empty)
                            sb.Append("<br /><br />");

                        sb.Append(linkedAppointment.ChildAppointment.Time.ToMediumFormatFromUtc(_tzInfo) + "<br />");
                    }

                    tdChildAppointments.InnerHtml = sb.ToString();

                    if (a.Status == AppointmentStatus.Scheduled)
                    {
                        hCancelAppointment.NavigateUrl = "~/Patient/PatientAppointmentCancel.aspx?appointmentId=" + a.Id.ToString();
                        hCancelAppointment.Visible = true;

                        hRescheduleAppointment.NavigateUrl = "~/Patient/PatientAppointmentReschedule.aspx?appointmentId=" + a.Id.ToString();
                        hRescheduleAppointment.Visible = true;
                    }
                    else
                    {
                        hCancelAppointment.Visible = false;
                        hRescheduleAppointment.Visible = false;
                    }

                    tdIsTeleHealth.InnerText = (string.IsNullOrEmpty(a.TelehealthLocation)) ? "No" : "Yes";

                    hViewInProviderRoster.NavigateUrl = "~/Patient/ProviderRoster.aspx?appointmentId=" + a.Id.ToString();
                }
            }
        }

        string GetAppointmentInfo(Appointment a, string relation)
        {
            System.Text.StringBuilder sb = new System.Text.StringBuilder();
            sb.Append(a.Time.ToMediumFormatFromUtc(_tzInfo) + " for " + a.Length + " minutes<br />");

            if (a.Section != null)
                sb.Append("At " + a.Section.Name + "<br />");
            else
                sb.Append("Section not found<br />");

            sb.Append("Status - " + a.Status.ToString() + "<br />");

            if (a.AppointmentType != null)
                sb.Append("Appointment Type - " + a.AppointmentType.Name + "<br />");
            else
                sb.Append("Appointment Type - Unknown<br />");

            foreach (var r in a.Resources)
                sb.Append(r.Name + "<br />");

            if (a.HighPriorityReschedule)
                sb.Append("High Priority Reschedule");
            else
                sb.Append("Normal Priority");

            // the type of relationship of this appointment to the parent
            if (relation != null)
                sb.Append("<br />" + relation);

            return sb.ToString();
        }

        void LoadAppointmentTypeDDL()
        {
            ExtensionMethods.FillDropDownListWithEnumValues(typeof(AppointmentStatus), ddlAppoinmentStatus);
            ddlAppoinmentStatus.Items.Insert(0, new ListItem("Show All", "-1"));
            ddlAppoinmentStatus.SelectedIndex = 0;

            ddlAppoinmentStatus.SelectedIndex = -1;
            var li = ddlAppoinmentStatus.Items.FindByText(AppointmentStatus.Scheduled.ToString());

            if (li != null)
                li.Selected = true;
            else
                ddlAppoinmentStatus.SelectedIndex = 0;
        }

        protected void ddlAppoinmentStatus_SelectedIndexChanged(object sender, EventArgs e)
        {
            _refreshData = true;
        }

        protected void lv_ItemCommand(object sender, ListViewCommandEventArgs e)
        {
            switch (e.CommandName)
            {
                case "ChangeProvider":
                    ShowChangeProvider(e.CommandArgument.ToString());
                    break;

                /*
            case "EditAppointment":
                Response.Redirect("~/Patient/PatientAppointmentEdit.aspx?id=" + e.CommandArgument.ToString(), false);
                break;

            case "CheckIn":
                Response.Redirect("~/Patient/PatientCheckIn.aspx?id=" + e.CommandArgument.ToString(), false);
                break;
                 */
            }
        }

        protected void lvppointmentRequests_ItemCommand(object sender, ListViewCommandEventArgs e)
        {
            if (e.CommandName == "AddAppointment")
            {
                // add an appointment
                Response.Redirect("~/Patient/PatientAppointmentAdd.aspx?AppointmentRequestId=" + e.CommandArgument.ToString(), false);
            }
        }

        protected void tbDateFrom_TextChanged(object sender, EventArgs e)
        {
            _refreshData = true;
        }

        protected void lvppointmentRequests_ItemDataBound(object sender, ListViewItemEventArgs e)
        {
            AppointmentRequest r = e.Item.DataItem as AppointmentRequest;
            HtmlTableCell tdRequestedTime = e.Item.RecursiveFindControl("tdRequestedTime") as HtmlTableCell;

            HtmlTableCell tdRequestSection = e.Item.RecursiveFindControl("tdRequestSection") as HtmlTableCell;

            Label lblAtAnotherSite = e.Item.RecursiveFindControl("lblAtAnotherSite") as Label;

            if (r != null && tdRequestedTime != null && tdRequestSection != null && lblAtAnotherSite != null)
            {
                tdRequestedTime.InnerText = r.Time.ToMediumFormatFromUtc(_tzInfo);

                if (r.RequestedSection != null)
                {
                    tdRequestSection.InnerHtml = r.RequestedSection.Facility.Site.Name + "<br/>" + r.RequestedSection.Facility.Name + "<br />" + r.RequestedSection.Name;

                    lblAtAnotherSite.Visible = (Helper.GetUserSession(Page).CurrentSite.Id != r.RequestedSection.Facility.Site.Id);
                }
            }
        }

        protected void btnViewCurrentAppointments_Click(object sender, EventArgs e)
        {
            mvAppointments.ActiveViewIndex = 0;
            _refreshData = true;
        }

        protected void btnViewAppointmentRequests_Click(object sender, EventArgs e)
        {
            mvAppointments.ActiveViewIndex = 1;
            _refreshData = true;
        }

        protected void btnViewAppointmentHistory_Click(object sender, EventArgs e)
        {
            mvAppointments.ActiveViewIndex = 2;
            _refreshData = true;
        }

        protected void tbHistoryFrom_TextChanged(object sender, EventArgs e)
        {
            _refreshData = true;
        }

        protected void tbHistoryTo_TextChanged(object sender, EventArgs e)
        {
            _refreshData = true;
        }

        void LoadDataAppointmentHistory()
        {
            try
            {
                // needed to convert dates
                _tzInfo = TimeZoneInfo.FindSystemTimeZoneById(Helper.GetUserSession(Page).CurrentSite.TimeZoneId);

                // we work in UTC when not displaying dates
                DateTime dtFromLocal = DateTime.Parse(tbHistoryFrom.Text);
                DateTime dtToLocal = DateTime.Parse(tbHistoryTo.Text);

                DateTime dtFromUtc = TimeZoneInfo.ConvertTimeToUtc(dtFromLocal, _tzInfo);
                DateTime dtToUtc = TimeZoneInfo.ConvertTimeToUtc(dtToLocal, _tzInfo);

                int patientId = Helper.GetCurrentPatient(Page).Id;

                Factory factory = Helper.GetFactory(Page);
                var appointmentService = factory.GetAppointmentService();

                // appointments in UTC
                var allAppointments = (from a in appointmentService.GetForPatient(patientId, dtFromUtc, dtToUtc)
                                       orderby a.Time
                                       select a).ToList();


                lvHistory.DataSource = allAppointments;
                lvHistory.DataBind();
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void lvHistory_ItemDataBound(object sender, ListViewItemEventArgs e)
        {
            if (e.Item.ItemType == ListViewItemType.DataItem)
            {
                Appointment a = e.Item.DataItem as Appointment;

                HtmlTableCell tdTime = e.Item.RecursiveFindControl("tdTime") as HtmlTableCell;
                HtmlTableCell tdStatus = e.Item.RecursiveFindControl("tdStatus") as HtmlTableCell;
                HtmlTableCell tdProvider = e.Item.RecursiveFindControl("tdProvider") as HtmlTableCell;
                HtmlTableCell tdActivity = e.Item.RecursiveFindControl("tdActivity") as HtmlTableCell;

                // show UTC in locat time
                tdTime.InnerText = TimeZoneInfo.ConvertTimeFromUtc(a.Time, _tzInfo).ToMediumFormat();

                tdStatus.InnerText = a.Status.ToString();

                var providerResource = (from r in a.Resources
                                        where r.Type == ResourceType.Provider
                                        select r).FirstOrDefault();

                Factory factory = Helper.GetFactory(Page);

                var providerService = factory.GetProviderService();
                var provider = providerService.Get(providerResource.FullfillingResourceId);

                if (provider != null)
                {
                    tdProvider.InnerText = providerResource.Name + " (" + provider.Person.GetFullName() + ")";
                }

                var activities = (from x in a.Activity
                                  orderby x.Time
                                  select x).ToList();

                if (activities.Count > 0)
                {
                    System.Text.StringBuilder sb = new System.Text.StringBuilder();

                    sb.Append("<table>");
                    foreach (var activity in activities)
                    {
                        sb.Append("<tr>");
                        // activity should be in UTC
                        sb.Append("<td>" + TimeZoneInfo.ConvertTimeFromUtc(activity.Time, _tzInfo).ToMediumFormat() + "</td>");
                        sb.Append("<td>" + activity.ActivityType.ToString() + "</td>");
                        sb.Append("<td>" + activity.ActivityReason + "</td>");

                        sb.Append("</tr>");
                    }

                    sb.Append("</table>");

                    tdActivity.InnerHtml = sb.ToString();
                }
            }
        }


        void ShowChangeProvider(string appointmentIdString)
        {
            try
            {
                int apppointmentId = 0;

                if (int.TryParse(appointmentIdString, out apppointmentId))
                {
                    // get appointment
                    var factory = Helper.GetFactory(this.Page);
                    var appointmentService = factory.GetAppointmentService();
                    var appointment = appointmentService.Get(apppointmentId);

                    if (appointment != null)
                    {
                        wucChangeProvider1.InitialiseForAppointment(appointment);
                        mvAppointments.ActiveViewIndex = 3;
                        _refreshData = true;
                    }
                    else
                    {
                        lblMessage.Text = "Cannot find appointment with Id=" + apppointmentId.ToString();
                    }
                }
                else
                {
                    lblMessage.Text = "Appointment not found for Id=" + appointmentIdString;
                }
            }
            catch (System.Exception ex)
            {
                lblMessage.Text = ex.Message; ex.Log();
            }
        }

        protected void ddlRequestSitesDisplayed_SelectedIndexChanged(object sender, EventArgs e)
        {
            // just refresh
            _refreshData = true;
        }
    }
}